/*
 * Copyright (c) 2021 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Wed Sep 29 2021
 */
#ifndef H_TK3_MUX
#define H_TK3_MUX

#include <sys/time.h>

#include <stdint.h>

struct tk_chan_s {
  char path[1024];
  int fd, sl;		/* master, slave fd */

  struct { uint8_t buf[64], r, w; } rb, wb; /* read/write ring buffer */

  uint8_t msg[64], len; /* last decoded message */
  unsigned start:1;
  unsigned escape:1;
  unsigned revent:1;
};


struct tk_chan_s *	tk_open_tty(const char *device);
void			tk_close_tty(struct tk_chan_s **chan);
int			tk_recv_msg(struct tk_chan_s *chan, bool recv);
int			tk_send_msg(struct tk_chan_s *chan,
                                const char *fmt, ...);
int			tk_putc(struct tk_chan_s *chan, char c);

struct tk_chan_s *	tk_open_pty(const char *slave);
void			tk_close_pty(struct tk_chan_s **pty);
int			tk_recv(struct tk_chan_s *pty, bool recv);
int			tk_send(struct tk_chan_s *pty, const uint8_t *buf,
                                size_t len);
const char *		tk_tcdescr(const struct tk_chan_s *pty);


/* return an absolute deadline for given ms timeout */
static inline struct timeval
tk_deadline(int ms)
{
  struct timeval d;

  gettimeofday(&d, NULL);
  d.tv_usec += ms * 1000;
  if (d.tv_usec >= 1000000) {
    d.tv_usec -= 1000000;
    d.tv_sec++;
  }

  return d;
}

/* return ms timeout for a given deadline */
static inline int
tk_timeout(struct timeval deadline)
{
  struct timeval now;
  int timeout;

  gettimeofday(&now, NULL);
  timeout =
    (deadline.tv_sec - now.tv_sec) * 1000
    + (deadline.tv_usec - now.tv_usec) / 1000;
  return timeout < 0 ? 0 : timeout;
}

#endif /* H_TK3_MUX */
