/*
 * Copyright (c) 2021 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Wed Sep 29 2021
 */
#include "autoconf.h"

#define _XOPEN_SOURCE 600	/* posix_openpt(3), ptsname(3) */
#define _DEFAULT_SOURCE		/* EXTPROC */

#include <sys/ioctl.h>

#include <errno.h>
#include <fcntl.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <termios.h>
#include <unistd.h>

#include "tk3-mux.h"


/* --- tk_open_pty --------------------------------------------------------- */

struct tk_chan_s *
tk_open_pty(const char *slave)
{
  struct tk_chan_s *pty;
  int fd, sl = -1;

  /* open */
  fd = posix_openpt(O_RDWR | O_NOCTTY);
  if (fd < 0) return NULL;

  /* symlink */
  unlink(slave);
  if (symlink(ptsname(fd), slave)) goto err;

  /* configure */
  if (grantpt(fd) || unlockpt(fd)) goto err;

  /* open slave to avoid HUP events when clients disconnect later */
  sl = open(slave, O_RDWR | O_NOCTTY);
  if (sl < 0) goto err;

  /* configure packet mode and external processing */
#ifdef TIOCEXT
  if (ioctl(fd, TIOCEXT, &(int){1})) goto err;
#elif defined(EXTPROC)
  struct termios t;
  if (tcgetattr(sl, &t)) goto err;
  t.c_lflag |= EXTPROC;
  if (tcsetattr(sl, TCSANOW, &t)) goto err;
#else
#warning "no external processing - baudrate settings won't work"
#endif
  if (ioctl(fd, TIOCPKT, &(int){1})) goto err;

  /* return handle */
  pty = malloc(sizeof(*pty));
  if (!pty) goto err;

  *pty = (struct tk_chan_s){ .fd = fd, .sl = sl };
  snprintf(pty->path, sizeof(pty->path), "%s", slave);
  return pty;

err:
  close(fd);
  if (sl >= 0) close(sl);
  unlink(slave);
  return NULL;
}


/* --- tk_close_pty -------------------------------------------------------- */

void
tk_close_pty(struct tk_chan_s **pty)
{
  if (!*pty) return;

  unlink((*pty)->path);
  close((*pty)->sl);
  close((*pty)->fd);
  free(*pty);
  *pty = NULL;
}


/* --- tk_recv ------------------------------------------------------------- */

int
tk_recv(struct tk_chan_s *pty, bool recv)
{
  ssize_t s;

  if (!recv) return 0;

  do {
    s = read(pty->fd, pty->msg, sizeof(pty->msg));
  } while(s < 0 && errno == EINTR);
  if (s <= 0) return 0;

  pty->len = s;
  pty->revent = pty->len > 0;
  return pty->revent;
}


/* --- tk_send ------------------------------------------------------------- */

int
tk_send(struct tk_chan_s *pty, const uint8_t *buf, size_t len)
{
  ssize_t s;

  while (len) {
    do {
      s = write(pty->fd, buf, len);
    } while (s < 0 && errno == EINTR);
    if (s < 0) return -1;

    buf += s;
    len -= s;
  }

  return 0;
}


/* --- tk_tcdescr ---------------------------------------------------------- */

static uint32_t
tk_baudrate(speed_t x)
{
  /* only the most useful values */
  switch(x) {
    case B0:		return 0;
    case B9600:		return 9600;
    case B19200:	return 19200;
    case B38400:	return 38400;

#ifdef B57600
    case B57600:	return 57600;
#endif
#ifdef B115200
    case B115200:	return 115200;
#endif
#ifdef B500000
    case B500000:	return 500000;
#endif
#ifdef B1000000
    case B1000000:	return 1000000;
#endif
#ifdef B2000000
    case B2000000:	return 2000000;
#endif
#ifdef B3000000
    case B3000000:	return 3000000;
#endif
#ifdef B4000000
    case B4000000:	return 4000000;
#endif
  }

  /* for those who encode baudrates as the actual numeric value */
  return x;
}

const char *
tk_tcdescr(const struct tk_chan_s *pty)
{
  static char buf[16];
  struct termios t;
  uint32_t b;
  char n, p, s;

  tcgetattr(pty->sl, &t);
#ifdef EXTPROC
  /* check our flag remains set, since the slave might mess with that */
  if (!(t.c_lflag & EXTPROC)) {
    /* XXX there seems to be issues with ICANON */
    /* XXX at the moment ICANON is set only for SLCAN, so it's OK to forget
     * XXX about it since the baudrate is irrelevant here */
    if (!(t.c_lflag & ICANON)) {
      t.c_lflag |= EXTPROC;
      tcsetattr(pty->sl, TCSANOW, &t);
    }
  }
#endif

  b = tk_baudrate(cfgetispeed(&t));
  switch (t.c_cflag & CSIZE) {
    default: case CS8: n = 8; break;
    case CS7: n = 7; break;
  }

  if (t.c_cflag & PARENB) p = 2;
  else if (t.c_cflag & PARODD) p = 1;
  else p = 0;

  if (t.c_cflag & CSTOPB) s = 2;
  else s = 1;

  sprintf(buf, "%c%c%c%c%c%c%c", b >> 24, b >> 16, b >> 8, b, n, p, s);
  return buf;
}
