/*
 * Copyright (c) 2021 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Wed Sep 29 2021
 */
#include "autoconf.h"

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/time.h>

#include <err.h>
#include <errno.h>
#include <poll.h>
#include <getopt.h>
#include <libgen.h>
#include <unistd.h>
#include <signal.h>
#include <stdarg.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>

#include "tk3-mux.h"


/* --- usage --------------------------------------------------------------- */

/** Print usage on a channel.
 *
 * \param[in] channel	Output channel
 * \param[in] argv0	Program invocation name
 */

static const char opts[] = "b:h";
static const struct option longopts[] = {
  { "basedir",	required_argument,	NULL,	'b'},
  { "version",	no_argument,		NULL,	-'v'},
  { "help",	no_argument,		NULL,	'h'},
  { NULL }
};

static char *argv0;
static const char *basedir = "/tmp";

static void
usage(FILE *channel)
{
  fprintf(
    channel,
    PACKAGE_NAME " I/O multiplexer\n\n"
    "Usage:\n"
    "  %1$s [-b|--basedir dir] tty|serial\n"
    "  %1$s [-h] [--version]\n"
    "\n"
    "Options:\n"
    "  -b, -basedir   change base directory for pty slaves (defaults to /tmp)\n"
    "  --version      display version number\n"
    "  -h, --help     print usage summary (this text)\n",
    argv0);
}


/* --- main ---------------------------------------------------------------- */

/* context */
enum tkch {
  TK_PPZ, TK_CAN, TK_UART8,
  tkchans
};
static struct tk_chan_s *tkchan[tkchans];
static uint8_t tkpfix[] = {
  [TK_CAN] = '4',
  [TK_UART8] = '8'
};
static const char *tkpath[] = {
  [TK_CAN] = "can",
  [TK_UART8] = "uart8"
};

static int leave = 0;

static void	tk_signal(int sig);
static int	tk_init(void);
static void	tk_fini(void);
static void	tk_exit(int status);
static int	tk_openppz(const char *path);
static int	tk_createptys(const char *basedir);
static int	tk_waitio(struct timeval deadline);
static void	infox(const char *fmt, ...);

int
main(int argc, char *argv[])
{
  enum tkch in, out;
  int delay;
  int c;

  /* init context */
  argv0 = basename(argv[0]);

  /* parse command line options */
  while ((c = getopt_long(argc, argv, opts, longopts, NULL)) != -1) {
    switch (c) {
      case 'b': basedir = optarg; break;
      case -'v': puts(PACKAGE_VERSION); exit(0); break;
      case 'h': usage(stdout); exit(0); break;

      case '?':
      default:
        usage(stderr); exit(1); break;
    }
  }
  argc -= optind;
  argv += optind;

  if (argc < 1) {
    usage(stderr);
    exit(1);
  }

  if (tk_init()) tk_exit(2);

  /* main loop */
  while(!leave) {

    /* open/configure paparazzi, retry with exponential backoff */
    for(delay = 100; !leave;
        poll(NULL, 0, delay = (delay < 10000 ? delay*5/4 : delay))) {
      if (!tk_openppz(argv[0])) break;
    }
    if (leave) break;

    /* create ptys */
    if (tk_createptys(basedir)) tk_exit(2);

    /* process messages */
    while(!leave) {
      uint8_t *msg, len;

      if (tk_waitio(tk_deadline(10000 /* arbitrary */)) < 0) break;

      for(in = 0; in < tkchans; in++) {
        if (!tkchan[in]->revent) continue;

        msg = tkchan[in]->msg;
        len = tkchan[in]->len;
        switch(in) {
          case TK_PPZ:
            /* decode ppz messages */
            switch(*msg) {
              default:
                /* bus forwarders */
                for(out = 0; out < tkchans; out++)
                  if (*msg == tkpfix[out]) {
                    tk_send(tkchan[out], msg + 1, len - 1);
                    break;
                  }
                if (out == tkchans)
                  warnx("received unknown message");
                break;

              case '?': /* ignored messages */
                break;
            }
            break;

          case TK_CAN:
          case TK_UART8: {
            /* decode packet messages */
            uint32_t ctrl = (len--, *msg++);

            if (ctrl == TIOCPKT_DATA) {
              /* forward data */
              while(len) {
                /* limit message size to 48 bytes */
                uint8_t s = len > 48 ? 48 : len;
                tk_send_msg(tkchan[TK_PPZ], "%c%b", tkpfix[in], msg, s);
                msg += s;
                len -= s;
              }
            } else if (ctrl & TIOCPKT_IOCTL) {
              tk_send_msg(
                tkchan[TK_PPZ], "t%c%b", tkpfix[in], tk_tcdescr(tkchan[in]), 7);
            }
            break;
          }

          case tkchans: /* not reached */ break;
        }
      }
    }

    /* clean up and start over */
    tk_fini();
  }

  tk_exit(0);
  return 0;
}


/* --- tk_signal ----------------------------------------------------------- */

static void
tk_signal(int sig)
{
  (void)sig;
  leave = 1;
}


/* --- tk_init ------------------------------------------------------------- */

static int
tk_init(void)
{
  struct sigaction sa = {
    .sa_handler = tk_signal
  };

  sigaction(SIGINT, &sa, NULL);
  sigaction(SIGQUIT, &sa, NULL);
  sigaction(SIGTERM, &sa, NULL);

  /* create basedir */
  if (mkdir(basedir, S_IRWXU|S_IRWXG|S_IRWXO))
    switch (errno) {
      case EEXIST: {
        struct stat sb;
        if (stat(basedir, &sb) < 0)
          errno = EEXIST;
        else if (!S_ISDIR(sb.st_mode))
          errno = ENOTDIR;
        else break;
      }

      default:
        warn("%s", basedir);
        return -1;
    }

  return 0;
}


/* --- tk_fini ------------------------------------------------------------- */

static void
tk_fini()
{
  enum tkch i;

  for(i = 0; i < tkchans; i++)
    switch(i) {
      case TK_PPZ: tk_close_tty(&tkchan[i]); break;
      default: tk_close_pty(&tkchan[i]); break;
    }
  rmdir(basedir);
}


/* --- tk_exit ------------------------------------------------------------- */

static void
tk_exit(int status)
{
  tk_fini();
  infox("terminating");
  exit(status);
}


/* --- tk_openppz ---------------------------------------------------------- */

static int
tk_openppz(const char *path)
{
  struct timeval deadline;
  struct tk_chan_s *p;
  double rev;

  p = tk_open_tty(path);
  if (!p) {
    warn("cannot connect to %s", path);
    return -1;
  }
  tkchan[TK_PPZ] = p;

  for(deadline = tk_deadline(1000); tk_timeout(deadline);) {
    if (tk_send_msg(p, "?")) {
      warn("%s", p->path);
      tk_close_tty(&p);
      return -1;
    }

    while (tk_waitio(tk_deadline(200)) > 0)
      if (p->revent && p->msg[0] == '?') {
        if (sscanf((char *)&p->msg[1], "chimera%lf", &rev) != 1)
          warnx("unsupported hardware device `%.*s'", p->len, &p->msg[1]);
        else if (rev < 1.1)
          warnx("chimera firmware version `%g' too old (1.1 required)", rev);
        else { /* match */
          infox("connected to %s", p->path);
          return 0;
        }

        tk_close_tty(&p);
        return -1;
      }
  }

  errno = ETIMEDOUT;
  warn("%s", p->path);
  tk_close_tty(&p);
  return -1;
}


/* --- tk_createptys ------------------------------------------------------- */

static int
tk_createptys(const char *basedir)
{
  char path[256];
  size_t i;
  int s;

  for(i = 0; i < tkchans; i++) {
    if (!tkpath[i]) continue;

    s = snprintf(path, sizeof(path), "%s/%s", basedir, tkpath[i]);
    if (s >= sizeof(path)) {
      errno = ENAMETOOLONG;
      goto err;
    }

    tkchan[i] = tk_open_pty(path);
    if (!tkchan[i]) goto err;
  }

  return 0;

err:
  warn("%s/%s", basedir, tkpath[i]);
  return -1;
}


/* --- tk_waitio ----------------------------------------------------------- */

static int
tk_waitio(struct timeval deadline)
{
  struct pollfd pfd[tkchans];
  size_t i;
  int r, s;
  bool recv;

  /* fill fd array */
  for(i = 0; i < tkchans; i++)
    pfd[i] = (struct pollfd){
      .fd = tkchan[i] ? tkchan[i]->fd : -1,
      .events = POLLIN | POLLPRI
    };

  /* this loop first checks for already available data, then polls and reads
   * new data from devices that are ready */
  for(recv = false;; recv = true) {
    /* process pending messages */
    for(i = s = 0; i < tkchans; i++) {
      if (!tkchan[i]) continue;
      tkchan[i]->revent = false;

      if (pfd[i].revents & POLLHUP) {
        warnx("%s hangup", tkchan[i]->path);
        if (i == TK_PPZ) return -1;
      }
      if (pfd[i].revents || !recv) {
        switch(i) {
          case TK_PPZ: r = tk_recv_msg(tkchan[i], recv); break;
          default: r = tk_recv(tkchan[i], recv); break;
        }
        if (r == 1) s++;
        else if (r < 0) warnx("%s", tkchan[i]->path);
      }
    }
    if (s) return s;

    /* wait for data */
    s = poll(pfd, tkchans, tk_timeout(deadline));
    if (s <= 0) return s;
  }

  return s;
}


/* --- infox --------------------------------------------------------------- */

static void
infox(const char *fmt, ...)
{
  va_list ap;
  va_start(ap, fmt);
  printf("%s: ", argv0);
  vprintf(fmt, ap);
  printf("\n");
  va_end(ap);
}
