/*
 * Copyright (c) 2019-2021,2024-2025 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Tue Jul 23 2019
 */
#include "autoconf.h"

#include <stdarg.h>
#include <string.h>

#include "tk3-paparazzi.h"

/* dma buffers */
static uint8_t tk3_mem(".d3ram", i2c4dmabuf[4]);
static uint8_t tk3_mem(".d2ram", i2c2dmabuf[4]);
static uint8_t tk3_mem(".d3ram", spi6dmabuf[4]);

/* bus communication state for I2C4, I2C2, SPI6 */
enum {
  TK3BUS_I2C4,
  TK3BUS_I2C2,
  TK3BUS_SPI6
};

static struct {
  union {
    I2C_TypeDef * const I2C; SPI_TypeDef * const SPI;
  };						/* sensor bus */

  union {
    struct { const stm32_bdma_stream_t *tx, *rx; } bdma;
    struct { const stm32_dma_stream_t *tx, *rx; } dma;
  };						/* dma streams */
  uint8_t (* const dmabuf)[4];			/* dma buffer */

  struct tk3event aiocb;			/* async events */
  volatile uint32_t status;			/* async errors */
} tk3sens_bus[] = {
  [TK3BUS_I2C4] = { .I2C = I2C4, .dmabuf = &i2c4dmabuf },
  [TK3BUS_I2C2] = { .I2C = I2C2, .dmabuf = &i2c2dmabuf },
  [TK3BUS_SPI6] = { .SPI = SPI6, .dmabuf = &spi6dmabuf }
};

#define TK3BUS_N	(sizeof(tk3sens_bus)/sizeof(*tk3sens_bus))

/* dma vs. bdma function invocation depending on bus n */
#define tk3busdma(x, n, ...)                                                   \
  ((n == TK3BUS_I2C4) ? (bdma ## x (tk3sens_bus[n].bdma. __VA_ARGS__)) :       \
   (n == TK3BUS_I2C2) ? (dma ## x (tk3sens_bus[n].dma. __VA_ARGS__)) :         \
   (n == TK3BUS_SPI6) ? (bdma ## x (tk3sens_bus[n].bdma. __VA_ARGS__)) :       \
   0 /* nop */)

#define TK3SENS_BUSY	(1<<31)


static void tk3sens_spi_rxtc_handler(void *arg, uint32_t flags);


/* --- tk3sens_hwinit ------------------------------------------------------ */

void
tk3sens_hwinit()
{
  /* reset buses */
  tk3sens_hwfini();
  rccResetI2C4();
  rccEnableI2C4(true);
  rccResetSPI6();
  rccEnableSPI6(true);

  /* reset I2C4 devices: bitbang 10 clock pulses at 100kHz to unlock any locked
   * slaves (10th pulse acts as NACK), then send a STOP (SDA from low to high
   * while CLK is high) */
  palSetPadMode(GPIOD, 12, PAL_MODE_OUTPUT_OPENDRAIN);
  palSetPadMode(GPIOD, 13, PAL_MODE_OUTPUT_OPENDRAIN);
  palSetPad(GPIOD, 12);
  palSetPad(GPIOD, 13);
  for (int i = 0; i < 10; i++) {
    tk3clk_delay(5);
    palSetPad(GPIOD, 12);
    tk3clk_delay(5);
    palClearPad(GPIOD, 12);
  }

  palClearPad(GPIOD, 13);
  tk3clk_delay(5);
  palSetPad(GPIOD, 12);
  tk3clk_delay(2);
  palSetPad(GPIOD, 13);
  tk3clk_delay(2);

  /* configure GPIO lines */
  palSetPadMode(GPIOD, 12, PAL_MODE_ALTERNATE(4) | PAL_STM32_OTYPE_OPENDRAIN);
  palSetPadMode(GPIOD, 13, PAL_MODE_ALTERNATE(4) | PAL_STM32_OTYPE_OPENDRAIN);

  palSetPadMode(GPIOB,  4, PAL_MODE_ALTERNATE(8));
  palSetPadMode(GPIOB,  5, PAL_MODE_ALTERNATE(8));
  palSetPadMode(GPIOA,  5, PAL_MODE_ALTERNATE(8));
  palSetPadMode(GPIOE,  3, PAL_MODE_OUTPUT_PUSHPULL);
  palSetPad(GPIOE, 3);

  /* setup DMA */
  tk3sens_bus[TK3BUS_I2C4].bdma.tx =
    bdmaStreamAlloc(STM32_I2C_I2C4_TX_BDMA_STREAM, 0, NULL, NULL);
  tk3busdma(
    StreamSetMode, TK3BUS_I2C4, tx,
    STM32_BDMA_CR_PSIZE_BYTE | STM32_BDMA_CR_MSIZE_BYTE |
    STM32_BDMA_CR_MINC | STM32_BDMA_CR_DIR_M2P |
    STM32_BDMA_CR_PL(STM32_I2C_I2C4_BDMA_PRIORITY));
  tk3busdma(SetRequestSource, TK3BUS_I2C4, tx, STM32_DMAMUX2_I2C4_TX);
  tk3busdma(StreamSetPeripheral, TK3BUS_I2C4, tx, &I2C4->TXDR);

  tk3sens_bus[TK3BUS_I2C4].bdma.rx =
    bdmaStreamAlloc(STM32_I2C_I2C4_RX_BDMA_STREAM, 0, NULL, NULL);
  tk3busdma(
    StreamSetMode, TK3BUS_I2C4, rx,
    STM32_BDMA_CR_PSIZE_BYTE | STM32_BDMA_CR_MSIZE_BYTE |
    STM32_BDMA_CR_MINC | STM32_BDMA_CR_DIR_P2M |
    STM32_BDMA_CR_PL(STM32_I2C_I2C4_BDMA_PRIORITY));
  tk3busdma(SetRequestSource, TK3BUS_I2C4, rx, STM32_DMAMUX2_I2C4_RX);
  tk3busdma(StreamSetPeripheral, TK3BUS_I2C4, rx, &I2C4->RXDR);

  tk3sens_bus[TK3BUS_SPI6].bdma.tx =
    bdmaStreamAlloc(STM32_SPI_SPI6_TX_BDMA_STREAM, 0, NULL, NULL);
  tk3busdma(
    StreamSetMode, TK3BUS_SPI6, tx,
    STM32_BDMA_CR_PSIZE_BYTE | STM32_BDMA_CR_MSIZE_BYTE |
    STM32_BDMA_CR_MINC | STM32_BDMA_CR_CIRC | STM32_BDMA_CR_DIR_M2P |
    STM32_BDMA_CR_PL(STM32_SPI_SPI6_BDMA_PRIORITY));
  tk3busdma(SetRequestSource, TK3BUS_SPI6, tx, STM32_DMAMUX2_SPI6_TX);
  tk3busdma(StreamSetPeripheral, TK3BUS_SPI6, tx, &SPI6->TXDR);

  tk3sens_bus[TK3BUS_SPI6].bdma.rx =
    bdmaStreamAlloc(STM32_SPI_SPI6_RX_BDMA_STREAM,
                    STM32_SPI_SPI6_IRQ_PRIORITY,
                    tk3sens_spi_rxtc_handler, (void *)TK3BUS_SPI6);
  tk3busdma(
    StreamSetMode, TK3BUS_SPI6, rx,
    STM32_BDMA_CR_PSIZE_BYTE | STM32_BDMA_CR_MSIZE_BYTE |
    STM32_BDMA_CR_MINC | STM32_BDMA_CR_DIR_P2M |
    STM32_BDMA_CR_PL(STM32_SPI_SPI6_BDMA_PRIORITY) |
    STM32_BDMA_CR_TCIE);
  tk3busdma(SetRequestSource, TK3BUS_SPI6, rx, STM32_DMAMUX2_SPI6_RX);
  tk3busdma(StreamSetPeripheral, TK3BUS_SPI6, rx, &SPI6->RXDR);

  /* I2C 400kHz */
  I2C4->TIMINGR =
    ( 14 << I2C_TIMINGR_PRESC_Pos) |	/* 8MHz - 125ns */
    ( 3 << I2C_TIMINGR_SCLDEL_Pos) |
    ( 3 << I2C_TIMINGR_SDADEL_Pos) |
    ( 8 << I2C_TIMINGR_SCLH_Pos) |	/* 1125ns */
    ( 9 << I2C_TIMINGR_SCLL_Pos);	/* 1250ns */
  I2C4->TIMEOUTR = 58;			/* × 2048 / 120MHz = 1ms SCL low */

  /* SPI 24MHz */
  SPI6->CR1 =
    SPI_CR1_SSI | SPI_CR1_MASRX; /* required _before_ enabling master mode */
  SPI6->CFG1 =
    SPI_CFG1_TXDMAEN | SPI_CFG1_RXDMAEN | (7 << SPI_CFG1_DSIZE_Pos);
  SPI6->CFG2 = SPI_CFG2_MASTER | SPI_CFG2_SSM;

  /* setup interrupts */
  nvicEnableVector(STM32_I2C4_EVENT_NUMBER, STM32_I2C_I2C4_IRQ_PRIORITY);
  nvicEnableVector(STM32_I2C4_ERROR_NUMBER, STM32_I2C_I2C4_IRQ_PRIORITY);
  I2C4->CR1 =
    I2C_CR1_TXDMAEN | I2C_CR1_RXDMAEN |
    I2C_CR1_TCIE | I2C_CR1_STOPIE | I2C_CR1_ERRIE | I2C_CR1_NACKIE;

  /* enable peripherals */
  for(int bus = 0; bus < TK3BUS_N; bus++)
    tk3sens_bus[bus].status = 0;
  I2C4->CR1 |= I2C_CR1_PE;
}


/* --- tk3sens_hwfini ------------------------------------------------------ */

void
tk3sens_hwfini()
{
  for(int bus = 0; bus < TK3BUS_N; bus++) {
    tk3sens_bus[bus].status |= 1;
    tk3sens_bus[bus].status &= ~TK3SENS_BUSY;
    tk3ev_schedule(&tk3sens_bus[bus].aiocb, TK3EV_NOW);
  }

  I2C4->CR1 &= ~I2C_CR1_PE;
  SPI6->CR1 &= ~SPI_CR1_SPE;
  if (tk3sens_bus[TK3BUS_I2C4].bdma.tx) {
    tk3busdma(StreamFree, TK3BUS_I2C4, tx);
    tk3sens_bus[TK3BUS_I2C4].bdma.tx = NULL;
  }
  if (tk3sens_bus[TK3BUS_I2C4].bdma.rx) {
    tk3busdma(StreamFree, TK3BUS_I2C4, rx);
    tk3sens_bus[TK3BUS_I2C4].bdma.rx = NULL;
  }
  if (tk3sens_bus[TK3BUS_SPI6].bdma.tx) {
    tk3busdma(StreamFree, TK3BUS_SPI6, tx);
    tk3sens_bus[TK3BUS_SPI6].bdma.tx = NULL;
  }
  if (tk3sens_bus[TK3BUS_SPI6].bdma.rx) {
    tk3busdma(StreamFree, TK3BUS_SPI6, rx);
    tk3sens_bus[TK3BUS_SPI6].bdma.rx = NULL;
  }

  nvicDisableVector(STM32_I2C4_EVENT_NUMBER);
  nvicDisableVector(STM32_I2C4_ERROR_NUMBER);
  nvicDisableVector(STM32_SPI6_NUMBER);
  rccDisableI2C4();
  rccDisableSPI6();

  palSetPadMode(GPIOD, 12, PAL_MODE_RESET);
  palSetPadMode(GPIOD, 13, PAL_MODE_RESET);
  palSetPadMode(GPIOB,  4, PAL_MODE_RESET);
  palSetPadMode(GPIOB,  5, PAL_MODE_RESET);
  palSetPadMode(GPIOA,  5, PAL_MODE_RESET);
  palSetPadMode(GPIOE,  3, PAL_MODE_RESET);
}


/* --- tk3sens_write ------------------------------------------------------- */

int
tk3sens_write(uint32_t dev, uint8_t n, ...)
{
  unsigned int bus = dev >> 16;
  va_list ap;

  if (bus >= sizeof(tk3sens_bus)/sizeof(*tk3sens_bus))
    return TK3SENS_BUSY;

  /* loop on writes */
  va_start(ap, n);
  while (n--) {
    (*tk3sens_bus[bus].dmabuf)[0] = va_arg(ap, unsigned int);
    (*tk3sens_bus[bus].dmabuf)[1] = va_arg(ap, unsigned int);
    __DMB(); /* sync dmabuf */

    tk3busdma(StreamSetMemory0, bus, tx, *tk3sens_bus[bus].dmabuf);
    tk3busdma(StreamSetTransactionSize, bus, tx, 2);
    tk3busdma(StreamEnable, bus, tx);

    tk3sens_bus[bus].status = TK3SENS_BUSY;
    switch(bus) {
      case TK3BUS_I2C4:
        tk3busdma(StreamSetTransactionSize, bus, rx, 0);

        tk3sens_bus[bus].I2C->TIMEOUTR |= I2C_TIMEOUTR_TIMOUTEN;
        tk3sens_bus[bus].I2C->CR2 =
          I2C_CR2_START |
          2 << I2C_CR2_NBYTES_Pos |
          (dev & 0xff) << 1;
        break;

      case TK3BUS_SPI6:
        /* to receive the unused data */
        tk3busdma(StreamSetTransactionSize, bus, rx, 2);
        tk3busdma(StreamSetMemory0, bus, rx, &(*tk3sens_bus[bus].dmabuf)[2]);
        tk3busdma(StreamEnable, bus, rx);

        palClearPad(GPIOE,  3); /* nCS - enable chip select */
        tk3sens_bus[bus].SPI->CR2 = 2;
        tk3sens_bus[bus].SPI->IFCR = 0xffff; /* required to clear flags */
        tk3sens_bus[bus].SPI->CR1 |= SPI_CR1_SPE;
        tk3sens_bus[bus].SPI->CR1 |= SPI_CR1_CSTART;
        break;
    }

    do { __WFI(); } while(tk3sens_bus[bus].status & TK3SENS_BUSY);

    if (tk3sens_bus[bus].status) break;
  }
  va_end(ap);

  return tk3sens_bus[bus].status;
}


/* --- tk3sens_read -------------------------------------------------------- */

int
tk3sens_read(uint32_t dev, uint8_t n, uint8_t addr, volatile void *value)
{
  unsigned int bus = dev >> 16;

  tk3sens_aioread(dev, n, addr, value, NULL);

  do { __WFI(); } while(tk3sens_bus[bus].status & TK3SENS_BUSY);

  return tk3sens_bus[bus].status;
}

void
tk3sens_aioread(uint32_t dev, uint8_t n, uint8_t addr, volatile void *value,
               void (*cb)(void *))
{
  unsigned int bus = dev >> 16;

  if (bus >= sizeof(tk3sens_bus)/sizeof(*tk3sens_bus)) return;

  tk3ev_set(&tk3sens_bus[bus].aiocb, cb, (void *)&tk3sens_bus[bus].status);

  (*tk3sens_bus[bus].dmabuf)[0] = addr;
  __DMB(); /* sync addr */

  tk3busdma(StreamSetMemory0, bus, tx, *tk3sens_bus[bus].dmabuf);
  tk3busdma(StreamSetTransactionSize, bus, tx, 1);
  tk3busdma(StreamEnable, bus, tx);

  tk3sens_bus[bus].status = TK3SENS_BUSY;
  switch(bus) {
    case TK3BUS_I2C4:
      tk3busdma(StreamSetTransactionSize, bus, rx, n);
      tk3busdma(StreamSetMemory0, bus, rx, value);

      tk3sens_bus[bus].I2C->TIMEOUTR |= I2C_TIMEOUTR_TIMOUTEN;
      tk3sens_bus[bus].I2C->CR2 =
        I2C_CR2_START |
        1 << I2C_CR2_NBYTES_Pos |
        (dev & 0xff) << 1;
      break;

    case TK3BUS_SPI6:
      /* one extra byte is received during transmission of read address, so
       * the receive buffer in 'value' must have one extra padding byte at the
       * beginning. This is done with a tk3_padmem() buffer. */
      tk3busdma(StreamSetTransactionSize, bus, rx, n + 1);
      tk3busdma(StreamSetMemory0, bus, rx, value - 1);
      tk3busdma(StreamEnable, bus, rx);

      palClearPad(GPIOE,  3); /* nCS - enable chip select */
      tk3sens_bus[bus].SPI->CR2 = n + 1;
      tk3sens_bus[bus].SPI->CR1 |= SPI_CR1_SPE;
      tk3sens_bus[bus].SPI->CR1 |= SPI_CR1_CSTART;
      break;
  }
}


/* --- STM32_I2C4_EVENT_HANDLER -------------------------------------------- */

static void
tk3sens_i2c_event_handler(unsigned int bus)
{
  uint32_t isr = tk3sens_bus[bus].I2C->ISR;
  tk3sens_bus[bus].I2C->ICR = isr & (I2C_ICR_STOPCF | I2C_ICR_NACKCF);

  /* abort transaction on NACK */
  if (isr & I2C_ISR_NACKF) {
    tk3busdma(StreamDisable, bus, tx);
    tk3busdma(StreamDisable, bus, rx);

    tk3sens_bus[bus].I2C->ICR |= I2C_ICR_ADDRCF;
    tk3sens_bus[bus].I2C->CR2 |= I2C_CR2_STOP;
    tk3sens_bus[bus].status |= I2C_ISR_NACKF;
    return;
  }

  /* transfer complete */
  if (isr & I2C_ISR_TC) {
    tk3busdma(StreamDisable, bus, tx);

    /* pending read */
    if (tk3busdma(StreamGetTransactionSize, bus, rx) > 0) {
      tk3busdma(StreamEnable, bus, rx);

      tk3sens_bus[bus].I2C->CR2 =
        (tk3sens_bus[bus].I2C->CR2 & ~I2C_CR2_NBYTES) |
        I2C_CR2_RD_WRN |
        (tk3busdma(StreamGetTransactionSize, bus, rx) << I2C_CR2_NBYTES_Pos);

      tk3sens_bus[bus].I2C->CR2 |= I2C_CR2_START;
      return;
    }

    tk3busdma(StreamDisable, bus, rx);

    tk3sens_bus[bus].I2C->CR2 |= I2C_CR2_STOP;
    return;
  }

  /* transaction complete */
  if (isr & I2C_ISR_STOPF) {
    tk3sens_bus[bus].I2C->TIMEOUTR &= ~I2C_TIMEOUTR_TIMOUTEN;
    tk3sens_bus[bus].status &= ~TK3SENS_BUSY;
    tk3ev_schedule(&tk3sens_bus[bus].aiocb, TK3EV_NOW);
    return;
  }
}

OSAL_IRQ_HANDLER(STM32_I2C4_EVENT_HANDLER)
{
  tk3sens_i2c_event_handler(0);
}


/* --- STM32_I2C4_ERROR_HANDLER -------------------------------------------- */

static void
tk3sens_i2c_error_handler(unsigned int bus)
{
  uint32_t isr;

  isr = tk3sens_bus[bus].I2C->ISR &
        (I2C_ISR_TIMEOUT | I2C_ISR_BERR | I2C_ISR_ARLO);
  tk3sens_bus[bus].I2C->ICR = isr;
  tk3sens_bus[bus].status |= isr;

  if (isr & I2C_ISR_TIMEOUT) tk3msg_log(TK3CH_USB0, "Esensor bus timeout");
  if (isr & I2C_ISR_BERR) tk3msg_log(TK3CH_USB0, "Esensor bus error");
  if (isr & I2C_ISR_ARLO)
    tk3msg_log(TK3CH_USB0, "Esensor bus arbitration lost");

  /* cancel transaction */
  tk3sens_bus[bus].I2C->TIMEOUTR &= ~I2C_TIMEOUTR_TIMOUTEN;
  tk3busdma(StreamDisable, bus, tx);
  tk3busdma(StreamDisable, bus, rx);

  /* invoke callback */
  tk3sens_bus[bus].status &= ~TK3SENS_BUSY;
  tk3ev_schedule(&tk3sens_bus[bus].aiocb, TK3EV_NOW);
}

OSAL_IRQ_HANDLER(STM32_I2C4_ERROR_HANDLER)
{
  tk3sens_i2c_error_handler(0);
}


/* --- STM32_SPI6_HANDLER -------------------------------------------------- */

/* this is actually the BDMA IRQ but corresponding to full SPI6 transfer */

static void
tk3sens_spi_rxtc_handler(void *arg, uint32_t flags)
{
  unsigned int bus = (unsigned int)arg;

  /* transaction done */

  /* bdmaStreamDisable() removes STM32_BDMA_CR_TCIE, so disable rx manually */
  tk3sens_bus[bus].bdma.rx->channel->CCR &= ~STM32_BDMA_CR_EN;
  tk3busdma(StreamDisable, bus, tx);

  palSetPad(GPIOE, 3); /* nCS - clear chip select */
  tk3sens_bus[bus].SPI->CR1 &= ~SPI_CR1_SPE;
  tk3sens_bus[bus].SPI->IFCR |= SPI_IFCR_TXTFC | SPI_IFCR_EOTC;

  tk3sens_bus[bus].status &= ~TK3SENS_BUSY;
  tk3ev_schedule(&tk3sens_bus[bus].aiocb, TK3EV_NOW);
}
