/*
 * Copyright (c) 2019-2021,2024 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Thu Aug  1 2019
 */
#include "autoconf.h"

#include "tk3-paparazzi.h"

static const stm32_dma_stream_t *dma;

static struct __attribute__((__packed__)) {
  uint16_t vbat;
} tk3_nocache(measure);

static struct {
  uint16_t vbat;
} minpeak;

static tk3timer(powercb);


/* --- tk3pwr_hwinit ------------------------------------------------------- */

/* Measure VBATmeas (PC0) */

int
tk3pwr_hwinit()
{
  /* reset min peak values */
  minpeak.vbat = 0xffff;

  /* configure VBAT_MEAS GPIO */
  palSetPadMode(GPIOC, 0, PAL_MODE_INPUT_ANALOG);

  /* setup dma for ADC1 */
  dma = dmaStreamAlloc(STM32_ADC1_DMA_STREAM, 0/*unused*/, NULL, NULL);
  dmaStreamSetMode(
    dma,
    STM32_DMA_CR_PSIZE_HWORD | STM32_DMA_CR_MSIZE_HWORD |
    STM32_DMA_CR_MINC | STM32_DMA_CR_CIRC | STM32_DMA_CR_DIR_P2M |
    STM32_DMA_CR_PL(STM32_ADC1_DMA_PRIORITY));
  dmaSetRequestSource(dma, STM32_DMAMUX1_ADC1);
  dmaStreamSetPeripheral(dma, &ADC1->DR);
  dmaStreamSetMemory0(dma, &measure);
  dmaStreamSetTransactionSize(dma, 2);
  dmaStreamEnable(dma);

  /* enable ADC1 */
  rccEnableADC12(true);
  nvicEnableVector(STM32_ADC12_NUMBER, STM32_ADC_ADC12_IRQ_PRIORITY);

  /* clock PLL2_P/2 5MHz, 1.5 ticks delay between samples */
  ADC12_COMMON->CCR = 0;

  /* power-on */
  ADC1->CR = ADC_CR_ADVREGEN;
  while (!(ADC1->ISR & ADC_ISR_LDORDY)) /* wait */;

  /* self calibration */
  ADC1->CR |= ADC_CR_ADCALLIN;
  ADC1->CR |= ADC_CR_ADCAL;
  while (ADC1->CR & ADC_CR_ADCAL) /* wait */;

  /* VBATmeas sensing */
  ADC1->SMPR2 = 7 << ADC_SMPR2_SMP10_Pos;	/* 810.5 ticks on channel 10 */
  ADC1->PCSEL = ADC_PCSEL_PCSEL_10;		/* preselect #10 */
  ADC1->SQR1 =
    0 << ADC_SQR1_L_Pos |			/* 1 channel */
    10 << ADC_SQR1_SQ1_Pos;			/* #1 channel 10 PC0 */

  ADC1->CFGR =
    (0 << ADC_CFGR_RES_Pos) |			/* 16 bits */
    ADC_CFGR_AWD1SGL |				/* single watchdog channel */
    (10 << ADC_CFGR_AWD1CH_Pos) |		/* watchdog on channel 10 */
    ADC_CFGR_CONT |				/* continuous mode */
    (3 << ADC_CFGR_DMNGT_Pos);			/* DMA circular mode */

  ADC1->HTR1 = 0x2200;				/* power on detect */
  ADC1->LTR1 = 0;

  ADC1->IER =
    ADC_IER_EOCIE |				/* interrupt after group */
    ADC_IER_AWD1IE;				/* watchdog 1 */

  /* enable */
  ADC1->CR |= ADC_CR_ADEN;
  while (!(ADC1->ISR & ADC_ISR_ADRDY)) /* wait */;

  /* start */
  ADC1->CFGR |= ADC_CFGR_AWD1EN;
  ADC1->CR |= ADC_CR_ADSTART;

  return 0;
}


/* --- tk3pwr_vbat --------------------------------------------------------- */

uint16_t
tk3pwr_vbat()
{
  uint32_t vbat;

  /* 1000 (mV) * 3.3/0xffff (ADC) * (18k + 2.2k / 2.2k) (voltage divider) */
  vbat = (uint32_t)minpeak.vbat * 2020 / 4369;

  minpeak.vbat = 0xffff;
  return vbat;
}


/* --- STM32_ADC12_HANDLER ------------------------------------------------- */

OSAL_IRQ_HANDLER(STM32_ADC12_HANDLER)
{
  uint32_t isr = ADC1->ISR;
  ADC1->ISR |= isr;

  /* EOC flag is reset by DMA transfer, so never asserted here */
  if (minpeak.vbat > measure.vbat) minpeak.vbat = measure.vbat;

  /* deal with watchdog interrupts */
  if (!(isr & ADC_ISR_AWD1)) return;

  if (ADC1->LTR1 == 0) {
    /* power on event: switch to power off (< 4V) detection */
    ADC1->HTR1 = 0xffff;
    ADC1->LTR1 = 0x2200;

    /* trigger callback after 250ms steady on */
    tk3clk_settimer(&powercb, 0, 250000, TK3EV_IDLE, tk3pwr_on, tk3pwr_on);
  } else {
    /* power off event: switch to power on (> 4V) detection */
    ADC1->HTR1 = 0x2200;
    ADC1->LTR1 = 0x0;

    /* trigger callback after 100ms steady off */
    tk3clk_settimer(&powercb, 0, 100000, TK3EV_IDLE, tk3pwr_off, tk3pwr_off);
  }
}
