/*
 * Copyright (c) 2019-2021,2023-2025 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Fri Jun 14 2019
 */
#include "autoconf.h"

#include "tk3-paparazzi.h"

/* io buffers */
static uint8_t tk3_nocache(mem[2][128]);
static struct tk3iob inb = tk3iob_init("mkbl input", mem[0], 32);
static struct tk3iob outb = tk3iob_init("mkbl output", mem[1], 32);


/* I2C communication state */
static struct {
  uint32_t active;
} rx;

static struct {
  const stm32_dma_stream_t *dma;
  uint32_t nack;
} tx;


/* --- tk3mkbl_hwinit ------------------------------------------------------ */

void
tk3mkbl_hwinit()
{
  /* reset I2C */
  rccResetI2C2();
  rccEnableI2C2(true);

  rx.active = 0;

  /* bitbang 10 clock pulses at 100kHz to unlock any locked slaves (10th pulse
   * acts as NACK), then send a STOP (SDA from low to high while CLK is high) */
  palSetPad(GPIOB, 10);
  palSetPad(GPIOB, 11);
  palSetPadMode(GPIOB, 10, PAL_MODE_OUTPUT_OPENDRAIN);
  palSetPadMode(GPIOB, 11, PAL_MODE_OUTPUT_OPENDRAIN);
  for (int i = 0; i < 10; i++) {
    tk3clk_delay(5);
    palSetPad(GPIOB, 10);
    tk3clk_delay(5);
    palClearPad(GPIOB, 10);
  }

  palClearPad(GPIOB, 11);
  tk3clk_delay(5);
  palSetPad(GPIOB, 10);
  tk3clk_delay(2);
  palSetPad(GPIOB, 11);
  tk3clk_delay(2);

  /* configure SDA/SCL lines */
  palSetPadMode(GPIOB, 10, PAL_MODE_ALTERNATE(4) | PAL_STM32_OTYPE_OPENDRAIN);
  palSetPadMode(GPIOB, 11, PAL_MODE_ALTERNATE(4) | PAL_STM32_OTYPE_OPENDRAIN);

  /* DMA */
  tx.dma = dmaStreamAlloc(STM32_I2C_I2C2_TX_DMA_STREAM, 0, NULL, NULL);
  dmaStreamSetMode(
    tx.dma,
    STM32_DMA_CR_PSIZE_BYTE | STM32_DMA_CR_MSIZE_BYTE |
    STM32_DMA_CR_MINC | STM32_DMA_CR_DIR_M2P |
    STM32_DMA_CR_CHSEL(
      STM32_DMA_GETCHANNEL(
        STM32_I2C_I2C2_TX_DMA_STREAM, STM32_I2C2_TX_DMA_CHN)) |
    STM32_DMA_CR_PL(STM32_I2C_I2C2_DMA_PRIORITY));
  dmaStreamSetPeripheral(tx.dma, &I2C2->TXDR);
  dmaSetRequestSource(tx.dma, STM32_DMAMUX1_I2C2_TX);
  dmaStreamDisable(tx.dma);

  /* 500kHz bus - including ANF and DNF (15×I2CCLK) filters */
  I2C2->TIMINGR =
    ( 7 << I2C_TIMINGR_PRESC_Pos) |	/* 15MHz - 66ns */
    ( 3 << I2C_TIMINGR_SCLDEL_Pos) |
    ( 3 << I2C_TIMINGR_SDADEL_Pos) |
    ( 12 << I2C_TIMINGR_SCLH_Pos) |	/* 867ns */
    ( 13 << I2C_TIMINGR_SCLL_Pos);	/* 933ns */

  /* GC address */
  I2C2->CR2 = 0;

  /* ~500µs SCL low timeout */
  I2C2->TIMEOUTR = /* × 2048 / 120MHz */
    (29 << I2C_TIMEOUTR_TIMEOUTA_Pos) | I2C_TIMEOUTR_TIMOUTEN;

  /* interrupts */
  nvicEnableVector(STM32_I2C2_EVENT_NUMBER, STM32_I2C_I2C2_IRQ_PRIORITY);
  nvicEnableVector(STM32_I2C2_ERROR_NUMBER, STM32_I2C_I2C2_IRQ_PRIORITY);
  I2C2->CR1 =
    (15 << I2C_CR1_DNF_Pos) | /* digital filter for spikes up to ~400ns */
    I2C_CR1_TXDMAEN |
    I2C_CR1_TCIE | I2C_CR1_RXIE |
    I2C_CR1_STOPIE | I2C_CR1_ERRIE;

  /* enable peripheral */
  I2C2->CR1 |= I2C_CR1_PE;
}


/* --- tk3mkbl_hwfini ------------------------------------------------------ */

void
tk3mkbl_hwfini()
{
  nvicDisableVector(STM32_I2C2_EVENT_NUMBER);
  nvicDisableVector(STM32_I2C2_ERROR_NUMBER);
  I2C2->CR1 &= ~I2C_CR1_PE;

  if (tx.dma) {
    dmaStreamFree(tx.dma);
    tx.dma = NULL;
  }

  /* empty output buffer */
  while(tk3iob_getc(&outb, &(uint8_t){0}))
    /* empty body */;

  rccDisableI2C2();
  palSetPadMode(GPIOB, 10, PAL_MODE_RESET);
  palSetPadMode(GPIOB, 11, PAL_MODE_RESET);
}


/* --- tk3mkbl_iob --------------------------------------------------------- */

struct tk3iob *
tk3mkbl_iniob()
{
  return &inb;
}

struct tk3iob *
tk3mkbl_outiob()
{
  return &outb;
}


/* --- tk3mkbl_hwsend ------------------------------------------------------ */

void
tk3mkbl_hwsend(const struct tk3iob_extent *e)
{
  /* master transmit, general call */
  dmaStreamSetMemory0(tx.dma, e->r);
  dmaStreamSetTransactionSize(tx.dma, e->len);
  dmaStreamEnable(tx.dma);

  rx.active = 0;
  I2C2->CR2 = I2C_CR2_START | (e->len << I2C_CR2_NBYTES_Pos);
}


/* --- tk3mkbl_hwrecv ------------------------------------------------------ */

void
tk3mkbl_hwrecv(const uint8_t sladdr)
{
  /* master receive, current slave address - at most 32 bytes will be
   * received in this transfer. There is no such big message and it is
   * important to limit the read length in case the end-of-message ('$')
   * marker gets lost or corrupted. Otherwise, too many 0xff will be put in
   * the input buffer, filling it and potentially blocking the
   * communication. */
  rx.active = 1;
  I2C2->CR2 =
    I2C_CR2_START |
    (sladdr << 1) | I2C_CR2_RD_WRN |
    (32 << I2C_CR2_NBYTES_Pos);
}


/* --- STM32_I2C2_EVENT_HANDLER -------------------------------------------- */

OSAL_IRQ_HANDLER(STM32_I2C2_EVENT_HANDLER)
{
  uint32_t isr = I2C2->ISR;
  I2C2->ICR = isr & (I2C_ICR_STOPCF | I2C_ICR_NACKCF);

  /* nack received: nop */
  if (isr & I2C_ISR_NACKF) tx.nack = 1;

  /* data received */
  if (isr & I2C_ISR_RXNE) {
    uint8_t c = I2C2->RXDR;

    if (rx.active) {
      tk3iob_putc(&inb, c);
      if (c == '$') {
        I2C2->CR2 |= I2C_CR2_STOP;
        rx.active = 0;
      }
    }
    /* discard extra meaningless byte (only one) after '$' */
    /* FALLTHROUGH in case stop condition was already detected */
  }

  /* transfer complete or stop condition */
  if (isr & (I2C_ISR_TC | I2C_ISR_STOPF)) {
    dmaStreamDisable(tx.dma);
    if (rx.active && !tx.nack && inb.cb) {
      /* in this case, data was recieved but not the end-of-message marker
       * (either corrupt or not yet arrived): signal the buffer so that it gets
       * read by the main loop and does not fill up with garbage. Since the
       * messages must be received atomically, this data must be discarded
       * anyway. */
      inb.cb(&inb);
    }
    tx.nack = 0;

    tk3mkbl_complete();
    return;
  }
}


/* --- STM32_I2C2_ERROR_HANDLER -------------------------------------------- */

OSAL_IRQ_HANDLER(STM32_I2C2_ERROR_HANDLER)
{
  uint32_t isr;

  isr = I2C2->ISR & (I2C_ISR_TIMEOUT | I2C_ISR_BERR | I2C_ISR_ARLO);
  I2C2->ICR = isr;

  /* reset bus state and start over after some time */
  tk3mkbl_restart(NULL);

  if (isr & I2C_ISR_TIMEOUT)
    tk3msg_log(TK3CH_USB0, "Emotor bus timeout");
  if (isr & I2C_ISR_ARLO)
    tk3msg_log(TK3CH_USB0, "Emotor bus arbitration lost");
}
