/*
 * Copyright (c) 2019-2021,2024-2025 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Sun Aug 18 2019
 */
#include "autoconf.h"

#include "tk3-paparazzi.h"

/* ensure this triggers a redifinition error if this is defined */
#define STM32_TIM17_IS_USED	tk3clk

#if STM32_TIMCLK2 / TK3CLK_FREQ < 1
#error "TK3CLK_FREQ too high"
#endif

static volatile uint32_t tk3clk_ush = 0;


/* --- tk3clk_hwinit ------------------------------------------------------- */

int
tk3clk_hwinit()
{
  /* configure TIM17 at TK3CLK_FREQ frequency to overflow every 1<<16 µs */
  rccEnableTIM17(true);
  rccResetTIM17();

  nvicEnableVector(STM32_TIM17_NUMBER, STM32_IRQ_TIM17_PRIORITY);

  TIM17->CR1 = TIM_CR1_UIFREMAP;
  TIM17->CR2 = 0;
  TIM17->ARR = 0xffff;
  TIM17->PSC = STM32_TIMCLK1 / TK3CLK_FREQ - 1;
  TIM17->DIER = TIM_DIER_CC1IE | TIM_DIER_UIE;
  TIM17->EGR = 0;
  TIM17->CCMR1 = 0;
  TIM17->CCR1 = 0;
  TIM17->CNT = 0;
  TIM17->SR = 0;
  TIM17->CR1 |= TIM_CR1_CEN;

  /* enable CYCCNT to count CPU utilization time */
  if (DWT->CTRL & DWT_CTRL_NOCYCCNT_Msk) {
    tk3msg_log(TK3CH_USB0, "Ano CPU counter available");
  } else {
    CoreDebug->DEMCR |= CoreDebug_DEMCR_TRCENA_Msk;
    DWT->LAR = 0xc5acce55;
    DWT->CYCCNT = 0;
    DWT->CTRL |= DWT_CTRL_CYCCNTENA_Msk;
  }

  return 0;
}


/* --- tk3clk_cpucounter --------------------------------------------------- */

/* cpu cycles in µs since last call */

uint32_t
tk3clk_cpucounter()
{
  uint32_t cpu;

  cpu = DWT->CYCCNT;
  DWT->CYCCNT = 0;
  return cpu/(STM32_SYS_CK/1000000);
}


/* --- tk3clk_signal ------------------------------------------------------- */

void
tk3clk_signal()
{
  TIM17->EGR |= TIM_EGR_CC1G;
}


/* --- tk3clk_us ----------------------------------------------------------- */

uint32_t
tk3clk_us()
{
  uint32_t h, l;

  with_syslock() {
    l = TIM17->CNT;
    h = tk3clk_ush;
  }

  return (h | (l & 0xffff)) + ((l & TIM_CNT_UIFCPY) >> 15);
}


/* --- STM32_TIM17_HANDLER ------------------------------------------------- */

OSAL_IRQ_HANDLER(STM32_TIM17_HANDLER)
{
  uint32_t sr;
  uint32_t now, next;

  with_syslock() {
    sr = TIM17->SR;
    TIM17->SR = 0;

    if (sr & TIM_SR_UIF) {
      tk3clk_ush += 1<<16;
    }
  }

  if (sr & TIM_SR_CC1IF) {
    now = tk3clk_us();
    do {
      next = tk3clk_alarm(now);
      TIM17->CCR1 = next & 0xffff;
      now = tk3clk_us();
    } while (next - now > 0xfffff000 && !(TIM17->SR & TIM_SR_CC1IF));
  }
}
