/*
 * Copyright (c) 2024-2025 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Mon Nov 18 2024
 */
#include "autoconf.h"

#include <string.h>

#include "tk3-paparazzi.h"
#include "tk3-boot.h"


void	set_option_bytes(void);


/* --- Reset_Handler ------------------------------------------------------- */

#define BOOTMODE_BOOTADDR	0xb0074dd2ULL	/* magic constant */

/* boot method - use a 64bit word and 8 bytes aligment to avoid ECC issues */
static uint64_t bootopt __attribute__ ((aligned(8)));


void __attribute__((section(".trampoline"), naked, noreturn))
Reset_Handler(void)
{
  extern uintptr_t _crt0_entry;
  uint32_t *app;

  /* check jump address after a software reset */
  if ((RCC->RSR & RCC_RSR_SFTRSTF) &&
      (uint32_t)(bootopt >> 32) == BOOTMODE_BOOTADDR) {
    app = (uint32_t *)(uintptr_t)bootopt;
    bootopt = 0;
    __DSB();
    __set_MSP(app[0]);
    goto *(void *)app[1];
  }

  /* reset reset flags */
  if (RCC->RSR & RCC_RSR_SFTRSTF) {
    RCC->RSR = RCC_RSR_RMVF;
    while (RCC->RSR & RCC_RSR_SFTRSTF);
  }
  bootopt = 0;

  /* copy bootloader to ram - can't use memcpy() yet */
  extern uint32_t _text_load, _text_start, _text_end;
  extern uint32_t _rodata_load, _rodata_start, _rodata_end;
  uint32_t *s, *d;

  if (&_text_start != &_text_end) {
    s = &_text_load;
    d = &_text_start;
    do { *d++ = *s++; } while(d < &_text_end);
  }

  if (&_rodata_start != &_rodata_end) {
    s = &_rodata_load;
    d = &_rodata_start;
    do { *d++ = *s++; } while(d < &_rodata_end);
  }

  /* go */
  __DSB();
  goto *&_crt0_entry;
}


/* --- main ---------------------------------------------------------------- */

int
main(void)
{
  uintptr_t addr;

  halInit();
  osalSysEnable();

  update_config_pending = 0;
  tk3set_init();
  tk3fb_init();
  tk3ev_hwinit();
  tk3usb_init(NULL, NULL);

  do
    addr = bootloader(tk3usb_iniob(0), tk3usb_outiob(0));
  while(update_config_pending && addr == -1);

  tk3usb_fini();

  /* update option bytes */
  set_option_bytes();

  /* reboot into application */
  if (addr == -1) { /* timeout: boot to flash */
    extern uint32_t __app_base__[];
    addr = (uintptr_t)__app_base__;
  }
  bootopt = addr | (BOOTMODE_BOOTADDR << 32);

  __DSB();
  SCB_CleanInvalidateDCache();
  NVIC_SystemReset();
  return 0;
}


/* --- set_option_bytes ---------------------------------------------------- */

void
set_option_bytes(void)
{
  uint32_t optsr, optwpsn, lock;

  /* no bank swap */
  optsr = FLASH->OPTSR_CUR & ~FLASH_OPTSR_SWAP_BANK_OPT;

  /* lock bootloader flash */
  optwpsn = FLASH->WPSN_CUR1 & ~1;
  lock = tk3set_get(TK3SET_BLLOCK, 0);
  if (lock == 1234567890) optwpsn |= 1;

  /* update if needed */
  if (FLASH->OPTSR_CUR == optsr && FLASH->WPSN_CUR1 == optwpsn) return;

  if (FLASH->OPTCR & FLASH_OPTCR_OPTLOCK) {
    FLASH->OPTKEYR = 0x08192a3b;
    FLASH->OPTKEYR = 0x4c5d6e7f;
    while (FLASH->OPTCR & FLASH_OPTCR_OPTLOCK) /* wait */;
  }

  FLASH->OPTSR_PRG = optsr;
  FLASH->WPSN_PRG1 = optwpsn;
  FLASH->OPTCR |= FLASH_OPTCR_OPTSTART;

  while (FLASH->OPTSR_CUR & FLASH_OPTSR_OPT_BUSY) /* wait */;
  FLASH->OPTCR |= FLASH_OPTCR_OPTLOCK;
}


/* --- flash access -------------------------------------------------------- */

static const uint32_t flash_errs =
  FLASH_SR_DBECCERR | FLASH_SR_SNECCERR | FLASH_SR_RDPERR | FLASH_SR_OPERR |
  FLASH_SR_INCERR | FLASH_SR_STRBERR | FLASH_SR_PGSERR | FLASH_SR_WRPERR;

static int	write_bank(int bank, uintptr_t addr, const uint8_t *data,
                        size_t size);

void
unlock_flash()
{
  /* unlock bank1 */
  if (FLASH->CR1 & FLASH_CR_LOCK) {
    while (FLASH->SR1 & FLASH_SR_BSY) /* wait */;

    FLASH->KEYR1 = 0x45670123;
    FLASH->KEYR1 = 0xcdef89ab;
    while (FLASH->CR1 & FLASH_CR_LOCK) /* wait */;
  }

  /* unlock bank2 */
  if (FLASH->CR2 & FLASH_CR_LOCK) {
    while (FLASH->SR2 & FLASH_SR_BSY) /* wait */;

    FLASH->KEYR2 = 0x45670123;
    FLASH->KEYR2 = 0xcdef89ab;
    while (FLASH->CR2 & FLASH_CR_LOCK) /* wait */;
  }
}

int
erase_flash(size_t sector)
{
  volatile uint32_t *SR, *CR, *CCR;
  uint8_t s;

  /* select bank */
  SR = sector & 8 ? &FLASH->SR2 : &FLASH->SR1;
  CR = sector & 8 ? &FLASH->CR2 : &FLASH->CR1;
  CCR = sector & 8 ? &FLASH->CCR2 : &FLASH->CCR1;
  s = sector & 7;

  /* clear any former error */
  while (*SR & FLASH_SR_BSY) /* wait */;
  *CCR |= flash_errs;

  /* erase */
  *CR =
    ((s << FLASH_CR_SNB_Pos) & FLASH_CR_SNB_Msk) | FLASH_CR_SER |
    (3 << FLASH_CR_PSIZE_Pos) /* 64 bits parallelism */;
  *CR |= FLASH_CR_START;

  while (*SR & FLASH_SR_QW) /* wait */;
  *CR = 0;

  /* rewrite config */
  if (sector == 0 && !(*SR & flash_errs)) {
    return write_bank(
      1 /* bank */,
      (uintptr_t)__config_base__,
      (uint8_t *)board_serial, strlen(board_serial)+1);
  }

  return *SR & flash_errs;
}

int
write_flash(uintptr_t addr, const uint8_t *data, size_t size)
{
  static const struct areas { int bank; uintptr_t end; } areas[] = {
    { 1, (uintptr_t)__config_base__ },		/* bootloader */
    { 0, (uintptr_t)&__config_base__[1] },	/* config (skip) */
    { 1, 0x8100000 },	/* rest of bank 1 */
    { 2, 0x8200000 },	/* bank 2 */
    { 0, 0}
  };
  const struct areas *a;
  size_t n;
  int e;

  /* the only writable flash is in bank1 & 2 */
  if (addr < 0x8000000) return FLASH_SR_WRPERR;

  /* split writes over banks */
  for(a = areas; a->end; a++) {
    if (addr >= a->end) continue;

    n = a->end - addr;
    if (n > size) n = size;
    e = a->bank ? write_bank(a->bank, addr, data, n) : 0;

    size -= n;
    data += n;
    addr += n;
    if (e || !size) return e;
  }

  return FLASH_SR_WRPERR;
}

static int
write_bank(int bank, uintptr_t addr, const uint8_t *data, size_t size)
{
  volatile uint32_t *SR, *CR, *CCR;
  size_t n;

  /* select bank */
  switch(bank) {
    case 1: SR = &FLASH->SR1; CR = &FLASH->CR1; CCR = &FLASH->CCR1; break;
    case 2: SR = &FLASH->SR2; CR = &FLASH->CR2; CCR = &FLASH->CCR2; break;
    default: return FLASH_SR_WRPERR;
  }

  /* clear any former error */
  while (*SR & FLASH_SR_BSY) /* wait */;
  *CCR |= flash_errs;

  /* write */
  *CR = FLASH_CR_PG | (3 << FLASH_CR_PSIZE_Pos) /* 64 bits parallelism */;

  /* first bytes of 32 bytes aligned word filled with 0xff */
  n = addr & 0x1f;
  if (n)
    memset((uint8_t *)addr - n, 0xff, n);

  /* 32 bytes writes */
  for(n = 32 - n; size >= n; n = 32) {
    memcpy((uint8_t *)addr, data, n);
    while (*SR & FLASH_SR_QW) /* wait */;
    if (*SR & flash_errs) goto done;

    addr += n;
    data += n;
    size -= n;
  }

  /* remainder of the last 32 bytes filled with 0xff */
  if (size) {
    memcpy((uint8_t *)addr, data, size);
    memset((uint8_t *)addr + size, 0xff, n - size);
    while (*SR & FLASH_SR_QW) /* wait */;
  }

done:
  *CR = 0;
  return *SR & flash_errs;
}

void
lock_flash()
{
  while (FLASH->SR1 & FLASH_SR_BSY) /* wait */;
  FLASH->CR1 |= FLASH_CR_LOCK;

  while (FLASH->SR2 & FLASH_SR_BSY) /* wait */;
  FLASH->CR2 |= FLASH_CR_LOCK;
}


/* --- update_config ------------------------------------------------------- */

_Atomic int update_config_pending;

void
update_config(void *arg)
{
  extern uint8_t __boot_base__[], __boot_end__[];
  extern uint8_t __boot_shadowbase__[];

  int *status = arg;
  uint8_t *end;
  int e;

  /* end of programmed boot sector (skip last 0xff bytes) */
  for(end = __boot_end__; end[-1] == 0xffffffff; end--);

  /* copy boot sector to shadow ram region */
  memcpy(__boot_shadowbase__, __boot_base__, end - __boot_base__);

  /* rewrite sector #0 */
  unlock_flash();

  e = erase_flash(0);
  if (e) goto done;

  e = write_flash((uintptr_t)__boot_base__,
                  __boot_shadowbase__, end - __boot_base__);

done:
  lock_flash();
  tk3set_init(); /* re-read config */
  if (status) *status = e;
}
