/*
 * Copyright (c) 2019-2020,2024 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Thu Jun 20 2019
 */
#ifndef H_TK3_USB
#define H_TK3_USB

#include "cmsis_compiler.h"


/* --- USB descriptors ----------------------------------------------------- */

struct __attribute__((__packed__)) usb_device_descriptor {
  uint8_t	bLength;
  uint8_t	bDescriptorType;
  uint16_t	bcdUSB;
  uint8_t	bDeviceClass;
  uint8_t	bDeviceSubClass;
  uint8_t	bDeviceProtocol;
  uint8_t	bMaxPacketSize;
  uint16_t	idVendor;
  uint16_t	idProduct;
  uint16_t	bcdDevice;
  uint8_t	iManufacturer;
  uint8_t	iProduct;
  uint8_t	iSerialNumber;
  uint8_t	bNumConfigurations;

#define usb_device_descriptor(...)                                      \
  {                                                                     \
    .bLength = sizeof(struct usb_device_descriptor),                    \
    .bDescriptorType = 0x1,                                             \
    __VA_ARGS__                                                         \
  }
};

struct __attribute__((__packed__)) usb_configuration_descriptor {
  uint8_t	bLength;
  uint8_t	bDescriptorType;
  uint16_t	wTotalLength;
  uint8_t	bNumInterfaces;
  uint8_t	bConfigurationValue;
  uint8_t	iConfiguration;
  uint8_t	bmAttributes;
  uint8_t	bMaxPower;

#define usb_configuration_descriptor(...)                               \
  {                                                                     \
    .bLength = sizeof(struct usb_configuration_descriptor),             \
    .bDescriptorType = 0x2,                                             \
    __VA_ARGS__                                                         \
  }
};

struct __attribute__((__packed__)) usb_interface_association_descriptor {
  uint8_t	bLength;
  uint8_t	bDescriptorType;
  uint8_t	bFirstInterface;
  uint8_t	bInterfaceCount;
  uint8_t	bFunctionClass;
  uint8_t	bFunctionSubClass;
  uint8_t	bFunctionProcotol;
  uint8_t	iInterface;

#define usb_interface_association_descriptor(...)                       \
  {                                                                     \
    .bLength = sizeof(struct usb_interface_association_descriptor),     \
    .bDescriptorType = 0x0b,                                            \
    __VA_ARGS__                                                         \
  }
};

struct __attribute__((__packed__)) usb_interface_descriptor {
  uint8_t	bLength;
  uint8_t	bDescriptorType;
  uint8_t	bInterfaceNumber;
  uint8_t	bAlternateSetting;
  uint8_t	bNumEndpoints;
  uint8_t	bInterfaceClass;
  uint8_t	bInterfaceSubClass;
  uint8_t	bInterfaceProtocol;
  uint8_t	iInterface;

#define usb_interface_descriptor(...)                                   \
  {                                                                     \
    .bLength = sizeof(struct usb_interface_descriptor),                 \
    .bDescriptorType = 0x4,                                             \
    __VA_ARGS__                                                         \
  }
};

struct __attribute__((__packed__)) usb_endpoint_descriptor {
  uint8_t	bLength;
  uint8_t	bDescriptorType;
  uint8_t	bEndpointAddress;
  uint8_t	bmAttributes;
  uint16_t	wMaxPacketSize;
  uint8_t	bInterval;

#define usb_endpoint_descriptor(...)                                    \
  {                                                                     \
    .bLength = sizeof(struct usb_endpoint_descriptor),                  \
    .bDescriptorType = 0x5,                                             \
    __VA_ARGS__                                                         \
  }
};

struct __attribute__((__packed__)) usb_hid_report {
  uint8_t	bReportType;
  uint16_t	wReportLength;

#define usb_hid_report(...)                                             \
  {                                                                     \
    __VA_ARGS__                                                         \
  }
};

struct __attribute__((__packed__)) usb_hid_descriptor {
  uint8_t	bLength;
  uint8_t	bDescriptorType;
  uint16_t	bcdHID;
  uint8_t	bCountry;
  uint8_t	bNumDescriptors;

#define usb_hid_descriptor(reports, ...)                                \
  {                                                                     \
    .bLength = sizeof(struct usb_hid_descriptor)                        \
      + (reports) * sizeof(struct usb_hid_report),                      \
    .bDescriptorType = 0x21,                                            \
    __VA_ARGS__                                                         \
  }
};

struct __attribute__((__packed__)) usb_cdc_header_descriptor {
  uint8_t	bLength;
  uint8_t	bDescriptorType;
  uint8_t	bDescriptorSubtype;
  uint16_t	bcdCDC;

#define usb_cdc_header_descriptor(...)                                  \
  {                                                                     \
    .bLength = sizeof(struct usb_cdc_header_descriptor),                \
    .bDescriptorType = 0x24,                                            \
    .bDescriptorSubtype = 0,                                            \
    __VA_ARGS__                                                         \
  }
};

struct __attribute__((__packed__)) usb_cdc_call_management_descriptor {
  uint8_t	bLength;
  uint8_t	bDescriptorType;
  uint8_t	bDescriptorSubtype;
  uint8_t	bmCapabilities;
  uint8_t	bDataInterface;

#define usb_cdc_call_management_descriptor(...)                         \
  {                                                                     \
    .bLength = sizeof(struct usb_cdc_call_management_descriptor),       \
    .bDescriptorType = 0x24,                                            \
    .bDescriptorSubtype = 0x1,                                          \
    __VA_ARGS__                                                         \
  }
};

struct __attribute__((__packed__)) usb_cdc_acm_descriptor {
  uint8_t	bLength;
  uint8_t	bDescriptorType;
  uint8_t	bDescriptorSubtype;
  uint8_t	bmCapabilities;

#define usb_cdc_acm_descriptor(...)                                     \
  {                                                                     \
    .bLength = sizeof(struct usb_cdc_acm_descriptor),                   \
    .bDescriptorType = 0x24,                                            \
    .bDescriptorSubtype = 0x2,                                          \
    __VA_ARGS__                                                         \
  }
};

struct __attribute__((__packed__)) usb_cdc_union_descriptor {
  uint8_t	bLength;
  uint8_t	bDescriptorType;
  uint8_t	bDescriptorSubtype;
  uint8_t	bMasterInterface;
  uint8_t	bSlaveInterface0;

#define usb_cdc_union_descriptor(...)                                   \
  {                                                                     \
    .bLength = sizeof(struct usb_cdc_union_descriptor),                 \
    .bDescriptorType = 0x24,                                            \
    .bDescriptorSubtype = 0x6,                                          \
    __VA_ARGS__                                                         \
  }
};

struct __attribute__((__packed__)) usb_cdc_serial_state {
  uint8_t	bmRequestType;
  uint8_t	bNotification;
  uint16_t	wValue;
  uint16_t	wIndex;
  uint16_t	wLength;
  uint16_t	data;

#define usb_cdc_serial_state(iface, s)                                  \
  {                                                                     \
    .bmRequestType = 0xa1,                                              \
    .bNotification = 0x20,                                              \
    .wIndex = htousbs(iface),                                           \
    .wLength = htousbs(2),                                              \
    .data = htousbs(s),                                                 \
  }
};

#define usb_string(str)	(u ## str)
#define usb_string_descriptor(name, str)                                \
  usb_string_descriptor_sz(name, str, sizeof(str)-1)
#define usb_string_descriptor_sz(name, str, size)                       \
  struct __attribute__((__packed__)) {                                  \
    uint8_t	bLength;                                                \
    uint8_t	bDescriptorType;                                        \
    uint16_t	bstring[size];                                          \
  } name = {                                                            \
    .bLength = 2 + sizeof(usb_string(str)) - 2,                         \
    .bDescriptorType = 0x3,                                             \
    .bstring = usb_string(str)                                          \
  }

#ifdef __ARMEL__
# define htousbs(x) (x)
#else
# define htousbs(x) (((x) << 8) | ((x) >> 8))
#endif


/* --- USB CDC ------------------------------------------------------------- */

struct __attribute__((__packed__)) usb_cdc_configuration {
  struct usb_interface_association_descriptor iad;
  struct usb_interface_descriptor icontrol;
  struct {
    struct usb_cdc_header_descriptor header;
    struct usb_cdc_call_management_descriptor cm;
    struct usb_cdc_acm_descriptor acm;
    struct usb_cdc_union_descriptor u;
  } cdc;
  struct usb_endpoint_descriptor epctl;
  struct usb_interface_descriptor idata;
  struct usb_endpoint_descriptor epin;
  struct usb_endpoint_descriptor epout;

#define usb_cdc_configuration(iface, ep)                               \
  {                                                                    \
    .iad = usb_interface_association_descriptor(                       \
      .bFirstInterface =	(iface),                               \
      .bInterfaceCount =	2,                                     \
      .bFunctionClass =		0x2,		/* CDC */              \
      .bFunctionSubClass =	0,                                     \
      .bFunctionProcotol =	0),                                    \
                                                                       \
    .icontrol = usb_interface_descriptor(                              \
      .bInterfaceNumber =	(iface),                               \
      .bNumEndpoints =		1,                                     \
      .bInterfaceClass =	0x2,		/* CDC */              \
      .bInterfaceSubClass =	0x2,		/* ACM */              \
      .bInterfaceProtocol =	0),                                    \
                                                                       \
    .cdc = {                                                           \
      .header = usb_cdc_header_descriptor(                             \
        .bcdCDC =		htousbs(0x0120)),/* 1.20 */            \
      .cm = usb_cdc_call_management_descriptor(                        \
        .bmCapabilities =	0x0,                                   \
        .bDataInterface =	(iface) + 1),                          \
      .acm = usb_cdc_acm_descriptor(                                   \
        .bmCapabilities =	2),                                    \
      .u = usb_cdc_union_descriptor(                                   \
        .bMasterInterface =	(iface),                               \
        .bSlaveInterface0 =	(iface) + 1)                           \
    },                                                                 \
                                                                       \
    .epctl = usb_endpoint_descriptor(                                  \
      .bEndpointAddress =	((ep) + 1)|0x80,/* IN */               \
      .bmAttributes =		0x3,		/* Interrupt */        \
      .wMaxPacketSize =		htousbs(16),                           \
      .bInterval =		0xff),		/* 255ms */            \
                                                                       \
    .idata = usb_interface_descriptor(                                 \
      .bInterfaceNumber =	(iface) + 1,                           \
      .bNumEndpoints =		2,                                     \
      .bInterfaceClass =	0xa,		/* CDC */              \
      .bInterfaceSubClass =	0x0,		/* ACM */              \
      .bInterfaceProtocol =	0),                                    \
                                                                       \
    .epin = usb_endpoint_descriptor(                                   \
      .bEndpointAddress =	(ep),		/* OUT */              \
      .bmAttributes =		0x2,		/* Bulk */             \
      .wMaxPacketSize =		htousbs(64)),                          \
                                                                       \
    .epout = usb_endpoint_descriptor(                                  \
      .bEndpointAddress =	(ep)|0x80,	/* IN */               \
      .bmAttributes =		0x2,		/* Bulk */             \
      .wMaxPacketSize =		htousbs(64))                           \
  }
};


/* --- USB HID ------------------------------------------------------------- */

struct __attribute__((__packed__)) usb_hid_configuration {
  struct usb_interface_descriptor interface;
  struct usb_hid_descriptor hid;
  struct usb_hid_report report;
  struct usb_endpoint_descriptor endpoint;

#define usb_hid_configuration(iface, ep, reportsz)                     \
  {                                                                    \
    .interface = usb_interface_descriptor(                             \
      .bInterfaceNumber =	(iface),                               \
      .bNumEndpoints =		1,                                     \
      .bInterfaceClass =	0x3,		/* HID */              \
      .bInterfaceSubClass =	0,                                     \
      .bInterfaceProtocol =	0),                                    \
                                                                       \
    .hid = usb_hid_descriptor(                                         \
      1,					/* reports */          \
      .bcdHID =			htousbs(0x111),	/* HID 1.11 */         \
      .bCountry =		0,		/* not localized */    \
      .bNumDescriptors =	1),                                    \
    .report = usb_hid_report(                                          \
      .bReportType =		0x22,		/* Report */           \
      .wReportLength =	(reportsz)                                     \
      ),                                                               \
                                                                       \
    .endpoint = usb_endpoint_descriptor(                               \
      .bEndpointAddress =	(ep) | 0x80,	/* IN */               \
      .bmAttributes =		0x3,		/* interrupt */        \
      .wMaxPacketSize =		htousbs(32),                           \
      .bInterval =		0x40)		/* 40ms */             \
    }
};


#endif /* H_TK3_USB */
