/*
 * Copyright (c) 2019-2021,2023-2025 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Thu Jun 20 2019
 */
#include "autoconf.h"

#include "hal.h"
#include "hal_usb_cdc.h"

#include "tk3-paparazzi.h"
#include "usb.h"

#ifndef TK3_USB_CDC
# define TK3_USB_CDC	0 /* no CDC by default */
#endif

/* strings */
const usb_string_descriptor(tk3_usb_string0, "\u0409");	/* US English */
const usb_string_descriptor(tk3_usb_string1, "LAAS/CNRS");
const usb_string_descriptor(tk3_usb_string2, "tk3-paparazzi");
usb_string_descriptor_sz(tk3_usb_string3, "", 64);

static bool fbactivity;

#if TK3_USB_CDC > 0
static tk3event(tk3usb_cdcon, NULL, NULL);
static tk3event(tk3usb_cdcoff, NULL, NULL);
#endif

static void			tk3usb_init_serial(void);
static int			tk3usb_active(void);
static void			tk3usb_event(USBDriver *usbp, usbevent_t event);
static const USBDescriptor *	tk3usb_descriptor(USBDriver *usbp,
                                        uint8_t dtype, uint8_t dindex,
                                        uint16_t lang);
static bool			tk3usb_request(USBDriver *usbp);
static void			tk3usb_sof(USBDriver *usbp);


/* --- tk3usb_init --------------------------------------------------------- */

int
tk3usb_init(void (*oncb)(void *), void (*offcb)(void *))
{
  static const USBConfig tk3usb_config = {
    .event_cb =			tk3usb_event,
    .get_descriptor_cb =	tk3usb_descriptor,
    .requests_hook_cb =		tk3usb_request,
    .sof_cb =			tk3usb_sof
  };

  /* configure D+/D- lines */
  palSetPadMode(GPIOA, 11, PAL_MODE_ALTERNATE(10));
  palSetPadMode(GPIOA, 12, PAL_MODE_ALTERNATE(10));
  usbDisconnectBus(&TK3_USBD);

  /* init serial */
  tk3usb_init_serial();

#if TK3_USB_CDC > 0
  /* init cdc callback */
  tk3ev_set(&tk3usb_cdcon, oncb, tk3usb_init);
  tk3ev_set(&tk3usb_cdcoff, offcb, tk3usb_fini);
#endif

  /* latency for the host to detect disconnection */
  osalThreadSleepMilliseconds(5);

  usbStart(&TK3_USBD, &tk3usb_config);
  usbConnectBus(&TK3_USBD);
  return 0;
}

static void
tk3usb_init_serial()
{
  int i;

  for(i = 0; board_serial[i]; i++)
    tk3_usb_string3.bstring[i] = board_serial[i];
  tk3_usb_string3.bLength = 2 + 2 * i;
}


/* --- tk3usb_fini --------------------------------------------------------- */

int
tk3usb_fini()
{
  while (tk3usb_active())
    __WFI();

  usbDisconnectBus(&TK3_USBD);
  /* latency for the host to detect disconnection */
  osalThreadSleepMilliseconds(5);
  usbStop(&TK3_USBD);
  return 0;
}


/* === USB CDC endpoints =================================================== */

#if TK3_USB_CDC > 0

static void	tk3usb_cdc_transmitted(USBDriver *usbp, usbep_t ep);
static void	tk3usb_cdc_received(USBDriver *usbp, usbep_t ep);
static void	tk3usb_cdc_sof(USBDriver *usbp, int i);

static struct {
  uint8_t io[2][256];
  struct usb_cdc_serial_state linestate;
} tk3_nocache(tk3_cdcmem[TK3_USB_CDC]);

static struct {
  const USBEndpointConfig config[2];
  cdc_linecoding_t linecoding;
  uint32_t linestate;
  USBOutEndpointState out;
  USBInEndpointState in;
  USBInEndpointState ctrl;

  struct tk3iob iniob, outiob;

#define tk3_cdc(i)                                                      \
  {                                                                     \
    .config[0] = {                                                      \
      .ep_mode = USB_EP_MODE_TYPE_BULK,                                 \
      .in_cb = tk3usb_cdc_transmitted, .out_cb = tk3usb_cdc_received,   \
      .in_maxsize = 64, .out_maxsize = 64,                              \
      .in_state = &tk3_cdc[i].in, .out_state = &tk3_cdc[i].out          \
    },                                                                  \
    .config[1] = {                                                      \
      .ep_mode = USB_EP_MODE_TYPE_INTR,                                 \
      .in_maxsize = 16, .in_state = &tk3_cdc[i].ctrl                    \
    },                                                                  \
    .linecoding = {                                                     \
      {0x00, 0x96, 0x00, 0x00},	/* 38400 */                             \
      LC_STOP_1, LC_PARITY_NONE, 8                                      \
    },                                                                  \
    .linestate = 0,                                                     \
    .iniob = tk3iob_init("usb input " #i, tk3_cdcmem[i].io[0], 64),     \
    .outiob = tk3iob_init("usb output " #i, tk3_cdcmem[i].io[1], 64)    \
  }

} tk3_cdc[] = {
  tk3_cdc(0),
#if TK3_USB_CDC > 1
  tk3_cdc(1)
#endif
};


/* --- tk3usb_iob ---------------------------------------------------------- */

struct tk3iob *
tk3usb_iniob(int n)
{
  return &tk3_cdc[n].iniob;
}

struct tk3iob *
tk3usb_outiob(int n)
{
  return &tk3_cdc[n].outiob;
}


/* --- tk3usb_active ------------------------------------------------------- */

static int
tk3usb_active()
{
  struct tk3iob *outb;
  int i;

  for (i = 0; i < TK3_USB_CDC; i++) {
    with_syslock()
      if (usbGetTransmitStatusI(&TK3_USBD, i*2+1)) return 1;
    if (tk3_cdc[i].linestate) {
      outb = tk3usb_outiob(i);
      if (outb->r != outb->w) return 1;
    }
  }

  return 0;
}


/* --- tk3usb_cdc_transmitted ---------------------------------------------- */

static void
tk3usb_cdc_transmitted(USBDriver *usbp, usbep_t ep)
{
  struct tk3iob_extent e;
  struct tk3iob *outb = tk3usb_outiob((ep-1)/2);
  size_t txlen = usbp->epc[ep]->in_state->txsize;

  if (txlen) {
    tk3iob_rcommit(outb, txlen);
    fbactivity = true;
  }
  if (!tk3_cdc[(ep-1)/2].linestate) return;

  e = tk3iob_rbegin(outb, 64);
  if (e.len > 0 ||
      (txlen && !(txlen % usbp->epc[ep]->in_maxsize))) {
    with_syslock() {
      if (TK3_USBD.state == USB_ACTIVE)
        usbStartTransmitI(usbp, ep, e.r, e.len);
    }
  }
}


/* --- tk3usb_cdc_received ------------------------------------------------- */

static void
tk3usb_cdc_received(USBDriver *usbp, usbep_t ep)
{
  struct tk3iob_extent e;
  struct tk3iob *inb = tk3usb_iniob((ep-1)/2);
  size_t size;

  size = 0;
  with_syslock() {
    if (TK3_USBD.state == USB_ACTIVE)
      size = usbGetReceiveTransactionSizeX(usbp, ep);
  }
  if (size > 0) {
    tk3iob_wcommit(inb, size);
    fbactivity = true;
  }

  /* restart a read if there is enough space */
  e = tk3iob_wbegin(inb, 64);
  if (e.len == 64) { /* OUT transfers must be multiple of the max size */
    with_syslock() {
      if (TK3_USBD.state == USB_ACTIVE)
        usbStartReceiveI(usbp, ep, e.w, 64);
    }
  }
}


/* --- tk3usb_cdc_sof ------------------------------------------------------ */

static void
tk3usb_cdc_sof(USBDriver *usbp, int i)
{
  struct tk3iob_extent e;

  /* send interrupt endpoint data */
  if (!tk3_cdc[i].linestate != !tk3_cdcmem[i].linestate.data) {
    with_syslock() {
      if (TK3_USBD.state != USB_ACTIVE) break;
      if (usbGetTransmitStatusI(&TK3_USBD, i*2 + 2)) break;

      tk3_cdcmem[i].linestate.data = htousbs(tk3_cdc[i].linestate ? 3 : 0);
      usbStartTransmitI(
        &TK3_USBD, i*2 + 2,
        (uint8_t *)&tk3_cdcmem[i].linestate, sizeof(tk3_cdcmem[i].linestate));
    }
  }

  /* send bulk endpoint data if needed */
  if (tk3_cdc[i].linestate) {
    e = tk3iob_rbegin(tk3usb_outiob(i), 64);
    if (e.len > 0) {
      with_syslock() {
        if (TK3_USBD.state != USB_ACTIVE) break;
        if (usbGetTransmitStatusI(&TK3_USBD, i*2 + 1)) break;

        usbStartTransmitI(&TK3_USBD, i*2 + 1, e.r, e.len);
      }
    }
  }

  /* receive bulk endpoint data if needed */
  with_syslock() {
    if (TK3_USBD.state != USB_ACTIVE) break;
    if (usbGetReceiveStatusI(&TK3_USBD, i*2 + 1)) break;

    e = tk3iob_wbegin(tk3usb_iniob(i), 64);
    if (e.len == 64) /* OUT transfers must be multiple of the max size */
      usbStartReceiveI(&TK3_USBD, i*2 + 1, e.w, 64);
  }
}

#endif


/* === USB HID endpoints =================================================== */

#if TK3_USB_GAMEPAD > 0

static void	tk3usb_gamepad_transmitted(USBDriver *usbp, usbep_t ep);

static void
tk3usb_gamepad_transmitted(USBDriver *usbp, usbep_t ep)
{
  static union { int16_t d[9]; unsigned char c[18]; } report;

  const int16_t *data;
  uint16_t bp, bm;
  int i;

  data = tk3rc_channel_data();
  with_syslock() {
    for (i = 0; i < 8; i++)
      report.d[i] = htousbs(data[i]);

    /* buttons 0-7 switched on when > 75%, off when < ~62%,
     * buttons 8-15 switched on when < -75%, off when > ~62% */
    for (i = 0, bp = 1, bm = 1<<8; i < 8; i++, bp<<=1, bm<<=1)
      report.d[8] ^=
        (report.d[8] & bp ? data[i] < 5<<12 : data[i] > 6<<12) ? bp : 0 |
        (report.d[8] & bm ? data[i] > -5<<12 : data[i] < -6<<12) ? bm : 0;
  }

  usbStartTransmitI(usbp, ep, report.c, sizeof(report.c));
}

static struct {
  const USBEndpointConfig config;
  USBInEndpointState in;
} tk3_gamepad = {
  .config = {
    .ep_mode = USB_EP_MODE_TYPE_INTR,
    .in_cb = tk3usb_gamepad_transmitted,
    .in_maxsize = 32, .in_state = &tk3_gamepad.in
  }
};

/* Gamepad report */
static const uint8_t tk3_gamepad_report[] = {
  0x05, 0x01,			/* USAGE_PAGE (Generic Desktop) */
  0x09, 0x05,			/* USAGE (Gamepad) */
  0xa1, 0x01,			/* COLLECTION (Application) */

  0x05, 0x01,			/*   USAGE_PAGE (Generic Desktop) */
  0x09, 0x01,			/*   USAGE (Pointer) */
  0xa1, 0x00,			/*   COLLECTION (Physical) */
  0x05, 0x01,			/*     USAGE_PAGE (Generic Desktop) */
  0x09, 0x30,			/*     USAGE (X) */
  0x09, 0x31,			/*     USAGE (Y) */
  0x09, 0x33,			/*     USAGE (Rx) */
  0x09, 0x34,			/*     USAGE (Ry) */
  0x09, 0x35,			/*     USAGE (Rz) */
  0x09, 0x38,			/*     USAGE (Wheel) */
  0x16, 0x01, 0x80,		/*     LOGICAL_MINIMUM (-32767) */
  0x26, 0xff, 0x7f,		/*     LOGICAL_MAXIMUM (+32767) */
  0x75, 0x10,			/*     REPORT_SIZE (16) */
  0x95, 0x06,			/*     REPORT_COUNT (6) */
  0x81, 0x02,			/*     INPUT (Data,Var,Abs) */

  0x05, 0x02,			/*     USAGE_PAGE (Simulation Controls) */
  0x09, 0xba,			/*     USAGE (Rudder) */
  0x09, 0xbb,			/*     USAGE (Throttle) */
  0x16, 0x01, 0x80,		/*     LOGICAL_MINIMUM (-32767) */
  0x26, 0xff, 0x7f,		/*     LOGICAL_MAXIMUM (+32767) */
  0x75, 0x10,			/*     REPORT_SIZE (16) */
  0x95, 0x02,			/*     REPORT_COUNT (2) */
  0x81, 0x02,			/*     INPUT (Data,Var,Abs) */

  0x05, 0x09,			/*     USAGE_PAGE (Button) */
  0x19, 0x01,			/*     USAGE_MINIMUM (Button 1) */
  0x29, 0x10,			/*     USAGE_MAXIMUM (Button 16) */
  0x15, 0x00,			/*     LOGICAL_MINIMUM (0) */
  0x25, 0x01,			/*     LOGICAL_MAXIMUM (1) */
  0x75, 0x01,			/*     REPORT_SIZE (1) */
  0x95, 0x10,			/*     REPORT_COUNT (16) */
  0x81, 0x02,			/*     INPUT (Data,Var,Abs) */
  0xc0,				/*   END_COLLECTION */

  0xc0				/* END_COLLECTION */
};

#endif

/* === USB device ========================================================== */

const struct __attribute__((__packed__)) {
  struct usb_configuration_descriptor configuration;

#if TK3_USB_CDC > 0
  struct usb_cdc_configuration cdc[TK3_USB_CDC];
#endif
#if TK3_USB_GAMEPAD > 0
  struct usb_hid_configuration gamepad;
#endif

} tk3_usb_configuration = {
  .configuration = usb_configuration_descriptor(
    .wTotalLength =		htousbs(sizeof(tk3_usb_configuration)),
    .bNumInterfaces =		TK3_USB_CDC * 2 + TK3_USB_GAMEPAD,
    .bConfigurationValue =	1,
    .iConfiguration =		0,
    .bmAttributes =		0x80,		/* no self-powered, no wakeup */
    .bMaxPower =		50		/* 100mA */
    ),

#if TK3_USB_CDC > 0
  .cdc[0] = usb_cdc_configuration(0, 1),
#endif
#if TK3_USB_CDC > 1
  .cdc[1] = usb_cdc_configuration(2, 3),
#endif
#if TK3_USB_GAMEPAD > 0
  .gamepad = usb_hid_configuration(
    TK3_USB_CDC * 2, TK3_USB_CDC * 2 + 1, sizeof(tk3_gamepad_report)),
#endif
};

const struct usb_device_descriptor tk3_usb_dev = usb_device_descriptor(
  .bcdUSB =		htousbs(0x200),		/* 2.00 */

  .bDeviceClass =	0xef,			/* Misc device class */
  .bDeviceSubClass =	0x2,			/* Common class */
  .bDeviceProtocol =	0x1,			/* IAD */

  .bMaxPacketSize =	0x40,

  .idVendor =		htousbs(0x1209),	/* http://pid.codes/1209 */
  .idProduct =		htousbs(0x1),		/* Test */
  .bcdDevice =		htousbs(
    (TK3_VERSION_MAJOR << 8) |
    (TK3_VERSION_MINOR << 4) |
    (TK3_VERSION_PATCH + 0)),			/* version */

  .iManufacturer =	1,
  .iProduct =		2,
  .iSerialNumber =	3,

  .bNumConfigurations = 1
  );


/* --- tk3usb_event -------------------------------------------------------- */

static void
tk3usb_event(USBDriver *usbp, usbevent_t event)
{
  struct tk3iob_extent e;

  switch (event) {
    case USB_EVENT_ADDRESS:
      return;

    case USB_EVENT_CONFIGURED:
      with_syslock() {
        int i;
        for(i = 0; i < TK3_USB_CDC; i++) {
          tk3_cdcmem[i].linestate =
            (struct usb_cdc_serial_state)usb_cdc_serial_state(i*2, 0);

          usbInitEndpointI(usbp, i*2+1, &tk3_cdc[i].config[0]);
          usbInitEndpointI(usbp, i*2+2, &tk3_cdc[i].config[1]);

          e = tk3iob_wbegin(tk3usb_iniob(i), 64);
          usbStartReceiveI(usbp, i*2+1, e.w, e.len);
        }

#if TK3_USB_GAMEPAD > 0
        usbInitEndpointI(usbp, TK3_USB_CDC * 2 + 1, &tk3_gamepad.config);
        tk3usb_gamepad_transmitted(usbp, TK3_USB_CDC * 2 + 1);
#endif
      }
      return;

    case USB_EVENT_RESET:
      /* recompute serial */
      tk3usb_init_serial();
      return;

    case USB_EVENT_UNCONFIGURED:
      return;

    case USB_EVENT_SUSPEND:
      tk3fb_off(TK3FB_COMM);
      return;

    case USB_EVENT_WAKEUP:
      return;

    case USB_EVENT_STALLED:
      return;
  }
}


/* --- tk3usb_request ------------------------------------------------------ */

bool tk3usb_vendor_request(USBDriver *usbp) __attribute__((weak));

static bool
tk3usb_request(USBDriver *usbp)
{
  /* handle vendor requests, if any */
  if (tk3usb_vendor_request)
    if (tk3usb_vendor_request(usbp)) return true;

  if ((usbp->setup[0] & USB_RTYPE_TYPE_MASK) != USB_RTYPE_TYPE_CLASS)
    return false;

  switch (usbp->setup[1]) {
    case CDC_GET_LINE_CODING:
    case CDC_SET_LINE_CODING:
#if TK3_USB_CDC > 0
      if (usbp->setup[4] == 0) {
        usbSetupTransfer(usbp, (uint8_t *)&tk3_cdc[0].linecoding,
                         sizeof(cdc_linecoding_t), NULL);
        return true;
      }
#endif
#if TK3_USB_CDC > 1
      if (usbp->setup[4] == 2) {
        usbSetupTransfer(usbp, (uint8_t *)&tk3_cdc[1].linecoding,
                         sizeof(cdc_linecoding_t), NULL);
        return true;
      }
#endif
      break;

    case CDC_SET_CONTROL_LINE_STATE:
#if TK3_USB_CDC > 0
      if (usbp->setup[4] == 0) {
        uint32_t linestate = (usbp->setup[2] & 1);
        if (tk3_cdc[0].linestate ^ linestate) {
          tk3_cdc[0].linestate = linestate;
#if TK3_USB_CDC > 1
          if (!tk3_cdc[1].linestate) {
#endif
            if (tk3_cdc[0].linestate)
              tk3ev_schedule(&tk3usb_cdcon, TK3EV_IDLE);
            else
              tk3ev_schedule(&tk3usb_cdcoff, TK3EV_IDLE);
#if TK3_USB_CDC > 1
          }
#endif
        }
        usbSetupTransfer(usbp, NULL, 0, NULL);
        return true;
      }
#endif
#if TK3_USB_CDC > 1
      if (usbp->setup[4] == 2) {
        uint32_t linestate = (usbp->setup[2] & 1);
        if (tk3_cdc[1].linestate ^ linestate) {
          tk3_cdc[1].linestate = linestate;
          if (!tk3_cdc[0].linestate) {
            if (tk3_cdc[1].linestate)
              tk3ev_schedule(&tk3usb_cdcon, TK3EV_IDLE);
            else
              tk3ev_schedule(&tk3usb_cdcoff, TK3EV_IDLE);
          }
        }
        usbSetupTransfer(usbp, NULL, 0, NULL);
        return true;
      }
#endif
      break;
  }

  return false;
}


/* --- tk3usb_sof ---------------------------------------------------------- */

static void
tk3usb_sof(USBDriver *usbp)
{
#if TK3_USB_CDC > 0
  tk3usb_cdc_sof(usbp, 0);
#endif
#if TK3_USB_CDC > 1
  tk3usb_cdc_sof(usbp, 1);
#endif

  tk3fb_set(TK3FB_COMM, fbactivity);
  fbactivity = 0;
}


/* === USB descriptors ===================================================== */

/*
 * Handles the GET_DESCRIPTOR callback. All required descriptors must be
 * handled here.
 */
static const USBDescriptor *
tk3usb_descriptor(USBDriver *usbp,
                  uint8_t dtype, uint8_t dindex, uint16_t lang)
{
#define usb_descriptor(x) { .ud_size = sizeof(x), .ud_string = (void *)&(x) }
  (void)usbp;
  (void)lang;

  switch (dtype) {
    case 0x1: {
      static const USBDescriptor d = usb_descriptor(tk3_usb_dev);
      return &d;
    }
    case 0x2: {
      static const USBDescriptor d = usb_descriptor(tk3_usb_configuration);
      return &d;
    }

    case 0x3: {
      static const USBDescriptor d[] = {
        usb_descriptor(tk3_usb_string0),
        usb_descriptor(tk3_usb_string1),
        usb_descriptor(tk3_usb_string2)
      };
      if (dindex < sizeof(d)/sizeof(d[0])) return &d[dindex];
      else if (dindex == sizeof(d)/sizeof(d[0])) {
        /* string3 is dynamically set, so must set length from .bLength */
        static USBDescriptor s = usb_descriptor(tk3_usb_string3);

        s.ud_size = tk3_usb_string3.bLength;
        return &s;
      }
      break;
    }

#if TK3_USB_GAMEPAD > 0
    case 0x22:	{
      static const USBDescriptor d = usb_descriptor(tk3_gamepad_report);
      if (dindex < 1) return &d;
      break;
    }
#endif
  }

  return NULL;
}
