/*
 * Copyright (c) 2019-2025 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Thu Jun 20 2019
 */
#ifndef H_TK3_PAPARAZZI
#define H_TK3_PAPARAZZI

#include <stdatomic.h>
#include <stddef.h>
#include <stdint.h>

#include "hal.h"
#include "osal.h"


/* --- memory -------------------------------------------------------------- */

/* cache-disabled memory */
#define tk3_nocache(x)		tk3_mem(TK3_NOCACHE_MEM, x)

/* named memory placement */
#define tk3_mem(m, x)           x __attribute__((section(m)))

/* static memory buffer with extra padding at the beginning and named memory
 * placement. */
#define tk3_padmem(type, kind)	tk3_padmembuf(type, kind,)
#define tk3_padmembuf(type, mem, pad, ...)                                     \
  ({                                                                           \
    static struct {                                                            \
      uint8_t filler[pad+0]; typeof(type) data __attribute__((aligned(4)));    \
    } tk3_mem(mem, padmem);                                                    \
    &padmem.data;                                                              \
  })


/* --- locking ------------------------------------------------------------- */

static inline uint32_t
syslock(void)
{
  uint32_t pri = __get_BASEPRI();
  __set_BASEPRI(OSAL_BASEPRI(OSAL_IRQ_MAXIMUM_PRIORITY));
  return pri;
}

static inline uint32_t
evlock(void)
{
  uint32_t pri = __get_BASEPRI();
  __set_BASEPRI(OSAL_BASEPRI((1 << CORTEX_PRIORITY_BITS)-1));
  return pri;
}

static inline uint32_t
sysunlock(void)
{
  uint32_t pri = __get_BASEPRI();
  __set_BASEPRI(0);
  return pri;
}

static inline void
syslock_restore(uint32_t *pri)
{
  __set_BASEPRI(*pri);
}

#define with_syslock()                                                  \
  for(uint32_t basepri                                                  \
        __attribute__((__cleanup__(syslock_restore))) = syslock(),      \
        do_ = 1; do_; do_ = 0)

#define without_syslock()                                               \
  for(uint32_t basepri                                                  \
        __attribute__((__cleanup__(syslock_restore))) = sysunlock(),    \
        do_ = 1; do_; do_ = 0)

#define with_evlock()                                                   \
  for(uint32_t basepri                                                  \
        __attribute__((__cleanup__(syslock_restore))) = evlock(),       \
        do_ = 1; do_; do_ = 0)


/* --- coroutine ----------------------------------------------------------- */

/* poor man's coroutine using static state and Duff's device */

#define coroutine()	coroutinex(0, 1)
#define coroutinex(i, n)                                                       \
  static uint32_t statev_[n];                                                  \
  for(uint32_t do_ = 1, * const state_ = &statev_[i]; do_; do_ = *state_ = 0)  \
    switch(*state_) default:

#define yield(x) \
  do { *state_ = __LINE__; return x; case __LINE__:; } while (0)


/* --- io buffers ---------------------------------------------------------- */

/* tk3iob is a ring buffer with extra mirroring after the buffer to allow
 * getting contiguous read and write buffers of a given maximum size:
 *
 *  bot       sizeof(mem)      top  rwblk  end
 *   +--------------------------+-----------+
 *   |   r     w                |           |
 *   +---^-----^----------------+-----------+
 */
struct tk3iob {
  const uint8_t * volatile r;		/* read pointer */
  uint8_t * volatile w;			/* write pointer */
  void (*cb)(struct tk3iob *iob);	/* signal callback */

  uint8_t * const bot, * const top;	/* limits of the ring buffer */
  const uint8_t * const end;		/* extra space for contiguous r/w */

  const char * const id;		/* human-readable identifier */

/* static intializer - rwblk is the maximum contiguous block size */
#define tk3iob_init(ident, mem, rwblk)                                  \
  {                                                                     \
    .r = (mem),                                                         \
    .w = (mem),                                                         \
    .cb = NULL,                                                         \
    .bot = (mem),                                                       \
    .top = (mem) + sizeof(mem) - ((rwblk) < 1 ? 0 : (rwblk)-1),         \
    .end = (mem) + sizeof(mem),                                         \
    .id = ident                                                         \
  }
};

/* contiguous read or write extent in a buffer */
struct tk3iob_extent {
  union {
    const uint8_t *r;
    uint8_t *w;
  };
  size_t len;
};

size_t			tk3iob_putc(struct tk3iob *buffer, uint8_t c);
size_t			tk3iob_write(struct tk3iob *buffer, const uint8_t *data,
                                size_t len);
struct tk3iob_extent	tk3iob_wbegin(struct tk3iob *buffer, size_t len);
void			tk3iob_wcommit(struct tk3iob *buffer, size_t len);

size_t			tk3iob_getc(struct tk3iob *buffer, uint8_t *c);
size_t			tk3iob_read(struct tk3iob *buffer, uint8_t *data,
                                size_t len);
struct tk3iob_extent	tk3iob_rbegin(struct tk3iob *buffer, size_t len);
void			tk3iob_rcommit(struct tk3iob *buffer, size_t len);


/* --- messages ------------------------------------------------------------ */

enum __attribute__((__packed__)) tk3ch {
  TK3CH_USB0,
  TK3CH_USB1,
  TK3CH_MKBL,

  TK3CH_MAX
};

struct tk3msg {
  enum tk3ch channel;
  struct {
    unsigned start:1;
    unsigned escape:1;
    unsigned eagain:1;
    unsigned magic:4;
  } state;

  size_t len;
  uint8_t *bot, *end;
  const uint8_t *top;

  /* static intializer - rwblk is the maximum contiguous block size */
#define tk3msg(chan, mem)                                               \
  {                                                                     \
    .channel = (chan),                                                  \
    .state = {},                                                        \
    .len = 0,                                                           \
    .bot = (mem), .end = (mem),                                         \
    .top = (mem) + sizeof(mem)                                          \
  }
};

void			tk3msg_init(void (*cb)(struct tk3iob *iob));
void			tk3msg_log(enum tk3ch channel, const char *fmt, ...);
void			tk3msg_log_buffer(enum tk3ch channel,
                                const uint8_t *buffer, uint8_t len);
const struct tk3msg *	tk3msg_recv(void);
void			tk3msg_pushback(enum tk3ch channel, size_t to,
                                size_t from);


/* --- events -------------------------------------------------------------- */

enum tk3evpri {
  TK3EV_INACTIVE = 0,
  TK3EV_IDLE,
  TK3EV_NOW
};

struct tk3event {
  struct tk3event *next;	/* next element */
  struct tk3event **prev;	/* address of previous next element */

  enum tk3evpri pri;
  void (*cb)(void *arg);
  void *arg;
};

#define tk3event(ev, cb_, arg_)	struct tk3event ev = { .cb = cb_, .arg = arg_ }

int			tk3ev_hwinit(void);

int			tk3ev_active(struct tk3event *event);
void			tk3ev_set(struct tk3event *event,
                                void (*cb)(void *arg), void *arg);
void			tk3ev_schedule(struct tk3event *event,
                                enum tk3evpri pri);
void			tk3ev_cancel(struct tk3event *event);
void			tk3ev_signal(void);
int			tk3ev_nextevent(enum tk3evpri pri);
int			tk3ev_process(void);


/* --- timers -------------------------------------------------------------- */

#define TK3CLK_FREQ	1000000		/* 1MHz - 1µs resolution */

struct tk3timer {
  struct tk3timer *next;	/* next element */
  struct tk3timer **prev;	/* address of previous next element */

  enum tk3evpri pri;
  uint32_t deadline;
  uint32_t interval;

  struct tk3event event;
};

#define tk3timer(tm)	struct tk3timer tm = {}

int			tk3clk_init(void);
int			tk3clk_hwinit(void);

uint32_t		tk3clk_cpucounter(void);
uint32_t		tk3clk_us(void);

void			tk3clk_delay(uint32_t delay);
void			tk3clk_delayuntil(uint32_t period);
void			tk3clk_settimer(struct tk3timer *timer,
                                uint32_t interval, uint32_t delay,
                                enum tk3evpri pri, void (*cb)(void *arg),
                                void *arg);
void			tk3clk_deltimer(struct tk3timer *timer);

void			tk3clk_signal(void);
uint32_t		tk3clk_alarm(uint32_t now);


/* --- visual feedback ----------------------------------------------------- */

enum tk3fb_sig {
  TK3FB_BL,
  TK3FB_IDLE,
  TK3FB_COMM,
  TK3FB_SENS,
  TK3FB_SENS_BUSY,
  TK3FB_SENS_ERR,
  TK3FB_ESC,
  TK3FB_ESC_ERR,
  TK3FB_ESC_0,
  TK3FB_ESC_1,
  TK3FB_ESC_2,
  TK3FB_ESC_3,
  TK3FB_ESC_4,
  TK3FB_ESC_5,
  TK3FB_ESC_6,
  TK3FB_RCKILL,
  TK3FB_ALL /* must be last */
};

struct tk3fb_led {
  const ioline_t line;
  _Atomic uint32_t bits[TK3FB_ALL/32+1];
};

struct tk3fb_pattern {
  const uint32_t n;
  const uint32_t *led;
  struct tk3timer off;
};

#define leds_list(...)                                                  \
  {                                                                     \
    .n = sizeof((uint8_t []){__VA_ARGS__}),                             \
    .led = (const uint32_t []){__VA_ARGS__}                             \
  }

extern struct tk3fb_led tk3fb_led[];
extern struct tk3fb_pattern tk3fb_pattern[];

int			tk3fb_init(void);

void			tk3fb_on(enum tk3fb_sig sig);
void			tk3fb_off(enum tk3fb_sig sig);
void			tk3fb_set(enum tk3fb_sig sig, bool on);
void			tk3fb_toggle(enum tk3fb_sig sig);
void			tk3fb_flash(enum tk3fb_sig sig, uint32_t delay);


/* --- power --------------------------------------------------------------- */

int			tk3pwr_hwinit(void);

uint16_t		tk3pwr_vbat(void);
void			tk3pwr_on(void *arg);
void			tk3pwr_off(void *arg);
void			tk3pwr_batdata(void *arg);


/* --- usb ----------------------------------------------------------------- */

int			tk3usb_init(void (*oncb)(void *),
                                void (*offcb)(void *));
int			tk3usb_fini(void);
struct tk3iob *		tk3usb_iniob(int n);
struct tk3iob *		tk3usb_outiob(int n);


/* --- RC ------------------------------------------------------------------ */

int			tk3rc_init(void);
int			tk3rc_hwinit(void);
const int16_t *		tk3rc_channel_data(void);

extern int		tk3rc_killswitch;


/* --- servos -------------------------------------------------------------- */

enum tk3srvmode {
  TK3SRV_OFF =		0,
  TK3SRV_PWM =		1,
  TK3SRV_DSHOT =	2,
  TK3SRV_BIDSHOT =	3
};

enum tk3srv3d {
  TK3SRV_3D =		1
};

int			tk3srv_init(void);
int			tk3srv_fini(void);
int			tk3srv_hwinit(enum tk3srvmode mode, uint32_t freq);
int			tk3srv_hwfini(void);

int			tk3srv_start(int id);
int			tk3srv_stop(int id);
void			tk3srv_setpid(int id, float Kp, float Ki, float Kd,
                                float dfc);
void			tk3srv_beep(void *data);
void			tk3srv_throttle(const int16_t throt[8]);
void			tk3srv_velocity(const int16_t hperiod[8]);

void			tk3srv_motdata(void *arg);

uint16_t		tk3srv_dshotmsg(uint16_t throt, bool tlm);
uint16_t		tk3srv_bidshotmsg(uint16_t throt, bool tlm);
void			tk3srv_reset(void);
int			tk3srv_send(uint16_t throt[8]);
void			tk3srv_read(uint32_t tlm[8]);


/* --- sensors ------------------------------------------------------------- */

void			tk3sens_init(void);
void			tk3sens_fini(void);
void			tk3sens_hwinit(void);
void			tk3sens_hwfini(void);

void			tk3sens_sendimu(void *arg);
void			tk3sens_sendmag(void *arg);

int			tk3sens_write(uint32_t dev, uint8_t n, ...);
int			tk3sens_read(uint32_t dev, uint8_t n, uint8_t addr,
                                volatile void *value);
void			tk3sens_aioread(uint32_t dev, uint8_t n, uint8_t addr,
                                volatile void *value, void (*cb)(void *));

#ifdef TK3_MPU9250
int			tk3sens_mpu9250_probe(void);
int			tk3sens_mpu9250_init(void);
void			tk3sens_mpu9250_aioread(void (*cb)(void *));
int			tk3sens_mpu9250_data(int32_t a[3], int32_t w[3],
                                int16_t *t);
#endif

#ifdef TK3_ICM42688
int			tk3sens_icm42688_probe(void);
int			tk3sens_icm42688_init(void);
void			tk3sens_icm42688_aioread(void (*cb)(void *));
int			tk3sens_icm42688_data(int32_t a[3], int32_t w[3],
                                int16_t *t);
#endif

#ifdef TK3_AK8963
int			tk3sens_ak8963_probe(void);
int			tk3sens_ak8963_init(void);
void			tk3sens_ak8963_aioread(void (*cb)(void *));
int			tk3sens_ak8963_data(int32_t m[3]);
#endif

#ifdef TK3_RM3100
int			tk3sens_rm3100_probe(void);
int			tk3sens_rm3100_init(void);
void			tk3sens_rm3100_aioread(void (*cb)(void *));
int			tk3sens_rm3100_data(int32_t m[3]);
#endif

#ifdef TK3_LIS3MDL
int			tk3sens_lis3mdl_probe(void);
int			tk3sens_lis3mdl_init(void);
void			tk3sens_lis3mdl_aioread(void (*cb)(void *));
int			tk3sens_lis3mdl_data(int32_t m[3]);
#endif


/* --- mkbl motors --------------------------------------------------------- */

void			tk3mkbl_settings(void);
void			tk3mkbl_init(void);
void			tk3mkbl_fini(void);
struct tk3iob *		tk3mkbl_iniob(void);
struct tk3iob *		tk3mkbl_outiob(void);
void			tk3mkbl_stop(void);
void			tk3mkbl_spin(const uint8_t *buffer, uint8_t len);
void			tk3mkbl_log(uint32_t period);
void			tk3mkbl_send(const uint8_t *buffer, uint8_t len);

void			tk3mkbl_hwinit(void);
void			tk3mkbl_hwfini(void);
void			tk3mkbl_restart(void *arg);
void			tk3mkbl_complete(void);
void			tk3mkbl_hwsend(const struct tk3iob_extent *e);
void			tk3mkbl_hwrecv(const uint8_t sladdr);


/* --- can ----------------------------------------------------------------- */

void			tk3can_init(void);
void			tk3can_fini(void);
size_t			tk3can_send(const uint8_t *buf, size_t len);


/* --- aux / uart ---------------------------------------------------------- */

enum tk3aux {
  TK3AUX_8
};

void			tk3aux_init(void);
void			tk3aux_fini(void);

void			tk3aux_start(enum tk3aux aux, uint32_t baud);
void			tk3aux_stop(enum tk3aux aux);
size_t			tk3aux_send(enum tk3aux aux, const uint8_t *data,
                                size_t len);


/* --- settings ------------------------------------------------------------ */

#define TK3SET_MAGIC		"tk3"
#define TK3SET_BLLOCK		"boot-lock"
#define TK3SET_BLTOUT		"boot-timeout"
#define TK3SET_MOTORS		"mkbl-motors"
#define TK3SET_POLES		"poles"
#define TK3SET_SRVFREQ		"servo-freq"
#define TK3SET_SRVMODE		"servo-mode"
#define TK3SET_SRVDIR		"servo-dir"
#define TK3SET_SRV3D		"servo-3d"
#define TK3SET_SRVIDMAP		"servo-idmap"
#define TK3SET_SRVIDLE		"idle-throttle"
#define TK3SET_SRVWDOG		"servo-wdog"
#define TK3SET_SRVTOUT		"servo-timeout"
#define TK3SET_SRVARMDLY	"arming-delay"
#define TK3SET_RCKILL		"rc-kill-chan"
#define TK3SET_RCKILL_MODE	"rc-kill-mode"
#define TK3SET_ACCRANGE		"acc-range"
#define TK3SET_GYRRANGE		"gyr-range"

#define tk3_settings_var(n)                                             \
  tk3_setting_ ## n __attribute__((used, aligned(1), section(".settings." #n)))

#define tk3setting(key, value, descr, order)                            \
  static const struct __attribute__((packed)) {                         \
    char k[sizeof(key)];                                                \
    uint8_t v[4];                                                       \
    char d[sizeof(descr)];                                              \
  } tk3_settings_var(order) = {                                         \
    key,                                                                \
    { (value) >> 24,                                                    \
      ((value) >> 16) & 0xff,                                           \
      ((value) >> 8) & 0xff,                                            \
      (value) & 0xff },                                                 \
    descr }

extern char board_serial[64];

void		tk3set_init(void);
int32_t		tk3set_get(const char *key, int32_t defval);

#endif /* H_TK3_PAPARAZZI */
