/*
 * Copyright (c) 2023-2025 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Wed Apr 12 2023
 */
#include "autoconf.h"

#include "tk3-paparazzi.h"

/* settings */
tk3setting(TK3SET_SRVMODE, 0, "0=off, 1=PWM, 2=DSHOT, 3=BI-DSHOT", 20);
tk3setting(TK3SET_SRVFREQ, 150, "PWM=50-490Hz, DSHOT=100-900kHz", 21);
tk3setting(TK3SET_SRV3D, 0, "0=off, 1=reversible", 22);
tk3setting(TK3SET_SRVDIR, 12121212, "1=normal, 2=reversed (8 digits)", 23);
tk3setting(TK3SET_SRVIDMAP, 12345678, "ESC ids for " TK3SET_SRVIDMAP_DESCR, 24);
tk3setting(TK3SET_SRVIDLE, 6, "idle throttle (0%-20%)", 25);
tk3setting(TK3SET_SRVWDOG, 500, "propeller watchdog (1-5000ms)", 26);
tk3setting(TK3SET_SRVTOUT, 10, "propeller timeout (1-300s)", 27);
tk3setting(TK3SET_SRVARMDLY, 3, "arming delay (1-10s)", 28);

/* DSHOT commands */
#define DSHOT_CMD_BEEP1			1
#define DSHOT_CMD_BEEP2			2
#define DSHOT_CMD_BEEP3			3
#define DSHOT_CMD_BEEP4			4
#define DSHOT_CMD_BEEP5			5
#define DSHOT_CMD_SPIN_DIRECTION_1	7
#define DSHOT_CMD_SPIN_DIRECTION_2	8
#define DSHOT_CMD_3D_MODE_OFF		9
#define DSHOT_CMD_3D_MODE_ON		10
#define DSHOT_CMD_SAVE_SETTINGS		12
#define DSHOT_EXTENDED_TELEMETRY_ENABLE	13

/* telemetry messages */
struct tlmmsg {
  enum {
    TK3TLM_NONE, TK3TLM_RPM, TK3TLM_TEMP, TK3TLM_VOLT, TK3TLM_AMP, TK3TLM_SAT
  } kind;
  uint32_t value;
};

/* servo state */
static struct srvcfg {
  int hwinit;			/* hardware initialized */

  enum tk3srvmode mode;		/* control mode */
  uint32_t freq;		/* control freq (Hz PWM or kHz DSHOT) */
  uint32_t pidloop;		/* control period (µs) */
  int rev3d;			/* reversible thrust */
  int zero, idle;		/* zero & idle thrusts */
  uint32_t pmul;		/* eRPM period multiplier */
  uint32_t armdly;		/* arming delay */

  struct {
    uint32_t dur;
    uint32_t deadline;
  } wdog, tout;			/* watchdog and inactivity */

  struct tk3timer ctrl;		/* control timer */

  struct srvmotor {
    struct tk3timer cmd;	/* init/startup timer */

    uint8_t id;			/* esc id - 0 = disabled */
    unsigned reversed:1;	/* reverse spinnning direction */
    unsigned armed:1;
    unsigned emerg:1;
    unsigned starting:1;
    unsigned spinning:1;
    unsigned servo:1;		/* velocity PID */
    struct {
      uint32_t valid;		/* count valid frames */
      uint64_t bits;		/* 64 frames history */
    } feedback;			/* eRPM valid frames feedback */

    union {
      int16_t throt;		/* current throttle */
      uint16_t raw;		/* raw DSHOT command */
    };
    float target;		/* desired eHz period (Hz) */
    uint32_t period;		/* current eRPM period (µs) */
    uint16_t current;		/* current draw (mA) */

    float pid;			/* PID output */
    float m0, m1;		/* PID measurements */
    float err0;			/* PID error history */
    float a0, a1, a0d, a1d;	/* PID parameters */
    float d0, fd0, a0fd, a1fd;	/* PID derivative filter */
  } motor[8];

  void (*arm)(void *);		/* mode-dependent routines */
  void (*tlm)(void);
  uint16_t (*map)(int16_t);
} srv;

static uint16_t	tk3srv_pwmmapthrottle(int16_t throttle);
static uint16_t	tk3srv_pwm3dmapthrottle(int16_t throttle);
static uint16_t	tk3srv_dshotmapthrottle(int16_t throttle);
static uint16_t	tk3srv_dshot3dmapthrottle(int16_t throttle);

static void	tk3srv_armpwm(void *data);
static void	tk3srv_armdshot(void *data);
static void	tk3srv_armed(void *data);
static void	tk3srv_started(void *data);

static void	tk3srv_tlm(void);
static void	tk3srv_servo(struct srvmotor *motor);

static void	tk3srv_ctrlloop(void *data);


/* --- tk3srv_init --------------------------------------------------------- */

int
tk3srv_init()
{
  uint32_t map, dir;
  uint16_t idle;
  int i;

  /* reset state */
  srv = (struct srvcfg){0};
  srv.rev3d = tk3set_get(TK3SET_SRV3D, 0);
  srv.pmul = tk3set_get(TK3SET_POLES, 14) / 2;
  idle = tk3set_get(TK3SET_SRVIDLE, 5);
  if (idle > 20) {
    tk3msg_log(TK3CH_USB0, "Einvalid " TK3SET_SRVIDLE " setting");
    goto err;
  }
  srv.idle = (1<<15) * idle / 100;

  srv.mode = tk3set_get(TK3SET_SRVMODE, 0);
  srv.freq = tk3set_get(TK3SET_SRVFREQ, 600);

  srv.armdly = tk3set_get(TK3SET_SRVARMDLY, 3);
  if (srv.armdly < 1 || srv.armdly > 10) {
    tk3msg_log(TK3CH_USB0, "Einvalid " TK3SET_SRVARMDLY " setting");
    goto err;
  }
  srv.armdly *= 1000000;

  srv.wdog.dur = tk3set_get(TK3SET_SRVWDOG, 500);
  if (srv.wdog.dur < 1 || srv.wdog.dur > 5000) {
    tk3msg_log(TK3CH_USB0, "Einvalid " TK3SET_SRVWDOG " setting");
    goto err;
  }
  srv.wdog.dur *= 1000;

  srv.tout.dur = tk3set_get(TK3SET_SRVTOUT, 10);
  if (srv.tout.dur < 1 || srv.tout.dur > 300) {
    tk3msg_log(TK3CH_USB0, "Einvalid " TK3SET_SRVTOUT " setting");
    goto err;
  }
  srv.tout.dur *= 1000000;

  switch(srv.mode) {
    case TK3SRV_OFF: return 0; /* no servo, done */

    case TK3SRV_PWM:
      /* PWM, frequency limited to below 500Hz */
      if (srv.freq < 50 || srv.freq > 490) {
        tk3msg_log(TK3CH_USB0, "Einvalid " TK3SET_SRVFREQ " setting");
        goto err;
      }
      srv.pidloop = 1000000/srv.freq;
      srv.arm = tk3srv_armpwm;
      srv.map =
        srv.rev3d ? tk3srv_pwm3dmapthrottle : tk3srv_pwmmapthrottle;
      srv.zero = srv.map(0);
      break;

    case TK3SRV_DSHOT: case TK3SRV_BIDSHOT:
      /* DSHOT modes, allow any freq within range */
      if (srv.freq < 100 || srv.freq > 900) {
        tk3msg_log(TK3CH_USB0, "Einvalid servo-freq settings");
        goto err;
      }
      srv.pidloop = 1000000/(srv.freq * 1000 / 100);
      srv.arm = tk3srv_armdshot;
      srv.map =
        srv.rev3d ? tk3srv_dshot3dmapthrottle : tk3srv_dshotmapthrottle;
      srv.zero = 0;
      if (srv.mode == TK3SRV_BIDSHOT) srv.tlm = tk3srv_tlm;
      break;

    default:
      tk3msg_log(TK3CH_USB0, "Einvalid servo-mode settings");
      goto err;
  }

  /* ESC ids and direction */
  dir = tk3set_get(TK3SET_SRVDIR, 12121212);
  map = tk3set_get(TK3SET_SRVIDMAP, 12345678);
  for(i = 7; i >= 0; i--, map /= 10, dir /= 10) {
    srv.motor[i].id = map % 10;
    srv.motor[i].reversed = !(dir % 2);
    if (srv.motor[i].reversed && srv.mode == TK3SRV_PWM) {
      tk3msg_log(TK3CH_USB0, "EESC cannot be reversed  in pwm mode");
      goto err;
    }
  }
  if (map || dir) {
    if (map) tk3msg_log(TK3CH_USB0, "Etoo many ESCs in servo-idmap setting");
    if (dir) tk3msg_log(TK3CH_USB0, "Etoo many ESCs in servo-dir setting");
    goto err;
  }

  /* default PID */
  for(i = 0; i < 8; i++) {
    if (!srv.motor[i].id) continue;
    tk3srv_setpid(srv.motor[i].id, 40., 3000., 0., 100.);
  }

  /* hardware init */
  if (tk3srv_hwinit(srv.mode, srv.freq)) goto err;
  srv.hwinit = 1;

  tk3clk_settimer(
    &srv.ctrl, srv.pidloop, 0, TK3EV_NOW, tk3srv_ctrlloop, NULL);

  /* arm motors */
  tk3msg_log(TK3CH_USB0, "Narming ESCs");
  for(i = 0; i < 8; i++) {
    if (!srv.motor[i].id) continue;
    srv.arm(&srv.motor[i]);
  }

  return 0;
err:
  tk3fb_on(TK3FB_ESC_ERR);
  return 1;
}


/* --- tk3srv_fini --------------------------------------------------------- */

int
tk3srv_fini()
{
  int i;

  with_evlock() {
    for(i = 0; i < 8; i++) tk3clk_deltimer(&srv.motor[i].cmd);
    tk3clk_deltimer(&srv.ctrl);

    if (srv.hwinit) tk3srv_hwfini();
    srv.hwinit = 0;
  }

  return 0;
}


/* --- tk3srv_dshotencode -------------------------------------------------- */

static uint16_t
tk3srv_dshotencode(uint16_t throt, bool tlm, uint8_t rev)
{
  uint16_t msg, crc;

  /* pack throt and telemetry bit */
  msg = ((throt & 0x7ff) << 1) | !!tlm;

  /* telemetry bit needs to be set for special commands */
  msg |= throt > 0 && throt < 48;

  /* xor nibbles */
  crc = msg ^ (msg >> 4) ^ (msg >> 8);

  /* pack frame, reverting CRC for bi-dshot */
  return (msg << 4) | ((crc ^ rev) & 0xf);
}

uint16_t
tk3srv_dshotmsg(uint16_t throt, bool tlm)
{
  return tk3srv_dshotencode(throt, tlm, 0 /* normal checksum */);
}

uint16_t
tk3srv_bidshotmsg(uint16_t throt, bool tlm)
{
  return tk3srv_dshotencode(throt, tlm, 0xf /* inverted cksum */);
}


/* --- tk3srv_dshotdecode -------------------------------------------------- */

static struct tlmmsg
tk3srv_dshotdecode(uint32_t tlm)
{
  uint16_t crc;

  /* get GCR */
  tlm = tlm ^ (tlm >> 1);

  /* decode GCR */
  static const uint8_t gcr[32] = {
    [0 ... 0x1f] = -1, /* invalid by default */

    [0x19] = 0x0, [0x1b] = 0x1, [0x12] = 0x2, [0x13] = 0x3,
    [0x1d] = 0x4, [0x15] = 0x5, [0x16] = 0x6, [0x17] = 0x7,
    [0x1a] = 0x8, [0x09] = 0x9, [0x0a] = 0xa, [0x0b] = 0xb,
    [0x1e] = 0xc, [0x0d] = 0xd, [0x0e] = 0xe, [0x0f] = 0xf
  };

  uint16_t erpm =
    gcr[tlm & 0x1f] |
    (gcr[(tlm >> 5) & 0x1f] << 4) |
    (gcr[(tlm >> 10) & 0x1f] << 8) |
    (gcr[(tlm >> 15) & 0x1f] << 12);

  /* check CRC */
  crc = erpm ^ (erpm >> 8);
  crc ^= (crc >> 4);
  if (~crc & 0xf) return (struct tlmmsg){ .kind = TK3TLM_NONE };

  erpm >>= 4;

  /* decode */
  switch(erpm & 0xf00) {
    case 0x200:
      return (struct tlmmsg){ .kind = TK3TLM_TEMP };

    case 0x400:
      return (struct tlmmsg){ .kind = TK3TLM_VOLT };

    case 0x600:
      return (struct tlmmsg){ .kind = TK3TLM_AMP, .value = erpm & 0xff };

    case 0x000:
    case 0x800:
    case 0xa00:
    case 0xc00:
    case 0xe00:
      return (struct tlmmsg){ .kind = TK3TLM_NONE };

    default:
      if ((erpm & 0xfff) == 0xfff) /* saturated */
        return (struct tlmmsg){ .kind = TK3TLM_SAT };

      return (struct tlmmsg){
        .kind = TK3TLM_RPM,
        .value = (erpm & 0x1ff) << ((erpm & 0xe00) >> 9)
      };
  }
}


/* --- tk3srv_mapthrottle -------------------------------------------------- */

static uint16_t
tk3srv_pwmmapthrottle(int16_t throttle)
{
  return throttle < 0 ? 0 : throttle;
}

static uint16_t
tk3srv_pwm3dmapthrottle(int16_t throttle)
{
  return (1 << 14) + (throttle >> 1);
}

static uint16_t
tk3srv_dshotmapthrottle(int16_t throttle)
{
  return 48 +
    (((throttle < 0 ? 0 : throttle) * 2000U) >> 15);
}

static uint16_t
tk3srv_dshot3dmapthrottle(int16_t throttle)
{
  return 48 +
    (throttle < 0 ? 1000 : 0) +
    (((throttle < 0 ? -throttle : throttle) * 1000U) >> 15);
}


/* --- tk3srv_tlm ---------------------------------------------------------- */

static void
tk3srv_tlm()
{
  uint32_t tlm[8];
  struct tlmmsg m;
  int i;

  tk3srv_read(tlm);

  with_evlock() {
    for(i = 0; i < 8; i++) {
      if (!srv.motor[i].id) continue;

      /* rotate feedback info */
      if (srv.motor[i].feedback.bits >> 63)
        srv.motor[i].feedback.valid--;
      srv.motor[i].feedback.bits <<= 1;

      /* decode message */
      m = tk3srv_dshotdecode(tlm[i]);
      switch (m.kind) {
        default: break;

        case TK3TLM_AMP:
          srv.motor[i].current = m.value * 1000;
          break;

        case TK3TLM_RPM:
          srv.motor[i].period = m.value;
          srv.motor[i].feedback.bits |= 1;
          srv.motor[i].feedback.valid++;
          break;

        case TK3TLM_SAT:
          /* this usually indicates a failure, except when starting, so don't
           * reset starting bit */
          srv.motor[i].period = 0;
          if (srv.motor[i].spinning) {
            srv.motor[i].raw = srv.zero;
            srv.motor[i].spinning = 0;
            srv.motor[i].emerg = 1;
            tk3msg_log(TK3CH_USB0, "EESC #%c stalled", '0' + srv.motor[i].id);
          }
          break;
      }
    }
  }
}


/* --- tk3srv_arm ---------------------------------------------------------- */

static void
tk3srv_armpwm(void *data)
{
  struct srvmotor *motor = data;

  coroutinex(motor - srv.motor, 8) {
    /* zero throttle a bit */
    motor->raw = srv.zero;
    tk3clk_settimer(
      &motor->cmd, 0, srv.armdly, TK3EV_IDLE, tk3srv_armpwm, data);
    yield();

    /* armed */
    tk3srv_armed(data);
  }
}

static void
tk3srv_armdshot(void *data)
{
  struct srvmotor *motor = data;

  coroutinex(motor - srv.motor, 8) {
    /* zero - wait for detection */
    motor->raw = 0;
    tk3clk_settimer(
      &motor->cmd, 0, srv.armdly, TK3EV_IDLE, tk3srv_armdshot, data);
    yield();

    /* set extended telemetry mode */
    motor->raw = DSHOT_EXTENDED_TELEMETRY_ENABLE;
    tk3clk_settimer(
      &motor->cmd, 0, 12*srv.pidloop, TK3EV_IDLE, tk3srv_armdshot, data);
    yield();
    motor->raw = 0;
    tk3clk_settimer(
      &motor->cmd, 0, 35000 /* 35ms */, TK3EV_IDLE, tk3srv_armdshot, data);
    yield();

    /* set 3D mode */
    motor->raw = srv.rev3d ? DSHOT_CMD_3D_MODE_ON : DSHOT_CMD_3D_MODE_OFF;
    tk3clk_settimer(
      &motor->cmd, 0, 12*srv.pidloop, TK3EV_IDLE, tk3srv_armdshot, data);
    yield();

    /* set direction */
    motor->raw = motor->reversed ?
                 DSHOT_CMD_SPIN_DIRECTION_2 : DSHOT_CMD_SPIN_DIRECTION_1;
    tk3clk_settimer(
      &motor->cmd, 0, 12*srv.pidloop, TK3EV_IDLE, tk3srv_armdshot, data);
    yield();

    /* save settings (required for some firmware) */
    motor->raw = DSHOT_CMD_SAVE_SETTINGS;
    tk3clk_settimer(
      &motor->cmd, 0, 12*srv.pidloop, TK3EV_IDLE, tk3srv_armdshot, data);
    yield();

    /* delay for beep */
    motor->raw = srv.zero;
    tk3clk_settimer(
      &motor->cmd, 0, 250000 * motor->id /* 250ms *id */,
      TK3EV_IDLE, tk3srv_armdshot, data);
    yield();

    /* beep */
    motor->raw = DSHOT_CMD_BEEP1;
    tk3clk_settimer(&motor->cmd, 0,
                    12*srv.pidloop, TK3EV_IDLE, tk3srv_armdshot, data);
    yield();

    /* delay after beep */
    motor->raw = srv.zero;
    tk3clk_settimer(
      &motor->cmd, 0, 250000 * (10 - motor->id) /* 250ms *(10-id) */,
      TK3EV_IDLE, tk3srv_armdshot, data);
    yield();

    /* armed */
    tk3srv_armed(data);
  }
}

static void
tk3srv_armed(void *data)
{
  struct srvmotor *motor = data;
  int start;

  with_evlock() {
    motor->armed = 1;
    motor->raw = srv.zero;
    start = motor->starting;
  }

  if (start) tk3srv_start(motor->id);
}


/* --- tk3srv_start -------------------------------------------------------- */

int
tk3srv_start(int id)
{
  int i;

  if (!srv.hwinit) return 1;

  with_evlock() {
    for(i = 0; i < 8; i++) {
      if (id > 0 && srv.motor[i].id != id) continue;
      if (!srv.motor[i].id) continue;
      if (srv.motor[i].spinning) continue;

      srv.motor[i].emerg = 0;
      srv.motor[i].starting = 1;
      if (!srv.motor[i].armed) continue; /* end of arming will start it */

      /* reset PID state */
      srv.motor[i].servo = 0;
      srv.motor[i].target = 0.;
      srv.motor[i].pid = srv.idle;
      srv.motor[i].m0 = srv.motor[i].m1 = 0.;
      srv.motor[i].err0 = 0.;
      srv.motor[i].d0 = srv.motor[i].fd0 = 0.;

      /* wait a bit to reach idle velocity */
      srv.motor[i].raw = srv.map(srv.idle);
      tk3clk_settimer(
        &srv.motor[i].cmd, 0, 500000 /* 500ms */, TK3EV_IDLE,
        tk3srv_started, &srv.motor[i]);
    }
  }

  return 0;
}

static void
tk3srv_started(void *data)
{
  struct srvmotor *motor = data;
  const uint32_t now = tk3clk_us();

  with_evlock() {
    motor->starting = 0;
    /* if using some telemetry, ensure data has been received */
    if (srv.tlm && !motor->feedback.bits) return;

    motor->spinning = 1;
    motor->throt = srv.idle;

    srv.wdog.deadline = now + srv.wdog.dur;
    srv.tout.deadline = now + srv.tout.dur;
  }
}


/* --- tk3srv_stop --------------------------------------------------------- */

int
tk3srv_stop(int id)
{
  int i;

  with_evlock() {
    for(i = 0; i < 8; i++) {
      if (id > 0 && srv.motor[i].id != id) continue;

      srv.motor[i].spinning = 0;
      srv.motor[i].starting = 0;
      srv.motor[i].raw = srv.zero;
      srv.motor[i].period = 0;
    }
  }

  return 0;
}


/* --- tk3srv_setpid ------------------------------------------------------- */

void
tk3srv_setpid(int id, float Kp, float Ki, float Kd, float dfc)
{
  int i, s;

  const double dt = srv.pidloop / 1e6;	/* control loop period */
  const double tau = 6.2832 * dfc * dt;	/* LP filter ωc * dt */

  const double a0 = Kp + Ki * dt;
  const double a1 = -Kp;

  const double a0d = Kd / dt;
  const double a1d = - 2 * Kd / dt;

  const double a0fd = tau / (2 + tau);	/* PID derivative filter */
  const double a1fd = (2 - tau) / (2 + tau);

  with_evlock() {
    for(i = s = 0; i < 8; i++) {
      if (srv.motor[i].id != id) continue;

      srv.motor[i].a0 = a0;
      srv.motor[i].a1 = a1;

      srv.motor[i].a0d = a0d;
      srv.motor[i].a1d = a1d;

      srv.motor[i].a0fd = a0fd;
      srv.motor[i].a1fd = a1fd;
      s++;
    }
  }

  if (!s) {
    tk3msg_log(TK3CH_USB0, "Ebad ESC id %c", '0'+id);
    return;
  }
}


/* --- tk3srv_beep --------------------------------------------------------- */

void
tk3srv_beep(void *data)
{
  int i;

  /* this is only for DSHOT modes */
  switch (srv.mode) {
    default: return;

    case TK3SRV_DSHOT:
    case TK3SRV_BIDSHOT: break;
  }

  with_evlock() {
    for(i = 0; i < 8; i++) {
      if (!srv.motor[i].id) continue;
      if (!srv.motor[i].armed) continue;
      if (srv.motor[i].starting || srv.motor[i].spinning) continue;

      if (!data)
        srv.motor[i].raw = srv.zero;
      else {
        srv.motor[i].raw = DSHOT_CMD_BEEP1;
        tk3clk_settimer(&srv.motor[i].cmd, 0,
                        12 * srv.pidloop, TK3EV_IDLE, tk3srv_beep, NULL);
      }
    }
  }
}


/* --- tk3srv_throttle ----------------------------------------------------- */

void
tk3srv_throttle(const int16_t throt[8])
{
  int i;
  const uint32_t now = tk3clk_us();

  with_evlock() {
    if (!tk3rc_killswitch) /* normal case */
      for(i = 0; i < 8; i++) {
        if (!srv.motor[i].id) continue;
        if (!srv.motor[i].spinning) continue;

        srv.motor[i].servo = 0;
        srv.motor[i].throt = throt[srv.motor[i].id - 1] << 5;
      }

    else /* kill switch active, stop motors */
      for(i = 0; i < 8; i++) {
        if (!srv.motor[i].spinning) continue;

        srv.motor[i].spinning = 0;
        srv.motor[i].raw = srv.zero;
      }

    srv.wdog.deadline = now + srv.wdog.dur;
    srv.tout.deadline = now + srv.tout.dur;
  }
}


/* --- tk3srv_velocity ----------------------------------------------------- */

void
tk3srv_velocity(const int16_t hperiod[8] /* half period */)
{
  const float k = (float)1e6 * srv.pmul / 2;
  const uint32_t now = tk3clk_us();
  int16_t h;
  int i;

  with_evlock() {
    if (!tk3rc_killswitch) /* normal case */
      for(i = 0; i < 8; i++) {
        if (!srv.motor[i].id) continue;
        if (!srv.motor[i].spinning) continue;

        srv.motor[i].servo = 1;
        h = hperiod[srv.motor[i].id - 1];
        if (!srv.rev3d && h < 0)
          srv.motor[i].target = k / 0x7fff;
        else {
          if (!__builtin_signbit(srv.motor[i].target) != !(h >> 15)) {
            srv.motor[i].pid = srv.idle;
            srv.motor[i].m0 = srv.motor[i].m1 = 0.;
            srv.motor[i].err0 = 0.;
            srv.motor[i].d0 = srv.motor[i].fd0 = 0.;
          }
          srv.motor[i].target = k / h;
        }
      }

    else /* kill switch active, stop motors */
      for(i = 0; i < 8; i++) {
        if (!srv.motor[i].spinning) continue;

        srv.motor[i].spinning = 0;
        srv.motor[i].raw = srv.zero;
      }

    srv.wdog.deadline = now + srv.wdog.dur;
    srv.tout.deadline = now + srv.tout.dur;
  }
}


/* --- tk3srv_motdata ------------------------------------------------------ */

void
tk3srv_motdata(void *arg)
{
  static uint8_t seq;
  uint16_t period, pwm, current;
  uint8_t state;
  uint32_t v;
  int i;

  for(i = 0; i < 8; i++) {
    if (!srv.motor[i].id) continue;

    with_evlock() {
      state =
        (srv.motor[i].emerg << 7) |
        (srv.motor[i].servo << 6) |
        (srv.motor[i].spinning << 5) |
        (srv.motor[i].starting << 4) |
        (srv.motor[i].id & 0xf);

      v = state ? srv.motor[i].period * srv.pmul : 0;
      period = (v >= 0xffff ? 0xffff : v)/2;
      if (srv.motor[i].throt < 0) period = -period;

      pwm = srv.motor[i].throt >> 5;

      current = srv.motor[i].current;
    }

    tk3msg_log(TK3CH_USB0, "M%1%1%2%2%2", seq, state, period, pwm, current);
  }
  seq++;
}


/* --- tk3srv_servo -------------------------------------------------------- */

static void
tk3srv_servo(struct srvmotor *motor)
{
  float m2;
  float err1, d1, fd1;

  /* Discrete PID implementation, see https://en.wikipedia.org/wiki/
   * Proportional%E2%80%93integral%E2%80%93derivative_controller#Pseudocode
   */
  m2 = motor->m1;
  motor->m1 = motor->m0;
  motor->m0 = (float)1e6/motor->period;

  err1 = motor->err0;
  motor->err0 = __builtin_fabsf(motor->target) - motor->m0;

  /* kP, kI */
  motor->pid += motor->a0 * motor->err0 + motor->a1 * err1;

  /* filtered kD */
  if (motor->a0d) {
    d1 = motor->d0;
    fd1 = motor->fd0;

    motor->d0 =
      motor->a0d * motor->m0 + motor->a1d * motor->m1 + motor->a0d * m2;
    motor->fd0 = motor->a0fd * (motor->d0 + d1) + motor->a1fd * fd1;
    motor->pid += motor->fd0;
  }

  if (motor->pid > 32767.) motor->pid = 32767.;
  else if (motor->pid < -32768.) motor->pid = -32768.;

  motor->throt = motor->pid < 1. ? 1 : motor->pid;
  if (__builtin_signbit(motor->target))
      motor->throt = -motor->throt;
}


/* --- tk3srv_ctrlloop ----------------------------------------------------- */

static void
tk3srv_ctrlloop(void *data)
{
  const uint32_t now = tk3clk_us();
  uint16_t target[8];
  int i;

  /* update telemetry if enabled */
  if (srv.tlm) srv.tlm();

  /* compute target throttle */
  with_evlock() {
    for(i = 0; i < 8; i++) {
      if (srv.motor[i].spinning) {
        if (now > srv.tout.deadline) {
          /* stop */
          srv.motor[i].spinning = 0;
          target[i] = srv.motor[i].raw = srv.zero;
          continue;
        } else if (now > srv.wdog.deadline) {
          /* go to idle pwm, reset wdog */
          srv.motor[i].servo = 0;
          srv.motor[i].throt = srv.motor[i].throt > 0 ? srv.idle : -srv.idle;
          srv.wdog.deadline = now + srv.wdog.dur;
        }

        if (srv.motor[i].servo && (srv.motor[i].feedback.bits & 1))
          tk3srv_servo(&srv.motor[i]);

        target[i] = srv.map(srv.motor[i].throt);
      } else
        target[i] = srv.motor[i].raw;
    }
  }

  tk3srv_send(target);
}
