/*
 * Copyright (c) 2019-2025 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Tue Jul 23 2019
 */
#include "autoconf.h"

#include <stdarg.h>
#include <string.h>

#include "tk3-paparazzi.h"

/* settings */
tk3setting(TK3SET_ACCRANGE, 8, "±2, 4, 8, 16 (g)", 40);
tk3setting(TK3SET_GYRRANGE, 1000, "±250, 500, 1000, 2000 (°/s)", 41);


/* active sensors */
static struct {
  int active;
#ifdef TK3_MPU9250
  int mpu9250;
#endif
#ifdef TK3_ICM42688
  int icm42688;
#endif
#ifdef TK3_AK8963
  int ak8963;
#endif
#ifdef TK3_RM3100
  int rm3100;
#endif
#ifdef TK3_LIS3MDL
  int lis3mdl;
#endif
} sensorp;

/* reader jobs */
static struct {
  struct {
    uint32_t chans;
    uint8_t seq;
  } imu, mag;
} reader;

static tk3timer(restart);

static void	tk3sens_restart(void *arg);
static void	tk3sens_reader(void *iostatus);


/* --- tk3sens_init -------------------------------------------------------- */

static uint32_t tk3sens_lastinit;

void
tk3sens_init()
{
  int s;
  tk3sens_fini();

  tk3fb_flash(TK3FB_SENS, 100000);
  tk3fb_on(TK3FB_SENS_BUSY);
  tk3sens_hwinit();

  /* probe sensors */
  s = 0;
#ifdef TK3_MPU9250
  if (!s && tk3sens_mpu9250_probe()) {
    if (!tk3sens_mpu9250_init())
      s = sensorp.mpu9250 = 1;
  }
#endif
#ifdef TK3_ICM42688
  if (!s && tk3sens_icm42688_probe()) {
    if (!tk3sens_icm42688_init())
      s = sensorp.icm42688 = 1;
  }
#endif

  if (!s) {
    tk3msg_log(TK3CH_USB0, "Eno IMU detected");
    /* start over after some time */
    tk3sens_restart(NULL);
    return;
  }

  s = 0;
#ifdef TK3_RM3100
  if (!s && tk3sens_rm3100_probe()) {
    if (!tk3sens_rm3100_init())
      s = sensorp.rm3100 = 1;
  }
#endif
#ifdef TK3_AK8963
  if (!s && tk3sens_ak8963_probe()) {
    if (!tk3sens_ak8963_init())
      s = sensorp.ak8963 = 1;
  }
#endif
#ifdef TK3_LIS3MDL
  if (!s && tk3sens_lis3mdl_probe()) {
    if (!tk3sens_lis3mdl_init())
      s = sensorp.lis3mdl = 1;
  }
#endif

  if (!s) {
    tk3msg_log(TK3CH_USB0, "Eno magnetometer detected");
    /* start over after some time */
    tk3sens_restart(NULL);
    return;
  }

  tk3fb_off(TK3FB_SENS_BUSY);
  tk3sens_lastinit = tk3clk_us();
  sensorp.active = 1;
}

static void
tk3sens_restart(void *arg)
{
  static uint32_t wait;
  uint32_t now;

  if (arg) {
    tk3sens_init();
    return;
  }

  /* exponential backoff timer */
  now = tk3clk_us();
  wait *= 2 * (now - tk3sens_lastinit < wait);
  if (wait < 10000) wait = 10000; /* 10ms min */
  else if (wait > 5000000) wait = 5000000; /* 5s max */

  /* schedule restart */
  tk3sens_fini();
  tk3fb_on(TK3FB_SENS_ERR);
  tk3clk_settimer(
    &restart, 0, wait, TK3EV_IDLE, tk3sens_restart, tk3sens_restart);
  tk3msg_log(TK3CH_USB0, "Nresetting sensors");
}


/* --- tk3sens_fini -------------------------------------------------------- */

void
tk3sens_fini()
{
  sensorp.active = 0;
  tk3clk_deltimer(&restart);
  memset(&sensorp, 0, sizeof(sensorp));
  tk3sens_hwfini();
  tk3fb_off(TK3FB_SENS_ERR);
  tk3fb_off(TK3FB_SENS_BUSY);
}


/* --- tk3sens_send* ------------------------------------------------------- */

void
tk3sens_sendimu(void *arg)
{
  enum tk3ch channel = (enum tk3ch)arg;

  if (!sensorp.active) return;

  reader.imu.chans |= 1<<channel;
  tk3sens_reader(NULL);
}

void
tk3sens_sendmag(void *arg)
{
  enum tk3ch channel = (enum tk3ch)arg;

  if (!sensorp.active) return;

  reader.mag.chans |= 1<<channel;
  tk3sens_reader(NULL);
}


/* --- tk3sens_reader ------------------------------------------------------ */

static void
tk3sens_reader(void *iostatus)
{
  unsigned int chans;
  uint32_t status;

  coroutine() {
    while (reader.imu.chans || reader.mag.chans) {
      /* IMU */
      if (reader.imu.chans) {
        int32_t a[3], w[3];
        int16_t t;

        status = 0;
        tk3fb_on(TK3FB_SENS);
#ifdef TK3_MPU9250
        status |= sensorp.mpu9250;
        if (sensorp.mpu9250) tk3sens_mpu9250_aioread(tk3sens_reader);
#endif
#ifdef TK3_ICM42688
        status |= sensorp.icm42688;
        if (sensorp.icm42688) tk3sens_icm42688_aioread(tk3sens_reader);
#endif
        if (status) {
          do yield(); while (!iostatus);
          status = *(uint32_t *)iostatus;
        }
        tk3fb_off(TK3FB_SENS);

        chans = reader.imu.chans;
        reader.imu.chans = 0;

        if (status) {
          if (sensorp.active) tk3sens_restart(NULL);
          break;
        }

        status = 1;
#ifdef TK3_MPU9250
        status = tk3sens_mpu9250_data(a, w, &t);
#endif
#ifdef TK3_ICM42688
        status = tk3sens_icm42688_data(a, w, &t);
#endif
        if (!status) {
          reader.imu.seq++;
          for(; chans; chans &= chans - 1 /* clear LSB */) {
            tk3msg_log(__builtin_ffs(chans)-1, "I%1%2%2%2%2%2%2%2",
                       reader.imu.seq, a[0], a[1], a[2], w[0], w[1], w[2], t);
          }
        }
      }

      /* magnetometer */
      if (reader.mag.chans) {
        int32_t m[3];

        status = 0;
        tk3fb_on(TK3FB_SENS);
#ifdef TK3_RM3100
        status |= sensorp.rm3100;
        if (sensorp.rm3100) tk3sens_rm3100_aioread(tk3sens_reader);
#endif
#ifdef TK3_AK8963
        status |= sensorp.ak8963;
        if (sensorp.ak8963) tk3sens_ak8963_aioread(tk3sens_reader);
#endif
#ifdef TK3_LIS3MDL
        status |= sensorp.lis3mdl;
        if (sensorp.lis3mdl) tk3sens_lis3mdl_aioread(tk3sens_reader);
#endif
        if (status) {
          do yield(); while (!iostatus);
          status = *(uint32_t *)iostatus;
        }
        tk3fb_off(TK3FB_SENS);

        chans = reader.mag.chans;
        reader.mag.chans = 0;
        if (status) {
          if (sensorp.active) tk3sens_restart(NULL);
          break;
        }

        status = 1;
#ifdef TK3_RM3100
        if (sensorp.rm3100) status = tk3sens_rm3100_data(m);
#endif
#ifdef TK3_AK8963
        if (sensorp.ak8963) status = tk3sens_ak8963_data(m);
#endif
#ifdef TK3_LIS3MDL
        if (sensorp.lis3mdl) status = tk3sens_lis3mdl_data(m);
#endif
        if (!status) {
        reader.mag.seq++;
          for(; chans; chans &= chans - 1 /* clear LSB */) {
            tk3msg_log(__builtin_ffs(chans)-1, "C%1%2%2%2",
                       reader.mag.seq, m[0], m[1], m[2]);
          }
        }
      }
    }
  }
}
