/*
 * Copyright (c) 2020-2021,2024-2025 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Thu Nov 26 2020
 */
#include "autoconf.h"

#include "tk3-paparazzi.h"

/* RM3100 regs */
enum {
  CMM =		0x1,

  CCXH =	0x4,
  CCXL,
  CCYH,
  CCYL,
  CCZH,
  CCZL,

  TMRC =	0xb,

  MX =		0x24,

  STATUS =	0x34,
  HSHAKE,
  REVID
};

/* sensor data */
struct __attribute__((packed)) rm3100_regs {
  int8_t	x2;	uint8_t x1, x0;
  int8_t	y2;	uint8_t y1, y0;
  int8_t	z2;	uint8_t z1, z0;

  uint8_t	status;
};

static __attribute__ ((const)) struct rm3100_regs *
rm3100_regs(void) { return tk3_padmem(struct rm3100_regs, TK3_RM3100_MEM); }


static int	tk3sens_rm3100_config(void);
/* static int	tk3sens_rm3100_selftest(void); */


/* --- tk3sens_rm3100_probe ------------------------------------------------ */

int
tk3sens_rm3100_probe()
{
  uint8_t * const v = tk3_padmem(uint8_t, TK3_RM3100_MEM);

  /* check REVID */
  if (tk3sens_read(TK3_RM3100, 1, REVID, v)) return 0;
  if (*v != 0x22) return 0;

  return 1;
}


/* --- tk3sens_rm3100_init ------------------------------------------------- */

int
tk3sens_rm3100_init()
{
  /* if (tk3sens_rm3100_selftest()) return 1; */
  if (tk3sens_rm3100_config()) return 1;

  tk3msg_log(TK3CH_USB0, "NRM3100 configured");
  return 0;
}


/* --- tk3sens_rm3100_config ----------------------------------------------- */

static int
tk3sens_rm3100_config()
{
  static uint16_t cc = 200;
  uint8_t h, l;

  /* stop measurement */
  if (tk3sens_write(
        TK3_RM3100, 1, CMM, 0x0))
    goto fail;

  /* cycle count */
  h = cc >> 8; l = cc & 0xff;
  if (tk3sens_write(
        TK3_RM3100, 6, CCXH, h, CCXL, l, CCYH, h, CCYL, l, CCZH, h, CCZL, l))
    goto fail;

  /* update rate */
  if (tk3sens_write(
        TK3_RM3100, 1, TMRC, 0x92 /* 600Hz */))
    goto fail;

  /* DRDY config */
  if (tk3sens_write(
        TK3_RM3100, 1, HSHAKE, 0xb /* DRC1 | DRC0 */))
    goto fail;

  /* continuous measurement */
  if (tk3sens_write(
        TK3_RM3100, 1, CMM, 0x71 /* CMZ | CMY | CMX | START */))
    goto fail;

  return 0;

fail:
  tk3msg_log(TK3CH_USB0, "Emagnetometer configuration failed");
  return 1;
}


/* --- tk3sens_rm3100_aioread ---------------------------------------------- */

static void (*gcb)(void *);

static void
tk3sens_rm3100_aioreadreg(void *iostatus)
{
  /* if (*(uint32_t *)iostatus) rm3100_regs->status = 0; */

  tk3sens_aioread(
    TK3_RM3100, 9/* sizeof(rm3100_regs) */, MX, rm3100_regs(), gcb);
}

void
tk3sens_rm3100_aioread(void (*cb)(void *))
{
  gcb = cb;
  tk3sens_aioread(
    TK3_RM3100, 1, STATUS, &rm3100_regs()->status, tk3sens_rm3100_aioreadreg);
}

int
tk3sens_rm3100_data(int32_t m[3])
{
  int32_t rm[3];
  int i;

  if (!(rm3100_regs()->status & 0x80 /* DRDY */)) return 1;

  rm[0] = rm3100_regs()->x2 << 16 | rm3100_regs()->x1 << 8 | rm3100_regs()->x0;
  rm[1] = rm3100_regs()->y2 << 16 | rm3100_regs()->y1 << 8 | rm3100_regs()->y0;
  rm[2] = rm3100_regs()->z2 << 16 | rm3100_regs()->z1 << 8 | rm3100_regs()->z0;

  for(i = 0; i < 3; i++)
    rm[i] = rm[i] * 4/3;

  m[0] = rm[0];
  m[1] = -rm[1];
  m[2] = -rm[2];

  return 0;
}
