/*
 * Copyright (c) 2019-2020,2024-2025 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Fri Jul  5 2019
 */
#include "autoconf.h"

#include "tk3-paparazzi.h"

/* settings */
tk3setting(TK3SET_RCKILL, -1, "RC kill channel [0-7], -1=disable", 30);
tk3setting(TK3SET_RCKILL_MODE, 0, "kill trigger: -1=low, 0=edge, +1=high", 31);

int tk3rc_killswitch;
static struct { int chan, mode; } rckill;

static void	tk3rc_icuwidth(ICUDriver *icup);
static void	tk3rc_icuovf(ICUDriver *icup);

static int16_t tk3rc_channels[8];
static unsigned int tk3rc_channel;


/* --- tk3rc_init ---------------------------------------------------------- */

static const ICUConfig tk3rc_icuconfig = {
  .mode =		ICU_INPUT_ACTIVE_HIGH,
  .channel =		TK3_RC_CHANNEL,
  .frequency =		1000000,		/* 1MHz ICU clock frequency */
  .width_cb =		tk3rc_icuwidth,
  .overflow_cb =	tk3rc_icuovf
};

int
tk3rc_init()
{
  tk3rc_killswitch = 0;
  rckill.chan = tk3set_get(TK3SET_RCKILL, -1);
  rckill.mode = tk3set_get(TK3SET_RCKILL_MODE, 0);

  tk3rc_hwinit();
  icuStart(&TK3_RCD, &tk3rc_icuconfig);
  icuStartCapture(&TK3_RCD);
  icuEnableNotifications(&TK3_RCD);

  return 0;
}


/* --- tk3rc_channel_data -------------------------------------------------- */

const int16_t *
tk3rc_channel_data()
{
  return tk3rc_channels;
}


/* --- tk3rc_icuwidth ------------------------------------------------------ */

static void
tk3rc_icuwidth(ICUDriver *icup)
{
  icucnt_t w = icuGetWidthX(icup);
  int16_t c;

  if (tk3rc_channel >= sizeof(tk3rc_channels)/sizeof(tk3rc_channels[0]))
    tk3rc_channel = 0;
  if (w > 5000) {
    tk3rc_channel = 0;
    return;
  }

  if (w > 1500) w = 1500; else if (w < 700) w = 700;
  c = (w - 1100) * 81;

  /* check kill switch */
  if (rckill.chan == tk3rc_channel) {
    if (rckill.mode) { /* mode -1/+1 */
      if (rckill.mode < 0 ? c < -1<<14 : c >  1<<14) {
        if (!tk3rc_killswitch) {
          tk3fb_on(TK3FB_RCKILL);
          tk3srv_stop(-1);
          tk3mkbl_stop();

          tk3rc_killswitch = 1;
          tk3msg_log(TK3CH_USB0, "ERC kill switch detected");
        }
      } else {
        if (tk3rc_killswitch) {
          tk3fb_off(TK3FB_RCKILL);
          tk3rc_killswitch = 0;
          tk3msg_log(TK3CH_USB0, "NRC kill switch released");
        }
      }
    } else { /* mode 0 */
      if (c - tk3rc_channels[tk3rc_channel] > 1<<14 ||
          c - tk3rc_channels[tk3rc_channel] < -1<<14) {
        if (!tk3rc_killswitch) {
          tk3srv_stop(-1);
          tk3mkbl_stop();
        }
        tk3rc_killswitch ^= 1;
        tk3fb_set(TK3FB_RCKILL, tk3rc_killswitch);
        tk3msg_log(TK3CH_USB0,
                   tk3rc_killswitch ? "ERC kill switch detected":
                   "NRC kill switch released");
      }
    }
  }

  with_syslock() {
    tk3rc_channels[tk3rc_channel++] = c;
  }
}


/* --- tk3rc_icuovf -------------------------------------------------------- */

static void
tk3rc_icuovf(ICUDriver *icup)
{
  tk3rc_channel = 0;

  with_syslock() {
    unsigned int i;

    for(i = 0; i < sizeof(tk3rc_channels)/sizeof(tk3rc_channels[0]); i++)
      tk3rc_channels[i] = 0;
  }
}
