/*
 * Copyright (c) 2019-2021 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Fri Jul 19 2019
 */
#include "autoconf.h"

#include <stdarg.h>

#include "tk3-paparazzi.h"

static inline void	tk3msg_encodec(struct tk3iob *iob, char c);
static int		tk3msg_bufrecv(struct tk3iob *iob, struct tk3msg *msg);

static struct { struct tk3iob *in, *out; } tk3ch_iob[TK3CH_MAX];


/* --- tk3msg_init ---------------------------------------------------------- */

void
tk3msg_init(void (*cb)(struct tk3iob *))
{
  enum tk3ch i;

  /* initialize channels io buffers */
  for(i = 0; i < TK3CH_MAX; i++) {
    switch(i) {
      case TK3CH_USB0:
        tk3ch_iob[i].in = tk3usb_iniob(0);
        tk3ch_iob[i].out = tk3usb_outiob(0);
        break;
      case TK3CH_USB1:
        tk3ch_iob[i].in = tk3usb_iniob(1);
        tk3ch_iob[i].out = tk3usb_outiob(1);
        break;
      case TK3CH_MKBL:
        tk3ch_iob[i].in = tk3mkbl_iniob();
        tk3ch_iob[i].out = tk3mkbl_outiob();
        break;

      case TK3CH_MAX: /* not reached */ break;
    }

    tk3ch_iob[i].in->cb = cb;
  }
}


/* --- tk3msg_log ---------------------------------------------------------- */

void
tk3msg_log(enum tk3ch channel, const char *fmt, ...)
{
  struct tk3iob *iob = tk3ch_iob[channel].out;
  va_list ap;
  char c;

  va_start(ap, fmt);

  with_syslock() {
    tk3iob_putc(iob, '^');
    while((c = *fmt++)) {
      if (__builtin_expect(c != '%', 1)) {
        tk3msg_encodec(iob, c);
        continue;
      }

      switch(*fmt++) {
        case '1':
          tk3msg_encodec(iob, va_arg(ap, int));
          break;

        case '2': {
          uint16_t x = va_arg(ap, unsigned int);
          tk3msg_encodec(iob, (x >> 8) & 0xff);
          tk3msg_encodec(iob, x & 0xff);
          break;
        }

        case '4': {
          uint32_t x = va_arg(ap, uint32_t);
          tk3msg_encodec(iob, (x >> 24) & 0xff);
          tk3msg_encodec(iob, (x >> 16) & 0xff);
          tk3msg_encodec(iob, (x >> 8) & 0xff);
          tk3msg_encodec(iob, x & 0xff);
          break;
        }

        case 'c': {
          char x = va_arg(ap, int);
          tk3msg_encodec(iob, x);
          break;
        }

        case 's': {
          char *x = va_arg(ap, char *);
          while(*x) tk3msg_encodec(iob, *x++);
          break;
        }

        case 'b': {
          uint8_t *x = va_arg(ap, uint8_t *);
          size_t s = va_arg(ap, size_t);;
          while(s--) tk3msg_encodec(iob, *x++);
          break;
        }
      }
    }
    tk3iob_putc(iob, '$');
  }

  va_end(ap);
}


/* --- tk3msg_log_buffer --------------------------------------------------- */

void
tk3msg_log_buffer(enum tk3ch channel, const uint8_t *buffer, uint8_t len)
{
  struct tk3iob *iob = tk3ch_iob[channel].out;

  with_syslock() {
    tk3iob_putc(iob, '^');
    while(len--)
      tk3msg_encodec(iob, *buffer++);
    tk3iob_putc(iob, '$');
  }
}


/* --- tk3msg_encodec ------------------------------------------------------ */

static inline void
tk3msg_encodec(struct tk3iob *iob, char c)
{
  if (__builtin_expect(c == '^' || c == '$' || c == '!' || c == '\\', 0)) {
    tk3iob_putc(iob, '\\');
    c = ~c;
  }
  tk3iob_putc(iob, c);
}


/* --- tk3msg_recv --------------------------------------------------------- */

static uint8_t tk3msg_buffer[TK3CH_MAX][64];
static struct tk3msg tk3msg_pending[TK3CH_MAX] = {
  [TK3CH_USB0] = tk3msg(TK3CH_USB0, tk3msg_buffer[TK3CH_USB0]),
  [TK3CH_USB1] = tk3msg(TK3CH_USB1, tk3msg_buffer[TK3CH_USB1]),
  [TK3CH_MKBL] = tk3msg(TK3CH_MKBL, tk3msg_buffer[TK3CH_MKBL])
};

const struct tk3msg *
tk3msg_recv(void)
{
  static enum tk3ch ch;
  struct tk3msg *msg;

  enum tk3ch i;

  /* round robin processing of all channels */
  for(i = 0; i < TK3CH_MAX; i++) {
    if (++ch >= TK3CH_MAX) ch = 0;
    msg = &tk3msg_pending[ch];

    if (msg->state.eagain || tk3msg_bufrecv(tk3ch_iob[ch].in, msg)) {
      msg->state.eagain = 0;
      return msg;
    }
  }

  return NULL;
}


/* --- tk3msg_pushback ----------------------------------------------------- */

void
tk3msg_pushback(enum tk3ch channel, size_t to, size_t from)
{
  /* move [from; end[ to "to" */
  if (to != from) {
    const uint8_t *f = &tk3msg_pending[channel].bot[from];
    uint8_t *t = &tk3msg_pending[channel].bot[to];

    while(f < tk3msg_pending[channel].end)
      *t++ = *f++;
    tk3msg_pending[channel].end = t;
    tk3msg_pending[channel].len = t - tk3msg_pending[channel].bot;
  }

  tk3msg_pending[channel].state.eagain = 1;
}


/* --- tk3msg_bufrecv ------------------------------------------------------ */

static int
tk3msg_bufrecv(struct tk3iob *iob, struct tk3msg *msg)
{
  static const char magic[] = "reset";
  static uint32_t skipped;
  uint8_t c;

  while(tk3iob_getc(iob, &c))
    switch (c) {
      case '^':
        if (msg->state.start)
          tk3msg_log(TK3CH_USB0, "Epartial message on %s", iob->id);
        if (skipped) {
          tk3msg_log(TK3CH_USB0, "Eskipped raw bytes on %s", iob->id);
          skipped = 0;
        }

        msg->state.start = 1;
        msg->state.magic = 0;
        msg->state.escape = 0;
        msg->len = 0;
        msg->end = msg->bot;
        break;

      case '$':
        if (!msg->state.start) break;
        msg->state.start = 0;
        msg->state.magic = 0;
        return 1;

      case '!':
        msg->state.start = 0;
        break;

      case '\\':
        msg->state.escape = 1;
        break;

      default:
        if (!msg->state.start) {
          /* detect magic bytes sequence */
          if (c != magic[msg->state.magic]) {
            msg->state.magic = 0;
          } else if (!magic[++msg->state.magic]) {
            msg->len = 1;
            *msg->bot = '\0';
            msg->end = msg->bot + 1;
            return 1;
          }
          skipped++;
          break;
        }

        if (msg->end >= msg->top) {
          tk3msg_log(msg->channel, "Emessage too long");
          msg->state.start = 0;
          break;
        }
        if (msg->state.escape) {
          msg->state.escape = 0;
          c = ~c;
        }

        *msg->end++ = c;
        msg->len++;
        break;
    }

  return 0;
}
