/*
 * Copyright (c) 2019-2022,2024-2025 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Tue Jul 23 2019
 */
#include "autoconf.h"

#include "tk3-paparazzi.h"

/* MPU9250 regs */
enum {
  MPU9250_SELF_TEST_X_GYRO =	0x0,

  MPU9250_SELF_TEST_X_ACCEL =	0xd,

  MPU9250_CONFIG =		0x1a,
  MPU9250_GYRO_CONFIG,
  MPU9250_ACCEL_CONFIG,
  MPU9250_ACCEL_CONFIG2,

  MPU9250_I2C_INT_PIN_CFG =	0x37,
  MPU9250_I2C_INT_ENABLE,

  MPU9250_INT_STATUS =		0x3a,

  MPU9250_PWR_MGMT_1 =		0x6b,

  MPU9250_WHOAMI =		0x75
};

/* sensor data */
struct __attribute__((packed)) mpu9250_regs {
  uint8_t	int_status;

  int8_t	accel_xout_h;	uint8_t accel_xout_l;
  int8_t	accel_yout_h;	uint8_t accel_yout_l;
  int8_t	accel_zout_h;	uint8_t accel_zout_l;

  int8_t	temp_out_h;	uint8_t temp_out_l;

  int8_t	gyro_xout_h;	uint8_t gyro_xout_l;
  int8_t	gyro_yout_h;	uint8_t gyro_yout_l;
  int8_t	gyro_zout_h;	uint8_t gyro_zout_l;
};

static __attribute__ ((const)) struct mpu9250_regs *
mpu9250_regs(void) { return tk3_padmem(struct mpu9250_regs, TK3_MPU9250_MEM); }

static struct { int a[3], g[3]; } stvalid;


static int	tk3sens_mpu9250_config(void);
static int	tk3sens_mpu9250_selftest(void);
static int	tk3sens_mpu9250_reset(void);
static double	powi(double x, uint32_t y);


/* --- tk3sens_mpu9250_probe ----------------------------------------------- */

int
tk3sens_mpu9250_probe()
{
  uint8_t * const v = tk3_padmem(uint8_t, TK3_MPU9250_MEM);

  /* check MPU9250 id */
  if (tk3sens_read(TK3_MPU9250, 1, MPU9250_WHOAMI, v)) return 0;
  if (*v != 0x71) return 0;

  return 1;
}


/* --- tk3sens_mpu9250_init ------------------------------------------------ */

int
tk3sens_mpu9250_init()
{
  char *arange = "NaN", *grange = "NaN";

  if (tk3sens_mpu9250_reset()) return 1;
  if (tk3sens_mpu9250_selftest()) return 1;
  if (tk3sens_mpu9250_config()) return 1;

  switch(tk3set_get(TK3SET_ACCRANGE, 8)) {
    case  2: arange = "2"; break;
    case  4: arange = "4"; break;
    case  8: arange = "8"; break;
    case 16: arange = "16"; break;
  }
  switch(tk3set_get(TK3SET_GYRRANGE, 1000)) {
    case  250: grange = "250"; break;
    case  500: grange = "500"; break;
    case 1000: grange = "1000"; break;
    case 2000: grange = "2000"; break;
  }

  tk3msg_log(TK3CH_USB0, "NMPU9250 configured, ±%sg, ±%s°/s", arange, grange);
  return 0;
}


/* --- tk3sens_mpu9250_config ---------------------------------------------- */

static int
tk3sens_mpu9250_config()
{
  int arange, grange;

  /* clock source */
  if (tk3sens_write(
        TK3_MPU9250, 1,
        MPU9250_PWR_MGMT_1,		0x01 /* PLL */))
    goto fail;

  /* sensors */
  arange = tk3set_get(TK3SET_ACCRANGE, 8);
  switch(arange) {
    case  2: arange = 0; break;
    case  4: arange = 1; break;
    case  8: arange = 2; break;
    case 16: arange = 3; break;
    default:
      tk3msg_log(TK3CH_USB0, "EMPU9250 bad accelerometer range setting");
      goto fail;
  }

  grange = tk3set_get(TK3SET_GYRRANGE, 1000);
  switch(grange) {
    case  250: grange = 0; break;
    case  500: grange = 1; break;
    case 1000: grange = 2; break;
    case 2000: grange = 3; break;
    default:
      tk3msg_log(TK3CH_USB0, "EMPU9250 bad gyroscope range setting");
      goto fail;
  }

  if (tk3sens_write(
        TK3_MPU9250, 4,
        MPU9250_CONFIG,			1 /* DLPF 184Hz */,
        MPU9250_GYRO_CONFIG,		grange << 3 /* scale, DLPF */,
        MPU9250_ACCEL_CONFIG,		arange << 3 /* scale */,
        MPU9250_ACCEL_CONFIG2,		0 /* DLPF 460Hz */))
    goto fail;

  /* INT enable */
  if (tk3sens_write(
        TK3_MPU9250, 2,
        MPU9250_I2C_INT_ENABLE,		0,
        MPU9250_I2C_INT_PIN_CFG,	0x2 /* BYPASS_EN */))
    goto fail;

  return 0;

fail:
  tk3msg_log(TK3CH_USB0, "EMPU9250 configuration failed");
  return 1;
}


/* --- tk3sens_mpu9250_selftest -------------------------------------------- */

static int
tk3sens_mpu9250_selftest()
{
  int32_t accel[3], gyro[3];
  uint8_t fst[3];
  double r;
  int i, t;

  /* enable self-test */
  if (tk3sens_write(
        TK3_MPU9250, 4,
        MPU9250_CONFIG,			0x2 /* DLPF 92Hz */,
        MPU9250_ACCEL_CONFIG2,		0x2 /* DLPF 92Hz */,
        MPU9250_GYRO_CONFIG,		0xe0 /* ST, ±250°/s, DLPF */,
        MPU9250_ACCEL_CONFIG,		0xe0 /* ST, ±2g */)) {
    tk3msg_log(TK3CH_USB0, "EMPU9250 self-test configuration failed");
    goto fail;
  }

  /* 20ms stabilization time */
  tk3clk_delay(20000);

  /* accumulate 256 samples at 1kHz */
  accel[0] = accel[1] = accel[2] = 0;
  gyro[0] = gyro[1] = gyro[2] = 0;
  for (i = t = 0; i < 256; i++) {
    tk3clk_delayuntil(1000 /* 1ms */);

    if (tk3sens_read(
          TK3_MPU9250, sizeof(*mpu9250_regs()),
          MPU9250_INT_STATUS, mpu9250_regs()) || ++t > 512) {
      tk3msg_log(TK3CH_USB0, "EMPU9250 self-test data acquisition failed");
      goto fail;
    }
    if (!(mpu9250_regs()->int_status & 1)) { i--; continue; }

    accel[0] +=
      mpu9250_regs()->accel_xout_h << 8 | mpu9250_regs()->accel_xout_l;
    accel[1] +=
      mpu9250_regs()->accel_yout_h << 8 | mpu9250_regs()->accel_yout_l;
    accel[2] +=
      mpu9250_regs()->accel_zout_h << 8 | mpu9250_regs()->accel_zout_l;

    gyro[0] +=
      mpu9250_regs()->gyro_xout_h << 8 | mpu9250_regs()->gyro_xout_l;
    gyro[1] +=
      mpu9250_regs()->gyro_yout_h << 8 | mpu9250_regs()->gyro_yout_l;
    gyro[2] +=
      mpu9250_regs()->gyro_zout_h << 8 | mpu9250_regs()->gyro_zout_l;
  }

  /* disable self-test */
  if (tk3sens_write(
        TK3_MPU9250, 2,
        MPU9250_GYRO_CONFIG,		0x0 /* ±250°/s, DLPF */,
        MPU9250_ACCEL_CONFIG,		0x0 /* ±2g */)) {
    tk3msg_log(TK3CH_USB0, "EMPU9250 self-test configuration failed");
    goto fail;
  }

  /* 20ms stabilization time */
  tk3clk_delay(20000);

  /* accumulate 256 samples at 1kHz */
  for (i = t = 0; i < 256; i++) {
    tk3clk_delayuntil(1000 /* 1ms */);

    if (tk3sens_read(
          TK3_MPU9250, sizeof(*mpu9250_regs()),
          MPU9250_INT_STATUS, mpu9250_regs()) || ++t > 512) {
      tk3msg_log(TK3CH_USB0, "EMPU9250 self-test data acquisition failed");
      goto fail;
    }
    if (!(mpu9250_regs()->int_status & 1)) { i--; continue; }

    accel[0] -=
      mpu9250_regs()->accel_xout_h << 8 | mpu9250_regs()->accel_xout_l;
    accel[1] -=
      mpu9250_regs()->accel_yout_h << 8 | mpu9250_regs()->accel_yout_l;
    accel[2] -=
      mpu9250_regs()->accel_zout_h << 8 | mpu9250_regs()->accel_zout_l;

    gyro[0] -=
      mpu9250_regs()->gyro_xout_h << 8 | mpu9250_regs()->gyro_xout_l;
    gyro[1] -=
      mpu9250_regs()->gyro_yout_h << 8 | mpu9250_regs()->gyro_yout_l;
    gyro[2] -=
      mpu9250_regs()->gyro_zout_h << 8 | mpu9250_regs()->gyro_zout_l;
  }

  /* read factory selftest value */
  if (tk3sens_read(TK3_MPU9250, 3, MPU9250_SELF_TEST_X_ACCEL, fst)) {
    tk3msg_log(TK3CH_USB0, "EMPU9250 cannot read self-test factory value");
    goto fail;
  }

  for (i = 0; i < 3; i++) {
    r = 2620 * powi(1.01, fst[i]-1);
    r = accel[i]/256/r;
    if (r < 0.5 || r > 1.5) {
      tk3msg_log(TK3CH_USB0, "Eaccelerometer %c axis not functional",
                 (const char []){'X', 'Y', 'Z'}[i]);
      tk3fb_on(TK3FB_SENS_ERR);
      continue;
    }

    stvalid.a[i] = 1;
  }

  if (tk3sens_read(TK3_MPU9250, 3, MPU9250_SELF_TEST_X_GYRO, fst))
      goto fail;

  for (i = 0; i < 3; i++) {
    r = 2620 * powi(1.01, fst[i]-1);
    r = gyro[i]/256/r;
    if (r < 0.5) {
      tk3msg_log(TK3CH_USB0, "Egyroscope %c axis not functional",
                 (const char []){'X', 'Y', 'Z'}[i]);
      tk3fb_on(TK3FB_SENS_ERR);
      continue;
    }

    stvalid.g[i] = 1;
  }

  return 0;

fail:
  return 1;
}


/* --- tk3sens_mpu9250_reset ----------------------------------------------- */

static int
tk3sens_mpu9250_reset()
{
  /* invalidate data */
  stvalid.a[0] = stvalid.a[1] = stvalid.a[2] = 0;
  stvalid.g[0] = stvalid.g[1] = stvalid.g[2] = 0;

  /* in case AK8963 was already configured, it needs to be powered down,
   * otherwise it hangs the whole chip during the reset - however,
   * I2C passthrough might not be configured yet, so errors are ignored. */
  tk3sens_write(TK3_AK8963, 1, 0xa, 0x0);

  /* reset */
  if (tk3sens_write(
        TK3_MPU9250, 1, MPU9250_PWR_MGMT_1,	0x80 /* H_RESET */))
    goto fail;

  tk3clk_delay(135000 /* 135ms boot time */);
  return 0;

fail:
  tk3msg_log(TK3CH_USB0, "EMPU9250 not responding");
  return 1;
}


/* --- tk3sens_mpu9250_aioread --------------------------------------------- */

void
tk3sens_mpu9250_aioread(void (*cb)(void *))
{
  tk3sens_aioread(
    TK3_MPU9250, sizeof(*mpu9250_regs()),
    MPU9250_INT_STATUS, mpu9250_regs(), cb);
}

int
tk3sens_mpu9250_data(int32_t a[3], int32_t w[3], int16_t *t)
{
  int i;

  if (!(mpu9250_regs()->int_status & 1)) return 1;

  a[0] = mpu9250_regs()->accel_xout_h << 8 | mpu9250_regs()->accel_xout_l;
  a[1] = mpu9250_regs()->accel_yout_h << 8 | mpu9250_regs()->accel_yout_l;
  a[2] = mpu9250_regs()->accel_zout_h << 8 | mpu9250_regs()->accel_zout_l;

  w[0] = mpu9250_regs()->gyro_xout_h << 8 | mpu9250_regs()->gyro_xout_l;
  w[1] = mpu9250_regs()->gyro_yout_h << 8 | mpu9250_regs()->gyro_yout_l;
  w[2] = mpu9250_regs()->gyro_zout_h << 8 | mpu9250_regs()->gyro_zout_l;

  *t = mpu9250_regs()->temp_out_h << 8 | mpu9250_regs()->temp_out_l;

  for(i = 0; i < 3; i++) {
    if (!stvalid.a[i]) a[i] = 0;
    if (!stvalid.g[i]) w[i] = 0;
  }

  return 0;
}


/* --- powi ---------------------------------------------------------------- */

/* efficient integer pow() */

static double
powi(double x, uint32_t y)
{
  double r = 1.;

  while (y) {
    if (y & 1) r *= x;
    x *= x;
    y /= 2;
  }

  return r;
}
