/*
 * Copyright (c) 2019-2021,2023-2025 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Fri Jun 14 2019
 */
#include "autoconf.h"

#include "tk3-paparazzi.h"

/* settings */
tk3setting(TK3SET_MOTORS, 6, "number of mkbl motors, 0=off", 10);
tk3setting(TK3SET_POLES, 14, "number of motor poles", 11);

/* communication state */
static struct {
  uint32_t sladdr, motors, logp;
} rx;

static struct {
  const stm32_dma_stream_t *dma;
  uint32_t len;
  uint32_t flush;
} tx;

static tk3timer(restart);
static tk3timer(clksync);

static void	mkbl_clkping(void *arg);
static void	mkbl_deinit(void);
static void	mkbl_start(void);


/* --- tk3mkbl_settings ---------------------------------------------------- */

void
tk3mkbl_settings()
{
  rx.motors = tk3set_get(TK3SET_MOTORS, 6);
}


/* --- tk3mkbl_init -------------------------------------------------------- */

static uint32_t tk3mkbl_lastinit;

void
tk3mkbl_init()
{
  if (!rx.motors) return;
  rx.sladdr = 1;
  tx.flush = 0;
  tx.len = 0;

  /* hardware bus */
  mkbl_deinit();
  tk3mkbl_hwinit();

  /* motors clock sync */
  tk3clk_settimer(&clksync, 1000000, 0, TK3EV_IDLE, mkbl_clkping, NULL);

  /* motors log rate */
  tk3mkbl_log(rx.logp);

  /* start */
  tk3mkbl_lastinit = tk3clk_us();
  tk3msg_log(TK3CH_USB0, "Nmkbl I2C bus configured");
  mkbl_start();
}

void
tk3mkbl_restart(void *arg)
{
  static uint32_t wait;
  uint32_t now;

  if (tx.flush) {
    mkbl_deinit();
    return;
  }

  if (arg == tk3mkbl_restart) {
    tk3mkbl_init();
    return;
  }

  /* exponential backoff timer */
  now = tk3clk_us();
  wait *= 2 * (now - tk3mkbl_lastinit < wait);
  if (wait < 500) wait = 500; /* 500µs min */
  else if (wait > 5000000) wait = 5000000; /* 5s max */

  /* schedule restart */
  mkbl_deinit();
  tk3fb_on(TK3FB_ESC_ERR);
  tk3clk_settimer(
    &restart, 0, wait, TK3EV_IDLE, tk3mkbl_restart, tk3mkbl_restart);
}


/* --- mkbl_deinit --------------------------------------------------------- */

static void
mkbl_deinit()
{
  tk3clk_deltimer(&clksync);
  tk3clk_deltimer(&restart);

  tk3mkbl_hwfini();
  tx.flush = 0;
  tk3fb_off(TK3FB_ESC_ERR);
}


/* --- tk3mkbl_fini -------------------------------------------------------- */

void
tk3mkbl_fini()
{
  if (!rx.motors) return;

  tk3mkbl_log(0);
  tk3mkbl_stop();
  tx.flush = 1; /* will deinit after emptying buffer */
}


/* --- tk3clk_ping --------------------------------------------------------- */

static void
mkbl_clkping(void *arg)
{
  if (tx.flush) return;

  uint32_t date = tk3clk_us();
  tk3msg_log(TK3CH_MKBL, "t%4", date);
}


/* --- tk3mkbl_stop -------------------------------------------------------- */

void
tk3mkbl_stop()
{
  tk3mkbl_send((uint8_t []){"x"}, 1);
}


/* --- tk3mkbl_spin -------------------------------------------------------- */

void
tk3mkbl_spin(const uint8_t *buffer, uint8_t len)
{
  if (!rx.motors) return;

  if (tk3rc_killswitch) {
    tk3mkbl_stop();
    return;
  }

  if (!tx.flush)
    tk3msg_log_buffer(TK3CH_MKBL, buffer, len);
}


/* --- tk3mkbl_log -------------------------------------------------------- */

void
tk3mkbl_log(uint32_t period)
{
  rx.logp = period;
  if (!tx.flush && rx.motors)
    tk3msg_log(TK3CH_MKBL, "m%4", period);
}


/* --- tk3mkbl_send -------------------------------------------------------- */

void
tk3mkbl_send(const uint8_t *buffer, uint8_t len)
{
  if (!tx.flush && rx.motors)
    tk3msg_log_buffer(TK3CH_MKBL, buffer, len);
}


/* --- mkbl_complete ------------------------------------------------------- */

void
tk3mkbl_complete()
{
  if (tx.len) {
    tk3iob_rcommit(tk3mkbl_outiob(), tx.len);
    tk3fb_flash(TK3FB_ESC_0, 1000);
  } else
    tk3fb_flash(TK3FB_ESC_0 + rx.sladdr, 1000);

  mkbl_start();
}


/* --- mkbl_start ---------------------------------------------------------- */

static void
mkbl_start()
{
  struct tk3iob_extent e;

  if (tx.len && !tx.flush) {
    /* after sending a message, check at least one slave for a message, to
     * avoid stalling slaves communication */
    tx.len = 0;
  } else {
    e = tk3iob_rbegin(tk3mkbl_outiob(), 32);
    tx.len = e.len;
  }

  if (tx.len) {
    tk3mkbl_hwsend(&e);
  } else if (tx.flush) {
    mkbl_deinit();
  } else {
    if (!--rx.sladdr) rx.sladdr = rx.motors;
    tk3mkbl_hwrecv(0x10 + rx.sladdr);
  }
}
