/*
 * Copyright (c) 2021, 2023-2025 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Fri Jan 29 2021
 */
#include "autoconf.h"

#include "tk3-paparazzi.h"

static void	tk3_cmdsignal(struct tk3iob *iob);
static void	tk3_cmd(void *arg);


/* --- main ---------------------------------------------------------------- */

static volatile int done;

int
main(void)
{
  /* hw init */
  halInit();
  osalSysEnable();

  tk3set_init();
  tk3fb_init();
  tk3ev_hwinit();
  tk3clk_init();
  tk3msg_init(tk3_cmdsignal);
  tk3usb_init(tk3pwr_on, tk3pwr_off);
  tk3rc_init();
  tk3aux_init();
  tk3pwr_hwinit();
  tk3mkbl_settings();

  /* process idle events */
  done = 0;
  do {
    if (!tk3ev_process()) __WFI();
    if (done) tk3msg_log(TK3CH_USB0, "a");

  } while(!done);

  /* reboot */
  tk3usb_fini();
  return 0;
}


/* --- tk3_cmdsignal ------------------------------------------------------- */

static void
tk3_cmdsignal(struct tk3iob *iob)
{
  static tk3event(tk3cmdev, tk3_cmd, NULL);

  tk3ev_schedule(&tk3cmdev, TK3EV_NOW);
}


/* --- tk3_cmd ------------------------------------------------------------- */

static void
tk3_cmd(void *arg)
{
  static tk3timer(imulog);
  static tk3timer(maglog);
  static tk3timer(motlog);
  static tk3timer(batlog);

  const struct tk3msg *msg;

  while ((msg = tk3msg_recv())) {
    uint8_t cmd = msg->bot[0];

    switch(cmd) {
      case 'i': /* IMU data */
      case 'c': /* magnetometer data */
      case 'b': /* battery level */
        if (msg->len == 5) {
          uint32_t p =
          ((uint32_t)msg->bot[1] << 24) |
          ((uint32_t)msg->bot[2] << 16) |
          ((uint32_t)msg->bot[3] << 8) |
          msg->bot[4];

          switch(cmd) {
            case 'i':
              tk3clk_settimer(&imulog, p, 0, TK3EV_NOW,
                              tk3sens_sendimu, (void *)msg->channel);
              break;
            case 'c':
              tk3clk_settimer(&maglog, p, 0, TK3EV_NOW,
                              tk3sens_sendmag, (void *)msg->channel);
              break;
            case 'b':
              tk3clk_settimer(
                &batlog, p, 0, TK3EV_NOW, tk3pwr_batdata, NULL);
              break;
          }
        }
        break;

      case 'g': /* start motors */
        tk3srv_start(msg->len < 2 ? -1/* all */ : msg->bot[1]);
        tk3mkbl_spin(msg->bot, msg->len);
        break;

      case 'x': /* stop motors */
        tk3srv_stop(msg->len < 2 ? -1/* all */ : msg->bot[1]);
        tk3mkbl_send(msg->bot, msg->len);
        break;

      case 'q': /* PWM array */
      case 'w': /* velocity array */
        if (msg->len >= 3) {
          int16_t data[8];
          int i;

          for(i = 0; i < (msg->len - 1)/2; i++) {
            data[i] =
              ((int16_t)msg->bot[2*i + 1] << 8) |
              ((uint16_t)msg->bot[2*i + 2]);
          }
          for(; i < 8; i++) data[i] = 0;

          cmd == 'q' ? tk3srv_throttle(data) : tk3srv_velocity(data);
        }
        tk3mkbl_spin(msg->bot, msg->len);
        break;

      case '~': /* beep */
        tk3srv_beep(&(int){1});
        tk3mkbl_send(msg->bot, msg->len);
        break;

      case 'm': /* motor data */
        if (msg->len == 5) {
          uint32_t p =
            ((uint32_t)msg->bot[1] << 24) |
            ((uint32_t)msg->bot[2] << 16) |
            ((uint32_t)msg->bot[3] << 8) |
            msg->bot[4];

          tk3clk_settimer(&motlog, p, 0, TK3EV_NOW, tk3srv_motdata, NULL);
          tk3mkbl_log(p);
        }
        break;

      case 'M': /* motor data forwarder */
        tk3msg_log_buffer(TK3CH_USB0, msg->bot, msg->len);
        break;

#ifdef TK3CH_CAN
      case '4': /* can bus forwarder */ {
        size_t l = 1 + tk3can_send(msg->bot + 1, msg->len - 1);
        if (l < msg->len)
          tk3msg_pushback(msg->channel, 1, l);
        break;
      }
#endif /* TK3CH_CAN */

      case '8': /* uart8 forwarder */ {
        size_t l = 1 + tk3aux_send(TK3AUX_8, msg->bot + 1, msg->len - 1);
        if (l < msg->len)
          tk3msg_pushback(msg->channel, 1, l);
        break;
      }

      case 't': /* configure uart */
        if (msg->len == 9) {
          uint32_t baud =
            ((uint32_t)msg->bot[2] << 24) |
            ((uint32_t)msg->bot[3] << 16) |
            ((uint32_t)msg->bot[4] << 8) |
            msg->bot[5];
          switch(msg->bot[1]) {
            case '8': tk3aux_start(TK3AUX_8, baud); break;
          }
        }
        break;

      case 'T': /* clock rate */
        tk3msg_log_buffer(TK3CH_USB0, msg->bot, msg->len);
        break;

      case '%': /* PID tuning */
        if (msg->len == 18) {
          static const float scale = 1. / (1<<16);
          uint32_t P, I, D, f;

          P = ((uint32_t)msg->bot[2] << 24) |
              ((uint32_t)msg->bot[3] << 16) |
              ((uint32_t)msg->bot[4] << 8) |
              msg->bot[5];
          I = ((uint32_t)msg->bot[6] << 24) |
              ((uint32_t)msg->bot[7] << 16) |
              ((uint32_t)msg->bot[8] << 8) |
              msg->bot[9];
          D = ((uint32_t)msg->bot[10] << 24) |
              ((uint32_t)msg->bot[11] << 16) |
              ((uint32_t)msg->bot[12] << 8) |
              msg->bot[13];
          f = ((uint32_t)msg->bot[14] << 24) |
              ((uint32_t)msg->bot[15] << 16) |
              ((uint32_t)msg->bot[16] << 8) |
              msg->bot[17];

          tk3srv_setpid(
            msg->bot[1], P * scale, I * scale, D * scale, f * scale);
        }
        break;

      case 'r': /* data resolution */
        if (msg->len == 2) {
          switch(msg->bot[1]) {
            case 'i': /* IMU */
              tk3msg_log(msg->channel, "Ri%4%4",
                         tk3set_get(TK3SET_ACCRANGE, 8),
                         tk3set_get(TK3SET_GYRRANGE, 1000));
              break;
          }
        }
        break;

      case '?': /* id */
        if (msg->len == 1) {
          tk3msg_log(msg->channel, "?" TK3_FIRMWARE);
        } else
          tk3msg_log(TK3CH_USB0, "%c", '0'+msg->bot[1]);

        break;

      case '\0':
        if (msg->state.magic) {
          tk3msg_log(msg->channel, "Nreset");
          done = 1;
        }
        break;
    }
  }
}
