/*
 * Copyright (c) 2024-2025 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Mon Dec 16 2024
 */
#include "autoconf.h"

#include "tk3-paparazzi.h"

/* LIS3MDL regs */
enum {
  LIS3MDL_WHOAMI =	0xf,

  LIS3MDL_CTL_REG1 =	0x20,
  LIS3MDL_CTL_REG2,
  LIS3MDL_CTL_REG3,
  LIS3MDL_CTL_REG4,
  LIS3MDL_CTL_REG5,

  LIS3MDL_STATUS_REG =	0x27
};

/* sensor data */
struct __attribute__((__packed__)) lis3mdl_regs {
  uint8_t	status;

  uint8_t	out_x_l; int8_t out_x_h;
  uint8_t	out_y_l; int8_t out_y_h;
  uint8_t	out_z_l; int8_t out_z_h;
};

static __attribute__ ((const)) struct lis3mdl_regs *
lis3mdl_regs(void) { return tk3_padmem(struct lis3mdl_regs, TK3_LIS3MDL_MEM); }

static uint16_t lis3mdl_res;
static struct { int m[3]; } stvalid;

static int	tk3sens_lis3mdl_config(void);
static int	tk3sens_lis3mdl_selftest(void);
static int	tk3sens_lis3mdl_reset(void);
static int	tk3sens_lis3mdl_rdata(int32_t m[3]);


/* --- tk3sens_lis3mld_probe ----------------------------------------------- */

int
tk3sens_lis3mdl_probe()
{
  uint8_t *v = tk3_padmem(uint8_t, TK3_LIS3MDL_MEM);

  /* check LIS3MLD ID */
  if (tk3sens_read(TK3_LIS3MDL, 1, LIS3MDL_WHOAMI, v)) return 0;
  if (*v != 0x3d) return 0;

  return 1;
}


/* --- tk3sens_lis3mdl_init ------------------------------------------------ */

int
tk3sens_lis3mdl_init()
{
  if (tk3sens_lis3mdl_reset()) return 1;
  if (tk3sens_lis3mdl_selftest()) return 1;
  if (tk3sens_lis3mdl_config()) return 1;

  tk3msg_log(TK3CH_USB0, "NLIS3MDL configured");
  return 0;
}


/* --- tk3sens_lis3mdl_config ---------------------------------------------- */

static int
tk3sens_lis3mdl_config()
{
  /* continuous mode 560Hz, 4G */
  if (tk3sens_write(
        TK3_LIS3MDL, 5,
        LIS3MDL_CTL_REG1, 0x2	/* X,Y low power, FAST_ODR */,
        LIS3MDL_CTL_REG4, 0	/* Z low power */,
        LIS3MDL_CTL_REG2, 0	/* FS 4G */,
        LIS3MDL_CTL_REG5, 0x40	/* block data update */,
        LIS3MDL_CTL_REG3, 0	/* continuous */)) goto fail;
  lis3mdl_res = 6842;

  return 0;

fail:
  tk3msg_log(TK3CH_USB0, "Emagnetometer configuration failed");
  return 1;
}


/* --- tk3sens_lis3mdl_selftest -------------------------------------------- */

static int
tk3sens_lis3mdl_selftest()
{
  int32_t m[3], st[3], nost[3];
  uint8_t i, t;

  /* continuous mode 80Hz, 12G */
  if (tk3sens_write(
        TK3_LIS3MDL, 5,
        LIS3MDL_CTL_REG1, 0x1c /* X, Y low power, 80Hz */,
        LIS3MDL_CTL_REG4, 0 /* Z low power */,
        LIS3MDL_CTL_REG2, 0x40 /* FS 12G */,
        LIS3MDL_CTL_REG5, 0x40 /* block data update */,
        LIS3MDL_CTL_REG3, 0 /* continuous */)) goto fail;
  lis3mdl_res = 2281;
  tk3clk_delay(40000); /* per AN4602 */

  /* wait for data ready and discard first sample */
  t = 0;
  do {
    if (t++ > 250) goto timeout;
    tk3clk_delayuntil(1000 /* 1ms */);
    if (tk3sens_read(
          TK3_LIS3MDL, 1, LIS3MDL_STATUS_REG, lis3mdl_regs())) goto fail;
  } while(!(lis3mdl_regs()->status & 0x8));

  /* average 8 samples */
  nost[0] = nost[1] = nost[2] = 0;
  for (i = 0; i < 8; i++) {
    t = 0;
    do {
      if (t++ > 250) goto timeout;
      tk3clk_delayuntil(1000 /* 1ms */);
      if (tk3sens_read(
            TK3_LIS3MDL, sizeof(*lis3mdl_regs()),
            LIS3MDL_STATUS_REG | 0x80, lis3mdl_regs())) goto fail;
    } while(tk3sens_lis3mdl_rdata(m));

    nost[0] += m[0]; nost[1] += m[1]; nost[2] += m[2];
  }
  nost[0] /= 8; nost[1] /= 8; nost[2] /= 8;


  /* self-test enable */
  if (tk3sens_write(
        TK3_LIS3MDL, 1,
        LIS3MDL_CTL_REG1, 0x1d /* X, Y low power, 80Hz, self-test */))
    goto fail;
  tk3clk_delay(60000); /* per AN4602 */

  /* wait for data ready and discard first sample */
  t = 0;
  do {
    if (t++ > 250) goto timeout;
    tk3clk_delayuntil(1000 /* 1ms */);
    if (tk3sens_read(
          TK3_LIS3MDL, 1, LIS3MDL_STATUS_REG, lis3mdl_regs())) goto fail;
  } while(!(lis3mdl_regs()->status & 0x8));

  /* average 8 samples */
  st[0] = st[1] = st[2] = 0;
  for (i = 0; i < 8; i++) {
    t = 0;
    do {
      if (t++ > 250) goto timeout;
      tk3clk_delayuntil(1000 /* 1ms */);
      if (tk3sens_read(
            TK3_LIS3MDL, sizeof(*lis3mdl_regs()),
            LIS3MDL_STATUS_REG | 0x80, lis3mdl_regs())) goto fail;
    } while(tk3sens_lis3mdl_rdata(m));

    st[0] += m[0]; st[1] += m[1]; st[2] += m[2];
  }
  st[0] /= 8; st[1] /= 8; st[2] /= 8;

  /* self-test disable */
  if (tk3sens_write(
        TK3_LIS3MDL, 2,
        LIS3MDL_CTL_REG1, 0x1c, /* X, Y low power, 80Hz */
        LIS3MDL_CTL_REG3, 0x3 /* power down */)) goto fail;


  /* check */
  for(int i = 0; i < 3; i++) {
    m[i] = st[i] - nost[i];
    m[i] = m[i] < 0 ? -m[i] : m[i];
  }

  if (m[0] < 10000 || m[0] > 30000) {
    tk3msg_log(TK3CH_USB0, "Emagnetometer %c axis not functional", 'X');
    tk3fb_on(TK3FB_SENS_ERR);
  } else
    stvalid.m[0] = 1;

  if (m[1] < 10000 || m[1] > 30000) {
    tk3msg_log(TK3CH_USB0, "Emagnetometer %c axis not functional", 'Y');
    tk3fb_on(TK3FB_SENS_ERR);
  } else
    stvalid.m[1] = 1;

  if (m[2] < 1000 || m[2] > 10000) {
    tk3msg_log(TK3CH_USB0, "Emagnetometer %c axis not functional", 'Z');
    tk3fb_on(TK3FB_SENS_ERR);
  } else
    stvalid.m[2] = 1;

  /* done */
  return 0;

timeout:
  tk3msg_log(TK3CH_USB0, "Emagnetometer timeout");
  return 1;

fail:
  tk3msg_log(TK3CH_USB0, "Emagnetometer self-test failed");
  return 1;
}


/* --- tk3sens_lis3mdl_reset ------------------------------------------------ */

static int
tk3sens_lis3mdl_reset()
{
  /* invalidate data */
  stvalid.m[0] = stvalid.m[1] = stvalid.m[2] = 0;

  /* reset */
  if (tk3sens_write(TK3_LIS3MDL, 2,
                    LIS3MDL_CTL_REG3, 0x03 /* power down */,
                    LIS3MDL_CTL_REG2, 0x04 /* SOFT_RST */))
    goto fail;
  tk3clk_delay(5);
  if (tk3sens_write(TK3_LIS3MDL, 1,
                    LIS3MDL_CTL_REG2, 0x08 /* reboot */))
    goto fail;

  /* 20ms POR */
  tk3clk_delay(20000);
  return 0;

fail:
  tk3msg_log(TK3CH_USB0, "ELIS3MDL not responding");
  return 1;
}


/* --- tk3sens_lis3mdl_aioread --------------------------------------------- */

void
tk3sens_lis3mdl_aioread(void (*cb)(void *))
{
  tk3sens_aioread(
    TK3_LIS3MDL, sizeof(*lis3mdl_regs()),
    LIS3MDL_STATUS_REG | 0x80, lis3mdl_regs(), cb);
}


/* --- tk3sens_lis3mdl_data ------------------------------------------------ */

static int
tk3sens_lis3mdl_rdata(int32_t m[3])
{
  if (!(lis3mdl_regs()->status & 0x8)) return 1;

  m[0] = lis3mdl_regs()->out_x_h << 8 | lis3mdl_regs()->out_x_l;
  m[1] = lis3mdl_regs()->out_y_h << 8 | lis3mdl_regs()->out_y_l;
  m[2] = lis3mdl_regs()->out_z_h << 8 | lis3mdl_regs()->out_z_l;

  for(int i = 0; i < 3; i++)
    m[i] =  m[i] * 10000 / lis3mdl_res;
  return 0;
}

int
tk3sens_lis3mdl_data(int32_t m[3])
{
  int32_t rm[3];

  if (tk3sens_lis3mdl_rdata(rm)) return 1;

  m[0] = stvalid.m[0] ? -rm[1] : 0;
  m[1] = stvalid.m[1] ? -rm[0] : 0;
  m[2] = stvalid.m[2] ? -rm[2] : 0;
  return 0;
}
