/*
 * Copyright (c) 2019,2021,2024 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Fri Jun 21 2019
 */
#include <stddef.h>
#include <string.h>

#include "tk3-paparazzi.h"


/* --- tk3iob_putc --------------------------------------------------------- */

size_t
tk3iob_putc(struct tk3iob *buffer, uint8_t c)
{
  uint8_t *w, *wp;

  w = buffer->w;

  wp = w + 1;
  if (wp >= buffer->top) wp = buffer->bot;
  if (wp == buffer->r) return 0;

  *w = c;
  buffer->w = wp;

  /* signal event */
  if (c == '$' && buffer->cb) buffer->cb(buffer);

  return 1;
}


/* --- tk3iob_write -------------------------------------------------------- */

size_t
tk3iob_write(struct tk3iob *buffer, const uint8_t *data, size_t len)
{
  const uint8_t *p, *t;
  size_t n;

  p = data;
  do {
    /* available space */
    with_syslock() {
      if (buffer->r > buffer->w)
        t = buffer->r - 1;
      else if (buffer->r == buffer->bot)
        t = buffer->top - 1;
      else
        t = buffer->top;
    }

    n = t - buffer->w;
    if (n == 0) break;
    if (n > len) n = len;

    /* copy */
    memcpy(buffer->w, p, n);

    /* update indexes */
    with_syslock() {
      buffer->w += n;
      if (buffer->w >= buffer->top) buffer->w = buffer->bot;
    }

    /* next segment */
    len -= n;
    p += n;
  } while (len && t == buffer->top);

  /* signal event */
  if (buffer->cb) buffer->cb(buffer);

  return p - data;
}


/* --- tk3iob_wbegin ------------------------------------------------------- */

struct tk3iob_extent
tk3iob_wbegin(struct tk3iob *buffer, size_t len)
{
  size_t b, t;

  /* available space */
  with_syslock() {
    if (buffer->r > buffer->w) {
      b = 0;
      t = buffer->r - buffer->w - 1;
    } else if (buffer->r > buffer->bot) {
      b = buffer->r - buffer->bot - 1;
      t = buffer->top - buffer->w;
    } else {
      b = 0;
      t = buffer->top - buffer->w - 1;
    }
  }

  if (b > buffer->end - buffer->top) b = buffer->end - buffer->top;
  if (len > b + t) len = b + t;

  /* return write extent */
  return (struct tk3iob_extent){ .w = buffer->w, .len = len };
}


/* --- tk3iob_wcommit ------------------------------------------------------ */

void
tk3iob_wcommit(struct tk3iob *buffer, size_t len)
{
  ptrdiff_t m;

  /* unmirror */
  m = buffer->w + len - buffer->top;
  if (m > 0) memcpy(buffer->bot, buffer->top, m);

  /* update indexes */
  if (m >= 0)
    with_syslock() { buffer->w = buffer->bot + m; }
  else
    with_syslock() { buffer->w += len; }

  /* signal event */
  if (buffer->cb) buffer->cb(buffer);
}


/* --- tk3iob_getc --------------------------------------------------------- */

size_t
tk3iob_getc(struct tk3iob *buffer, uint8_t *c)
{
  const uint8_t *r = buffer->r;

  if (r == buffer->w) return 0;

  *c = *r++;
  if (r >= buffer->top) r = buffer->bot;
  buffer->r = r;

  return 1;
}


/* --- tk3iob_read --------------------------------------------------------- */

size_t
tk3iob_read(struct tk3iob *buffer, uint8_t *data, size_t len)
{
  const uint8_t *t;
  uint8_t *p;
  size_t n;

  p = data;
  do {
    /* available data */
    with_syslock() {
      t = (buffer->w >= buffer->r) ? buffer->w : buffer->top;
    }

    n = t - buffer->r;
    if (n == 0) break;
    if (n > len) n = len;

    /* copy */
    memcpy(p, buffer->r, n);
    with_syslock() {
      buffer->r += n;
      if (buffer->r >= buffer->top) buffer->r = buffer->bot;
    }

    /* next segment */
    len -= n;
    p += n;
  } while (len && t == buffer->top);

  return p - data;
}


/* --- tk3iob_rbegin ------------------------------------------------------- */

struct tk3iob_extent
tk3iob_rbegin(struct tk3iob *buffer, size_t len)
{
  const uint8_t *r = buffer->r;
  uint8_t *w = buffer->w;
  size_t b, t;

  /* available data */
  if (r <= w) {
    b = 0;
    t = w - r;
  } else {
    b = w - buffer->bot;
    t = buffer->top - r;
  }

  if (b > buffer->end - buffer->top) b = buffer->end - buffer->top;

  if (t >= len) { b = 0; t = len; }
  else if (b + t > len) { b = len - t; }
  else { len = b + t; }

  /* mirror */
  if (b > 0) memcpy(buffer->top, buffer->bot, b);

  /* return read extent */
  return (struct tk3iob_extent){ .r = r, .len = len };
}


/* --- tk3iob_rcommit ------------------------------------------------------ */

void
tk3iob_rcommit(struct tk3iob *buffer, size_t len)
{
  const uint8_t *r = buffer->r;

  /* update indexes */
  r += len;
  if (r >= buffer->top)
    r -= buffer->top - buffer->bot;

  buffer->r = r;
}
