/*
 * Copyright (c) 2025 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Fri Jan 10 2025
 */
#include "autoconf.h"

#include "tk3-paparazzi.h"

/* ICM42688 regs */
enum {
  /* bank 0 */
  ICM42688_DEVICE_CONFIG =	0x11,

  ICM42688_TEMP_DATA1 =		0x1d,

  ICM42688_PWR_MGMT0 =		0x4e,
  ICM42688_GYRO_CONFIG0,
  ICM42688_ACCEL_CONFIG0,

  ICM42688_GYRO_ACCEL_CONFIG0 =	0x52,

  ICM42688_SELF_TEST_CONFIG =	0x70,

  ICM42688_WHOAMI =		0x75,
  ICM42688_REG_BANK_SEL,

  /* bank 1 */
  ICM42688_XG_ST_DATA =		0x5f,

  /* bank 2 */
  ICM42688_XA_ST_DATA =		0x3b
};

/* sensor data */
struct __attribute__((packed)) icm42688_regs {
  int8_t	temp_h;		uint8_t temp_l;

  int8_t	accel_xout_h;	uint8_t accel_xout_l;
  int8_t	accel_yout_h;	uint8_t accel_yout_l;
  int8_t	accel_zout_h;	uint8_t accel_zout_l;

  int8_t	gyro_xout_h;	uint8_t gyro_xout_l;
  int8_t	gyro_yout_h;	uint8_t gyro_yout_l;
  int8_t	gyro_zout_h;	uint8_t gyro_zout_l;

  int8_t	tmst_fsync_h;	uint8_t tmst_fsync_l;

  uint8_t	int_status;
};

static __attribute__ ((const)) struct icm42688_regs *
icm42688_regs(void) {
  return tk3_padmem(struct icm42688_regs, TK3_ICM42688_MEM);
}

static struct { int a[3], g[3]; } stvalid;

static int	tk3sens_icm42688_config(void);
static int	tk3sens_icm42688_selftest(void);
static int	tk3sens_icm42688_reset(void);
static int	tk3sens_icm42688_rdata(int32_t a[3], int32_t w[3], int16_t *t);
static double	powi(double x, uint32_t y);


/* --- tk3sens_icm42688_probe ----------------------------------------------- */

int
tk3sens_icm42688_probe()
{
  uint8_t * const v = tk3_padmem(uint8_t, TK3_ICM42688_MEM);


  /* check ICM42688 id */
  if (tk3sens_read(TK3_ICM42688, 1, ICM42688_WHOAMI | 0x80, v)) return 0;
  if (*v != 0x47) return 0;

  return 1;
}


/* --- tk3sens_icm42688_init ------------------------------------------------ */

int
tk3sens_icm42688_init()
{
  char *arange = "NaN", *grange = "NaN";

  if (tk3sens_icm42688_reset()) return 1;
  if (tk3sens_icm42688_selftest()) return 1;
  if (tk3sens_icm42688_config()) return 1;

  switch(tk3set_get(TK3SET_ACCRANGE, 8)) {
    case  2: arange = "2"; break;
    case  4: arange = "4"; break;
    case  8: arange = "8"; break;
    case 16: arange = "16"; break;
  }
  switch(tk3set_get(TK3SET_GYRRANGE, 1000)) {
    case  250: grange = "250"; break;
    case  500: grange = "500"; break;
    case 1000: grange = "1000"; break;
    case 2000: grange = "2000"; break;
  }

  tk3msg_log(TK3CH_USB0, "NICM42688 configured, ±%sg, ±%s°/s", arange, grange);
  return 0;
}


/* --- tk3sens_icm42688_config ---------------------------------------------- */

static int
tk3sens_icm42688_config()
{
  int arange, grange;

  /* bank 0 */
  if (tk3sens_write(TK3_ICM42688, 1, ICM42688_REG_BANK_SEL, 0))
    goto fail;

  /* power */
  if (tk3sens_write(
        TK3_ICM42688, 1,
        ICM42688_PWR_MGMT0,		0xf /* low-noise gyro, accel */))
    goto fail;

  /* sensors */
  arange = tk3set_get(TK3SET_ACCRANGE, 8);
  switch(arange) {
    case  2: arange = 3; break;
    case  4: arange = 2; break;
    case  8: arange = 1; break;
    case 16: arange = 0; break;
    default:
      tk3msg_log(TK3CH_USB0, "EICM42688 bad accelerometer range setting");
      goto fail;
  }

  grange = tk3set_get(TK3SET_GYRRANGE, 1000);
  switch(grange) {
    case  250: grange = 3; break;
    case  500: grange = 2; break;
    case 1000: grange = 1; break;
    case 2000: grange = 0; break;
    default:
      tk3msg_log(TK3CH_USB0, "EICM42688 bad gyroscope range setting");
      goto fail;
  }

  if (tk3sens_write(
        TK3_ICM42688, 3,
        ICM42688_GYRO_CONFIG0,	(grange << 5) | 5 /* full scale, ODR 2kHz */,
        ICM42688_ACCEL_CONFIG0,	(arange << 5) | 5 /* full scale, ODR 2kHz */,
        ICM42688_GYRO_ACCEL_CONFIG0,	0x11 /* LP BW 500Hz */))
    goto fail;

  return 0;

fail:
  tk3msg_log(TK3CH_USB0, "EICM42688 configuration failed");
  return 1;
}


/* --- tk3sens_icm42688_selftest -------------------------------------------- */

static int
tk3sens_icm42688_selftest()
{
  int32_t accel[3], gyro[3], ra[3], rg[3];
  uint8_t (*fst)[3] = tk3_padmem(uint8_t [3], TK3_ICM42688_MEM);
  int16_t temp;
  double r;
  int i, t;

  /* bank 0 */
  if (tk3sens_write(TK3_ICM42688, 1, ICM42688_REG_BANK_SEL, 0))
    goto fail;

  /* enable self-test */
  if (tk3sens_write(
        TK3_ICM42688, 5,
        ICM42688_PWR_MGMT0,		0xf /* low-noise gyro, accel */,
        ICM42688_GYRO_CONFIG0,		(3 << 5) | 6 /* ±250⁰/s, ODR 1kHz */,
        ICM42688_ACCEL_CONFIG0,		(3 << 5) | 6 /* ±2g, ODR 1kHz */,
        ICM42688_GYRO_ACCEL_CONFIG0,	0x44 /* BW ODR/10 */,
        ICM42688_SELF_TEST_CONFIG,	0x7f /* enable self-test */))
    goto fail;

  /* 20ms stabilization time */
  tk3clk_delay(20000);

  /* accumulate 256 samples at 1kHz */
  accel[0] = accel[1] = accel[2] = 0;
  gyro[0] = gyro[1] = gyro[2] = 0;
  for (i = t = 0; i < 256; i++) {
    tk3clk_delayuntil(1000 /* 1ms */);

    if (tk3sens_read(
          TK3_ICM42688, sizeof(*icm42688_regs()),
          ICM42688_TEMP_DATA1 | 0x80, icm42688_regs()) || ++t > 512)
      goto timeout;
    if (tk3sens_icm42688_rdata(ra, rg, &temp)) { i--; continue; }

    accel[0] += ra[0];	gyro[0] += rg[0];
    accel[1] += ra[1];	gyro[1] += rg[1];
    accel[2] += ra[2];	gyro[2] += rg[2];
  }

  /* disable self-test */
  if (tk3sens_write(TK3_ICM42688, 1, ICM42688_SELF_TEST_CONFIG, 0))
    goto fail;

  /* 20ms stabilization time */
  tk3clk_delay(20000);

  /* accumulate 256 samples at 1kHz */
  for (i = t = 0; i < 256; i++) {
    tk3clk_delayuntil(1000 /* 1ms */);

    if (tk3sens_read(
          TK3_ICM42688, sizeof(*icm42688_regs()),
          ICM42688_TEMP_DATA1 | 0x80, icm42688_regs()) || ++t > 512)
      goto timeout;
    if (tk3sens_icm42688_rdata(ra, rg, &temp)) { i--; continue; }

    accel[0] -= ra[0];	gyro[0] -= rg[0];
    accel[1] -= ra[1];	gyro[1] -= rg[1];
    accel[2] -= ra[2];	gyro[2] -= rg[2];
  }

  /* read factory selftest value */
  if (tk3sens_write(TK3_ICM42688, 1, ICM42688_REG_BANK_SEL, 2))
    goto fail;
  if (tk3sens_read(TK3_ICM42688, 3, ICM42688_XA_ST_DATA | 0x80, fst))
    goto fail;

  for (i = 0; i < 3; i++) {
    r = 2620. * powi(1.01, (*fst)[i]-1);
    r = accel[i]/256/r;
    if (r < 0.5 || r > 1.5) {
      tk3msg_log(TK3CH_USB0, "Eaccelerometer %c axis not functional",
                 (const char []){'X', 'Y', 'Z'}[i]);
      tk3fb_on(TK3FB_SENS_ERR);
      continue;
    }

    stvalid.a[i] = 1;
  }

  if (tk3sens_write(TK3_ICM42688, 1, ICM42688_REG_BANK_SEL, 1))
    goto fail;
  if (tk3sens_read(TK3_ICM42688, 3, ICM42688_XG_ST_DATA | 0x80, fst))
    goto fail;

  for (i = 0; i < 3; i++) {
    r = 2620. * powi(1.01, (*fst)[i]-1);
    r = gyro[i]/256/r;
    if (r < 0.5) {
      tk3msg_log(TK3CH_USB0, "Egyroscope %c axis not functional",
                 (const char []){'X', 'Y', 'Z'}[i]);
      tk3fb_on(TK3FB_SENS_ERR);
      continue;
    }

    stvalid.g[i] = 1;
  }

  return 0;

timeout:
  tk3msg_log(TK3CH_USB0, "EICM42688 self-test timeout");
  return 1;
fail:
  tk3msg_log(TK3CH_USB0, "EICM42688 self-test configuration failed");
  return 1;
}


/* --- tk3sens_icm42688_reset ----------------------------------------------- */

static int
tk3sens_icm42688_reset()
{
  /* invalidate data */
  stvalid.a[0] = stvalid.a[1] = stvalid.a[2] = 0;
  stvalid.g[0] = stvalid.g[1] = stvalid.g[2] = 0;

  /* bank 0 */
  if (tk3sens_write(TK3_ICM42688, 1, ICM42688_REG_BANK_SEL, 0))
    goto fail;

  /* reset */
  if (tk3sens_write(
        TK3_ICM42688, 1, ICM42688_DEVICE_CONFIG, 0x1 /* SOFT_RESET_CONFIG */))
    goto fail;

  tk3clk_delay(1000 /* 1ms */);
  return 0;

fail:
  tk3msg_log(TK3CH_USB0, "EICM42688 not responding");
  return 1;
}


/* --- tk3sens_icm42688_aioread --------------------------------------------- */

void
tk3sens_icm42688_aioread(void (*cb)(void *))
{
  tk3sens_aioread(
    TK3_ICM42688, sizeof(*icm42688_regs()),
    ICM42688_TEMP_DATA1 | 0x80, icm42688_regs(), cb);
}


/* --- tk3sens_icm42688_data ----------------------------------------------- */

static int
tk3sens_icm42688_rdata(int32_t a[3], int32_t w[3], int16_t *t)
{
  if (!(icm42688_regs()->int_status & 8)) return 1;

  a[0] = icm42688_regs()->accel_xout_h << 8 | icm42688_regs()->accel_xout_l;
  a[1] = icm42688_regs()->accel_yout_h << 8 | icm42688_regs()->accel_yout_l;
  a[2] = icm42688_regs()->accel_zout_h << 8 | icm42688_regs()->accel_zout_l;

  w[0] = icm42688_regs()->gyro_xout_h << 8 | icm42688_regs()->gyro_xout_l;
  w[1] = icm42688_regs()->gyro_yout_h << 8 | icm42688_regs()->gyro_yout_l;
  w[2] = icm42688_regs()->gyro_zout_h << 8 | icm42688_regs()->gyro_zout_l;

  *t = icm42688_regs()->temp_h << 8 | icm42688_regs()->temp_l;

  return 0;
}

int
tk3sens_icm42688_data(int32_t a[3], int32_t w[3], int16_t *t)
{
  int32_t ra[3], rw[3];

  if (tk3sens_icm42688_rdata(ra, rw, t)) return 1;

  a[0] = stvalid.a[0] ?  ra[1] : 0;
  a[1] = stvalid.a[1] ?  ra[0] : 0;
  a[2] = stvalid.a[2] ? -ra[2] : 0;

  w[0] = stvalid.g[0] ?  rw[1] : 0;
  w[1] = stvalid.g[1] ?  rw[0] : 0;
  w[2] = stvalid.g[2] ? -rw[2] : 0;

  return 0;
}


/* --- powi ---------------------------------------------------------------- */

/* efficient integer pow() */

static double
powi(double x, uint32_t y)
{
  double r = 1.;

  while (y) {
    if (y & 1) r *= x;
    x *= x;
    y /= 2;
  }

  return r;
}
