/*
 * Copyright (c) 2019-2021,2024 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Thu Aug  1 2019
 */
#include "autoconf.h"

#include "tk3-paparazzi.h"

static struct {
  struct tk3event *first;
  struct tk3event **last;
} evlist = { .first = NULL, .last = &evlist.first };


/* --- tk3ev_active -------------------------------------------------------- */

int
tk3ev_active(struct tk3event *event)
{
  with_syslock() {
    return event->cb || event->pri != TK3EV_INACTIVE;
  }
  /* NOTREACHED */
  return 0;
}


/* --- tk3ev_set ----------------------------------------------------------- */

void
tk3ev_set(struct tk3event *event, void (*cb)(void *arg), void *arg)
{
  with_syslock() {
    event->cb = cb;
    event->arg = arg;
  }
}


/* --- tk3ev_schedule ------------------------------------------------------ */

void
tk3ev_schedule(struct tk3event *event, enum tk3evpri pri)
{
  if (!event->cb) return;
  if (pri == TK3EV_INACTIVE) return;

  with_syslock() {
    if (event->pri == TK3EV_INACTIVE) {
      /* insert last */
      event->next = NULL;
      event->prev = evlist.last;
      *evlist.last = event;
      evlist.last = &event->next;
    }

    event->pri = pri;
  }

  /* trigger */
  tk3ev_signal();
}


/* --- tk3ev_cancel -------------------------------------------------------- */

void
tk3ev_cancel(struct tk3event *event)
{
  with_syslock() {
    if (event->pri == TK3EV_INACTIVE) return;

    /* remove from list */
    if (event->next)
      event->next->prev = event->prev;
    else
      evlist.last = event->prev;
    *event->prev = event->next;

    event->pri = TK3EV_INACTIVE;
  }
}


/* --- tk3ev_nextevent ----------------------------------------------------- */

int
tk3ev_nextevent(enum tk3evpri pri)
{
  struct tk3event *event, *next;
  void (*cb)(void *arg);
  void *arg;

  cb = NULL;
  with_syslock() {
    for(event = evlist.first; event; event = next) {
      next = event->next;
      if (event->pri != pri) continue;

      /* cache callback */
      cb = event->cb;
      arg = event->arg;

      /* remove from list */
      if (next)
        next->prev = event->prev;
      else
        evlist.last = event->prev;
      *event->prev = next;

      /* inactivate */
      event->pri = TK3EV_INACTIVE;

      /* invoke */
      break;
    }
  }
  if (!event) return 0;

  if (cb) cb(arg);
  return 1;
}
