/*
 * Copyright (c) 2019-2021,2024 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Sun Aug 18 2019
 */
#include "autoconf.h"

#include "tk3-paparazzi.h"

static struct {
  struct tk3timer *first;
} tmlist = { .first = NULL };

static tk3timer(cpuled);


static void	tk3clk_cpuled(void *arg);


/* --- tk3clk_init --------------------------------------------------------- */

int
tk3clk_init()
{
  int s;

  s = tk3clk_hwinit();
  if (s) return s;

  tk3clk_settimer(&cpuled, 1000000, 0, TK3EV_NOW, tk3clk_cpuled, NULL);
  return 0;
}


/* --- tk3clk_cpuled ------------------------------------------------------- */

static void
tk3clk_cpuled(void *arg)
{
  uint32_t cpu;

  cpu = tk3clk_cpucounter();
  tk3fb_flash(TK3FB_IDLE, 1000000 - cpu);
}


/* --- tk3clk_delay -------------------------------------------------------- */

void
tk3clk_delay(uint32_t delay)
{
  uint32_t deadline;

  deadline = tk3clk_us() + delay;
  while ((int32_t)(deadline - tk3clk_us()) > 0)
    /* wait */;
}


/* --- tk3clk_delayuntil --------------------------------------------------- */

void
tk3clk_delayuntil(uint32_t period)
{
  uint32_t deadline;

  deadline = (tk3clk_us() + period - 1) / period * period;
  while ((int32_t)(deadline - tk3clk_us()) > 0)
    /* wait */;
}


/* --- tk3clk_settimer ----------------------------------------------------- */

void
tk3clk_settimer(struct tk3timer *timer, uint32_t interval, uint32_t delay,
                enum tk3evpri pri, void (*cb)(void *arg), void *arg)
{
  uint32_t deadline;

  if (!cb) return;
  if (pri == TK3EV_INACTIVE) return;

  /* compute next deadline */
  deadline = tk3clk_us();
  if (delay)
    deadline += delay;
  else if (interval)
    deadline = (deadline + interval - 1) / interval * interval;
  else {
    tk3clk_deltimer(timer);
    return;
  }

  /* update timings and callback */
  with_syslock() {
    if (timer->pri == TK3EV_INACTIVE) {
      /* insert first */
      timer->next = tmlist.first;
      if (tmlist.first)
        tmlist.first->prev = &timer->next;
      tmlist.first = timer;
      timer->prev = &tmlist.first;
    }

    timer->pri = pri;
    timer->deadline = deadline;
    timer->interval = interval;
    tk3ev_cancel(&timer->event);
    tk3ev_set(&timer->event, cb, arg);
  }

  /* force alarm update */
  tk3clk_signal();
}


/* --- tk3clk_deltimer ----------------------------------------------------- */

void
tk3clk_deltimer(struct tk3timer *timer)
{
  with_syslock() {
    tk3ev_cancel(&timer->event);
    if (timer->pri == TK3EV_INACTIVE) return;

    /* remove from list */
    if (timer->next)
      timer->next->prev = timer->prev;
    *timer->prev = timer->next;

    timer->pri = TK3EV_INACTIVE;
  }
}


/* --- tk3clk_alarm -------------------------------------------------------- */

uint32_t
tk3clk_alarm(uint32_t now)
{
  uint32_t deadline = now - 0x1000; /* latest next deadline */
  struct tk3timer *timer, *next;
  uint32_t d;

  with_syslock() {
    for(timer = tmlist.first; timer; timer = next) {
      next = timer->next;

      /* deadline is in the future if it more than 0x1000 µs late */
      d = timer->deadline - now;
      if (d - 1 < 0xfffff000) {
        if (d < deadline - now)
          deadline = timer->deadline;
        continue;
      }

      /* timer expired: schedule callback */
      tk3ev_schedule(&timer->event, timer->pri);

      /* update deadline or cancel */
      if (timer->interval) {
        timer->deadline += timer->interval;
        if (timer->deadline - now < deadline - now)
          deadline = timer->deadline;
      } else {
        if (next)
          next->prev = timer->prev;
        *timer->prev = next;

        timer->pri = TK3EV_INACTIVE;
      }
    }
  }

  return deadline;
}
