/*
 * Copyright (c) 2021,2024 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Mon Jun 21 2021
 */
#include "autoconf.h"

#include <string.h>

#include "tk3-paparazzi.h"

#ifndef TK3_CAND
# error "need TK3_CAND"
#endif
#ifndef TK3CH_CAN
# error "need TK3CH_CAN"
#endif

static void	tk3can_recv(CANDriver *can, uint32_t flags);
static void	tk3can_sendcb(CANDriver *can, uint32_t flags);
static void	tk3can_error(CANDriver *can, uint32_t flags);

static inline uint8_t	nibtohex(uint8_t x);
static inline uint8_t	hextonib(uint8_t x);


/* --- tk3can_init --------------------------------------------------------- */

void
tk3can_init()
{
  /* configure Rx/Tx lines */
  palSetPadMode(GPIOD, 0, PAL_MODE_ALTERNATE(9));
  palSetPadMode(GPIOD, 1, PAL_MODE_ALTERNATE(9));

  TK3_CAND.txempty_cb = tk3can_sendcb;
  TK3_CAND.rxfull_cb = tk3can_recv;
  TK3_CAND.error_cb = tk3can_error;

  canStart(&TK3_CAND, &(const CANConfig){
      .mcr = CAN_MCR_ABOM | CAN_MCR_AWUM | CAN_MCR_TXFP,
      .btr =
        CAN_BTR_SJW(0) | CAN_BTR_TS2(0) | CAN_BTR_TS1(6) | CAN_BTR_BRP(2)
  });
}


/* --- tk3can_fini --------------------------------------------------------- */

void
tk3can_fini()
{
  /* wait for all tx mailboxes empty */
  while ((CAN1->TSR & CAN_TSR_TME) != CAN_TSR_TME)
    __WFI();

  canStop(&CAND1);
}


/* --- tk3can_error -------------------------------------------------------- */

static void
tk3can_error(CANDriver *can, uint32_t flags)
{
  tk3msg_log(TK3CH_CAN, "Ecan bus error");
}


/* --- tk3can_send --------------------------------------------------------- */

size_t
tk3can_send(const uint8_t *buf, size_t len)
{
  static uint8_t fifo[64], *end = fifo;

  uint8_t *p, *s, *n;

  /* move data to fifo */
  if (end - fifo + len > sizeof(fifo))
    len = sizeof(fifo) - (end - fifo);
  if (len) {
    memcpy(end, buf, len);
    end += len;
  }

  /* empty fifo */
  for(s = p = n = fifo; n < end; s = p = n) {
    /* search for end of packet, quit if not found */
    for(; n < end; n++) if (*n == '\r' || *n == '\a') break;
    if (n == end) break;

    /* skip packet with \a (error) */
    if (*n++ == '\a') continue;

    /* handle message */
    switch (*p++) {
      case 'O': tk3can_init(); break;
      case 'C': tk3can_fini(); break;
      case 'S': /* to be implemented */; break;

      case 't': {
        CANTxFrame f = { .RTR = 0, .IDE = 0 };
        uint8_t i;

        /* id */
        f.SID = hextonib(*p++) << 8;
        f.SID |= hextonib(*p++) << 4;
        f.SID |= hextonib(*p++);

        /* len */
        f.DLC = hextonib(*p++);

        /* data */
        for(i = 0; i < f.DLC; i++) {
          f.data8[i] = hextonib(*p++) << 4;
          f.data8[i] |= hextonib(*p++);
        }

        with_syslock() {
          if (canTryTransmitI(&TK3_CAND, CAN_ANY_MAILBOX, &f)) {
            /* quit and try later */
            goto done;
          }
        }
        break;
      }
    }
  }

done:
  /* retain unprocessed data */
  if (s > fifo) {
    for(p = fifo; s < end; *p++ = *s++);
    end = p;
  }
  return len;
}

static void
tk3can_sendev(void *arg)
{
  tk3can_send(NULL, 0);
}

static void
tk3can_sendcb(CANDriver *can, uint32_t flags)
{
  static tk3event(ev, tk3can_sendev, NULL);

  /* ack message */
  tk3msg_log(TK3CH_CAN, "4\r");

  /* schedule next write */
  tk3ev_schedule(&ev, TK3EV_NOW);
}


/* --- tk3can_recv --------------------------------------------------------- */

static void
tk3can_recv(CANDriver *can, uint32_t flags)
{
  CANRxFrame f;
  uint8_t buffer[24], *p;
  int i;

  for(;;) {
    /* pop next frame */
    with_syslock() {
      if (canTryReceiveI(can, CAN_ANY_MAILBOX, &f)) return;
    }

    /* encaps */
    p = buffer;
    *p++ = 't';

    *p++ = nibtohex(f.SID >> 8); /* id */
    *p++ = nibtohex(f.SID >> 4);
    *p++ = nibtohex(f.SID);

    *p++ = nibtohex(f.DLC); /* len */

    for(i = 0; i < f.DLC; i++) { /* data */
      *p++ = nibtohex(f.data8[i] >> 4);
      *p++ = nibtohex(f.data8[i]);
    }

    *p++ = '\r'; /* end */

    tk3msg_log(TK3CH_CAN, "4%b", buffer, p - buffer);
  }
}


/* --- ascii / bin conversions --------------------------------------------- */

static inline uint8_t
nibtohex(uint8_t x)
{
  return (x & 0xf) + (((x & 0xf) < 0xa) ? '0' : 'A' - 0xa);
}

static inline uint8_t
hextonib(uint8_t x)
{
  return x - ((x < 'A') ? '0' : (x < 'a') ? 'A' - 0xa : 'a' - 0xa);
}
