/*
 * Copyright (c) 2019-2021,2024-2025 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Tue Jul 23 2019
 */
#include "autoconf.h"

#include "tk3-paparazzi.h"

/* AK8963 regs */
enum {
  AK8963_WIA =			0x0,
  AK8963_INFO,
  AK8963_ST1,

  AK8963_CNTL1 =		0xa,
  AK8963_CNTL2,
  AK8963_ASTC,

  AK8963_ASAX =			0x10
};

/* sensor data */
struct __attribute__((__packed__)) ak8963_regs {
  uint8_t	st1;

  uint8_t	xout_l;	int8_t xout_h;
  uint8_t	yout_l;	int8_t yout_h;
  uint8_t	zout_l;	int8_t zout_h;

  uint8_t	st2;
};

static __attribute__ ((const)) struct ak8963_regs *
ak8963_regs(void) { return tk3_padmem(struct ak8963_regs, TK3_AK8963_MEM); }

static uint16_t ak8963_asa[3];
static struct { int m[3]; } stvalid;


static int	tk3sens_ak8963_config(void);
static int	tk3sens_ak8963_selftest(void);
static int	tk3sens_ak8963_reset(void);


/* --- tk3sens_ak8963_probe ------------------------------------------------ */

int
tk3sens_ak8963_probe()
{
  uint8_t *v = tk3_padmem(uint8_t, TK3_AK8963_MEM);

  /* check AK8963 ID */
  if (tk3sens_read(TK3_AK8963, 1, AK8963_WIA, v)) return 0;
  if (*v != 0x48) return 0;

  return 1;
}


/* --- tk3sens_ak8963_init ------------------------------------------------- */

int
tk3sens_ak8963_init()
{
  if (tk3sens_ak8963_reset()) return 1;
  if (tk3sens_ak8963_selftest()) return 1;
  if (tk3sens_ak8963_config()) return 1;

  tk3msg_log(TK3CH_USB0, "NAK8963 configured");
  return 0;
}


/* --- tk3sens_ak8963_config ----------------------------------------------- */

static int
tk3sens_ak8963_config()
{
  /* Power down */
  if (tk3sens_write(
        TK3_AK8963, 1,
        AK8963_CNTL1, 0x0 /* Power down */))
    goto fail;

  /* 100µs POR */
  tk3clk_delay(100);

  /* AK8963 16 bits, continuous mode 100Hz */
  if (tk3sens_write(
        TK3_AK8963, 1,
        AK8963_CNTL1, 0x16 /* 16 bits | MODE 2 */))
    goto fail;

  /* 100µs POR */
  tk3clk_delay(100);

  return 0;

fail:
  tk3msg_log(TK3CH_USB0, "Emagnetometer configuration failed");
  return 1;
}


/* --- tk3sens_ak8963_selftest --------------------------------------------- */

static int
tk3sens_ak8963_selftest()
{
  uint8_t asa[3];
  uint8_t v;
  int32_t m;

  /* select fuse ROM */
  if (tk3sens_write(
        TK3_AK8963, 1,
        AK8963_CNTL1, 0x0f /* 16 bits | Fuse ROM */))
    goto fail;

  /* 100µs POR */
  tk3clk_delay(100);

  /* ASA */
  if (tk3sens_read(TK3_AK8963, 3, AK8963_ASAX, asa)) goto fail;
  ak8963_asa[0] = asa[0] + 128;
  ak8963_asa[1] = asa[1] + 128;
  ak8963_asa[2] = asa[2] + 128;

  /* power down */
  if (tk3sens_write(
        TK3_AK8963, 1,
        AK8963_CNTL1, 0x0 /* Power down */))
    goto fail;

  /* 100µs POR */
  tk3clk_delay(100);

  /* self-test enable */
  if (tk3sens_write(
        TK3_AK8963, 2,
        AK8963_ASTC, 0x40 /* SELF */,
        AK8963_CNTL1, 0x8 /* 14bits, self-test mode */))
    goto fail;

  /* 100µs POR */
  tk3clk_delay(100);

  /* flush data */
  if (tk3sens_read(
        TK3_AK8963, sizeof(*ak8963_regs()), AK8963_ST1, ak8963_regs()))
    goto fail;

  /* wait for data ready */
  v = 0;
  do {
    if (v++ > 250) goto fail;
    tk3clk_delayuntil(1000 /* 1ms */);
    if (tk3sens_read(TK3_AK8963, 1, AK8963_ST1, &ak8963_regs()->st1))
      goto fail;
  } while(!(ak8963_regs()->st1 & 1));
  if (tk3sens_read(
        TK3_AK8963, sizeof(*ak8963_regs()), AK8963_ST1, ak8963_regs()))
    goto fail;

  /* check */
  m = ak8963_regs()->xout_h << 8 | ak8963_regs()->xout_l;
  m *= ak8963_asa[0] / 256;
  if (m < -50 || m > 50) {
    tk3msg_log(TK3CH_USB0, "Emagnetometer %c axis not functional", 'X');
    tk3fb_on(TK3FB_SENS_ERR);
  } else
    stvalid.m[0] = 1;

  m = ak8963_regs()->yout_h << 8 | ak8963_regs()->yout_l;
  m *= ak8963_asa[1] / 256;
  if (m < -50 || m > 50) {
    tk3msg_log(TK3CH_USB0, "Emagnetometer %c axis not functional", 'Y');
    tk3fb_on(TK3FB_SENS_ERR);
  } else
    stvalid.m[1] = 1;

  m = ak8963_regs()->zout_h << 8 | ak8963_regs()->zout_l;
  m *= ak8963_asa[2] / 256;
  if (m < -800 || m > -200) {
    tk3msg_log(TK3CH_USB0, "Emagnetometer %c axis not functional", 'Z');
    tk3fb_on(TK3FB_SENS_ERR);
  } else
    stvalid.m[2] = 1;

  /* done */
  return 0;

fail:
  tk3msg_log(TK3CH_USB0, "Emagnetometer self-test failed");
  return 1;
}


/* --- tk3sens_ak8963_reset ------------------------------------------------ */

static int
tk3sens_ak8963_reset()
{
  /* invalidate data */
  stvalid.m[0] = stvalid.m[1] = stvalid.m[2] = 0;

  /* reset */
  if (tk3sens_write(TK3_AK8963, 1, AK8963_CNTL2, 0x01 /* SRST */))
    goto fail;

  /* 100µs POR */
  tk3clk_delay(100);
  return 0;

fail:
  tk3msg_log(TK3CH_USB0, "EAK8963 not responding");
  return 1;
}


/* --- tk3sens_ak8963_aioread ---------------------------------------------- */

void
tk3sens_ak8963_aioread(void (*cb)(void *))
{
  tk3sens_aioread(
    TK3_AK8963, sizeof(*ak8963_regs()),
    AK8963_ST1, ak8963_regs(), cb);
}

int
tk3sens_ak8963_data(int32_t m[3])
{
  int32_t rm[3];
  int i;

  if (!(ak8963_regs()->st1 & 1)) return 1;

  rm[0] = ak8963_regs()->xout_h << 8 | ak8963_regs()->xout_l;
  rm[1] = ak8963_regs()->yout_h << 8 | ak8963_regs()->yout_l;
  rm[2] = ak8963_regs()->zout_h << 8 | ak8963_regs()->zout_l;

  for(i = 0; i < 3; i++)
    rm[i] = stvalid.m[i] ? rm[i] * 15 * ak8963_asa[i] / 256 : 0;

  m[0] = rm[1];
  m[1] = rm[0];
  m[2] = -rm[2];

  return 0;
}
