/*
 * Copyright (c) 2021,2024 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Mon Oct 11 2021
 */

#include "autoconf.h"

#include "tk3-paparazzi.h"

static void	tk3aux_sendcb(void *arg, uint32_t flags);
static void	tk3aux_recv(void *arg);
static void	tk3aux_recvcb(void *arg, uint32_t flags);

/* io buffers */
static uint8_t tk3_nocache(mem[2][128]);

static struct tk3_uart {
  USART_TypeDef * const hw;
  struct tk3iob inb, outb;
  const stm32_dma_stream_t *rx, *tx;
  uint32_t rxlen, txlen;
  struct tk3event flush;
} tk3_uart[] = {
  [TK3AUX_8] = {
    .hw = UART8,
    .inb = tk3iob_init("uart8 input", mem[0], 32),
    .outb = tk3iob_init("uart8 output", mem[1], 32),
    .flush = { .cb = tk3aux_recv, .arg = &tk3_uart[0] }
  }
};


/* --- tk3aux_init --------------------------------------------------------- */

void
tk3aux_init()
{
  tk3aux_fini();
  rccResetUART8();
  rccEnableUART8(true);
  nvicEnableVector(STM32_UART8_NUMBER, STM32_IRQ_UART8_PRIORITY);
}


/* --- tk3aux_fini --------------------------------------------------------- */

void
tk3aux_fini()
{
  if (tk3_uart[0].tx) {
    dmaStreamDisable(tk3_uart[0].tx);
    dmaStreamFree(tk3_uart[0].tx);
    tk3_uart[0].tx = NULL;
  }
  if (tk3_uart[0].rx) {
    dmaStreamDisable(tk3_uart[0].rx);
    dmaStreamFree(tk3_uart[0].rx);
    tk3_uart[0].rx = NULL;
  }

  UART8->CR1 = 0;
  UART8->CR3 = 0;
  nvicDisableVector(STM32_UART8_NUMBER);
  rccDisableUART8();
}


/* --- tk3aux_start -------------------------------------------------------- */

void
tk3aux_start(enum tk3aux aux, uint32_t baud)
{
  tk3aux_stop(aux);

  /* setup DMA */
  tk3_uart[aux].tx = dmaStreamAlloc(
    STM32_UART_UART8_TX_DMA_STREAM,
    STM32_IRQ_UART8_PRIORITY, tk3aux_sendcb, &tk3_uart[aux]);
  dmaStreamSetMode(
    tk3_uart[aux].tx,
    STM32_DMA_CR_TCIE | STM32_DMA_CR_TEIE |
    STM32_DMA_CR_PSIZE_BYTE | STM32_DMA_CR_MSIZE_BYTE |
    STM32_DMA_CR_MINC | STM32_DMA_CR_DIR_M2P |
    STM32_DMA_CR_CHSEL(
      STM32_DMA_GETCHANNEL(
        STM32_UART_UART8_TX_DMA_STREAM, STM32_UART8_TX_DMA_CHN)) |
    STM32_DMA_CR_PL(STM32_UART_UART8_DMA_PRIORITY));
  dmaStreamSetPeripheral(tk3_uart[aux].tx, &UART8->TDR);

  tk3_uart[aux].rx = dmaStreamAlloc(
    STM32_UART_UART8_RX_DMA_STREAM,
    STM32_IRQ_UART8_PRIORITY, tk3aux_recvcb, &tk3_uart[aux]);
  dmaStreamSetMode(
    tk3_uart[aux].rx,
    STM32_DMA_CR_TCIE | STM32_DMA_CR_TEIE |
    STM32_DMA_CR_PSIZE_BYTE | STM32_DMA_CR_MSIZE_BYTE |
    STM32_DMA_CR_MINC | STM32_DMA_CR_DIR_P2M |
    STM32_DMA_CR_CHSEL(
      STM32_DMA_GETCHANNEL(
        STM32_UART_UART8_RX_DMA_STREAM, STM32_UART8_RX_DMA_CHN)) |
    STM32_DMA_CR_PL(STM32_UART_UART8_DMA_PRIORITY));
  dmaStreamSetPeripheral(tk3_uart[aux].rx, &UART8->RDR);

  /* configure GPIO */
  palSetPadMode(GPIOE, 0, PAL_MODE_ALTERNATE(8));
  palSetPadMode(GPIOE, 1, PAL_MODE_ALTERNATE(8));

  /* configure UART */
  UART8->BRR = STM32_UART8CLK / baud;
  UART8->ICR = UART8->ISR;
  UART8->CR3 = USART_CR3_DMAT | USART_CR3_DMAR;
  UART8->CR2 = 0;
  UART8->CR1 =
    USART_CR1_IDLEIE |
    USART_CR1_TE | USART_CR1_RE | USART_CR1_UE;

  /* start reception */
  tk3aux_recvcb(&tk3_uart[aux], 0);
}


/* --- tk3aux_stop --------------------------------------------------------- */

void
tk3aux_stop(enum tk3aux aux)
{
  if (tk3_uart[aux].tx) {
    dmaStreamDisable(tk3_uart[aux].tx);
    dmaStreamFree(tk3_uart[aux].tx);
    tk3_uart[aux].tx = NULL;
  }
  if (tk3_uart[aux].rx) {
    dmaStreamDisable(tk3_uart[aux].rx);
    dmaStreamFree(tk3_uart[aux].rx);
    tk3_uart[aux].rx = NULL;
  }

  UART8->CR1 = 0;
  UART8->CR2 = 0;
  UART8->CR3 = 0;
}


/* --- tk3aux_send --------------------------------------------------------- */

size_t
tk3aux_send(enum tk3aux aux, const uint8_t *data, size_t len)
{
  len = tk3iob_write(&tk3_uart[aux].outb, data, len);

  if (!tk3_uart[aux].tx)
    tk3aux_start(aux, 38400);

  if (!(tk3_uart[aux].tx->stream->CR & STM32_DMA_CR_EN))
    /* no transfer in progress, start one */
    tk3aux_sendcb(&tk3_uart[aux], 0);

  return len;
}

static void
tk3aux_sendcb(void *arg, uint32_t flags)
{
  struct tk3_uart *uart = arg;
  struct tk3iob_extent e;

  /* transaction complete (or DMA bus error) */
  if (flags & (STM32_DMA_ISR_TCIF | STM32_DMA_ISR_TEIF)) {
    e.len = uart->txlen - dmaStreamGetTransactionSize(uart->tx);
    if (e.len)
      tk3iob_rcommit(&uart->outb, e.len);
  }

  /* start new transaction */
  e = tk3iob_rbegin(&uart->outb, 32);
  if (e.len) {
    dmaStreamSetMemory0(uart->tx, e.r);
    dmaStreamSetTransactionSize(uart->tx, e.len);
    uart->txlen = e.len;
    dmaStreamEnable(uart->tx);
  }
}


/* --- tk3aux_recv --------------------------------------------------------- */

static void
tk3aux_recv(void *arg)
{
  struct tk3_uart *uart = arg;
  struct tk3iob_extent e;

  e = tk3iob_rbegin(&uart->inb, 32);
  if (!e.len) return;

  tk3msg_log(TK3CH_USB1, "8%b", e.r, e.len);
  tk3iob_rcommit(&uart->inb, e.len);

  if (e.len == 32) /* redo if there may be more data */
    tk3ev_schedule(&uart->flush, TK3EV_IDLE);
}

static void
tk3aux_recvcb(void *arg, uint32_t flags)
{
  struct tk3_uart *uart = arg;
  struct tk3iob_extent e;

  /* transaction complete (or DMA bus error) */
  if (flags & (STM32_DMA_ISR_TCIF | STM32_DMA_ISR_TEIF)) {
    e.len = uart->rxlen - dmaStreamGetTransactionSize(uart->rx);
    if (e.len) {
      tk3iob_wcommit(&uart->inb, e.len);
      tk3ev_schedule(&uart->flush, TK3EV_IDLE);
    }
  }

  /* start new transaction */
  e = tk3iob_wbegin(&uart->inb, 32);
  dmaStreamSetMemory0(uart->rx, e.w);
  dmaStreamSetTransactionSize(uart->rx, e.len);
  uart->rxlen = e.len;
  dmaStreamEnable(uart->rx);
  if (!e.len) tk3msg_log(TK3CH_USB1, "E%s overflow", uart->inb.id);
}


/* --- STM32_UART8_HANDLER ------------------------------------------------- */

OSAL_IRQ_HANDLER(STM32_UART8_HANDLER)
{
  uint32_t isr = UART8->ISR;
  UART8->ICR = isr;

  if (isr & USART_ISR_IDLE)
    tk3_uart[0].rx->stream->CR &= ~STM32_DMA_CR_EN;
}
