/*
 * Copyright (c) 2019-2021,2024 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Tue Jul 23 2019
 */
#include "autoconf.h"

#include <stdarg.h>

#include "tk3-paparazzi.h"

/* I2C communication data */
static struct {
  const stm32_dma_stream_t *	dma;
} rx;

static struct {
  const stm32_dma_stream_t *dma;

  volatile uint32_t status;
} tx;

/* async events */
static tk3event(aiocb, NULL, NULL);


/* --- tk3sens_hwinit ------------------------------------------------------ */

void
tk3sens_hwinit()
{
  /* reset I2C */
  tk3sens_hwfini();
  rccResetI2C1();
  rccEnableI2C1(true);

  /* bitbang 10 clock pulses at 100kHz to unlock any locked slaves (10th pulse
   * acts as NACK), then send a STOP (SDA from low to high while CLK is high) */
  palSetPad(GPIOB, 8);
  palSetPad(GPIOB, 9);
  palSetPadMode(GPIOB, 8, PAL_MODE_OUTPUT_OPENDRAIN);
  palSetPadMode(GPIOB, 9, PAL_MODE_OUTPUT_OPENDRAIN);
  for (int i = 0; i < 10; i++) {
    tk3clk_delay(5);
    palSetPad(GPIOB, 8);
    tk3clk_delay(5);
    palClearPad(GPIOB, 8);
  }

  palClearPad(GPIOB, 9);
  tk3clk_delay(5);
  palSetPad(GPIOB, 8);
  tk3clk_delay(2);
  palSetPad(GPIOB, 9);
  tk3clk_delay(2);

  /* configure SDA/SCL lines */
  palSetPadMode(GPIOB, 8, PAL_MODE_ALTERNATE(4) | PAL_STM32_OTYPE_OPENDRAIN);
  palSetPadMode(GPIOB, 9, PAL_MODE_ALTERNATE(4) | PAL_STM32_OTYPE_OPENDRAIN);

  /* setup DMA */
  tx.dma = dmaStreamAlloc(STM32_I2C_I2C1_TX_DMA_STREAM, 0, NULL, NULL);
  dmaStreamSetMode(
    tx.dma,
    STM32_DMA_CR_PSIZE_BYTE | STM32_DMA_CR_MSIZE_BYTE |
    STM32_DMA_CR_MINC | STM32_DMA_CR_DIR_M2P |
    STM32_DMA_CR_CHSEL(
      STM32_DMA_GETCHANNEL(
        STM32_I2C_I2C1_TX_DMA_STREAM, STM32_I2C1_TX_DMA_CHN)) |
    STM32_DMA_CR_PL(STM32_I2C_I2C1_DMA_PRIORITY));
  dmaStreamSetPeripheral(tx.dma, &I2C1->TXDR);
  dmaStreamDisable(tx.dma);

  rx.dma = dmaStreamAlloc(STM32_I2C_I2C1_RX_DMA_STREAM, 0, NULL, NULL);
  dmaStreamSetMode(
    rx.dma,
    STM32_DMA_CR_PSIZE_BYTE | STM32_DMA_CR_MSIZE_BYTE |
    STM32_DMA_CR_MINC | STM32_DMA_CR_DIR_P2M |
    STM32_DMA_CR_CHSEL(
      STM32_DMA_GETCHANNEL(
        STM32_I2C_I2C1_RX_DMA_STREAM, STM32_I2C1_RX_DMA_CHN)) |
    STM32_DMA_CR_PL(STM32_I2C_I2C1_DMA_PRIORITY));
  dmaStreamSetPeripheral(rx.dma, &I2C1->RXDR);
  dmaStreamDisable(rx.dma);

  /* 600kHz bus - normally not supported but seems to work */
  I2C1->TIMINGR =
    ( 0 << I2C_TIMINGR_PRESC_Pos) |
    ( 2 << I2C_TIMINGR_SCLDEL_Pos) |
    ( 2 << I2C_TIMINGR_SDADEL_Pos) |
    ( 16 << I2C_TIMINGR_SCLH_Pos) |
    ( 16 << I2C_TIMINGR_SCLL_Pos);

  /* 1.2ms SCL low timeout */
  I2C1->TIMEOUTR = 15;

  /* setup interrupts */
  nvicEnableVector(STM32_I2C1_EVENT_NUMBER, STM32_I2C_I2C1_IRQ_PRIORITY);
  nvicEnableVector(STM32_I2C1_ERROR_NUMBER, STM32_I2C_I2C1_IRQ_PRIORITY);
  I2C1->CR1 =
    I2C_CR1_TXDMAEN | I2C_CR1_RXDMAEN |
    I2C_CR1_TCIE | I2C_CR1_STOPIE | I2C_CR1_ERRIE | I2C_CR1_NACKIE;

  /* enable peripheral */
  tx.status = 0;
  I2C1->CR1 |= I2C_CR1_PE;
}


/* --- tk3sens_hwfini ------------------------------------------------------ */

void
tk3sens_hwfini()
{
  tx.status |= I2C_ISR_STOPF;
  tx.status &= ~I2C_ISR_BUSY;
  tk3ev_schedule(&aiocb, TK3EV_NOW);

  I2C1->CR1 &= ~I2C_CR1_PE;
  dmaStreamFree(tx.dma);
  dmaStreamFree(rx.dma);

  nvicDisableVector(STM32_I2C1_EVENT_NUMBER);
  nvicDisableVector(STM32_I2C1_ERROR_NUMBER);
  rccDisableI2C1();

  palSetPadMode(GPIOB, 8, PAL_MODE_RESET);
  palSetPadMode(GPIOB, 9, PAL_MODE_RESET);
}


/* --- tk3sens_write ------------------------------------------------------- */

int
tk3sens_write(uint32_t dev, uint8_t n, ...)
{
  uint8_t regs[2];
  va_list ap;

  /* loop on writes */
  va_start(ap, n);
  while (n--) {
    regs[0] = va_arg(ap, unsigned int);
    regs[1] = va_arg(ap, unsigned int);
    dmaStreamSetMemory0(tx.dma, regs);
    dmaStreamSetTransactionSize(tx.dma, 2);
    dmaStreamSetTransactionSize(rx.dma, 0);
    __DMB(); /* sync 'regs' */
    dmaStreamEnable(tx.dma);

    tx.status = I2C_ISR_BUSY;
    I2C1->TIMEOUTR |= I2C_TIMEOUTR_TIMOUTEN;
    I2C1->CR2 =
      I2C_CR2_START |
      2 << I2C_CR2_NBYTES_Pos |
      (dev & 0xff) << 1;

    do { __WFI(); } while(tx.status & I2C_ISR_BUSY);

    if (tx.status) break;
  }
  va_end(ap);

  return tx.status;
}


/* --- tk3sens_read -------------------------------------------------------- */

int
tk3sens_read(uint32_t dev, uint8_t n, uint8_t addr, volatile void *value)
{
  tk3sens_aioread(dev, n, addr, value, NULL);

  do { __WFI(); } while(tx.status & I2C_ISR_BUSY);
  __DMB(); /* sync 'value' array */

  return tx.status;
}

void
tk3sens_aioread(uint32_t dev, uint8_t n, uint8_t addr, volatile void *value,
               void (*cb)(void *))
{
  static uint8_t tk3_nocache(reg);

  tk3ev_set(&aiocb, cb, (void *)&tx.status);

  reg = addr;
  __DMB(); /* sync 'addr' */
  dmaStreamSetMemory0(tx.dma, &reg);
  dmaStreamSetTransactionSize(tx.dma, 1);
  dmaStreamSetMemory0(rx.dma, value);
  dmaStreamSetTransactionSize(rx.dma, n);
  dmaStreamEnable(tx.dma);

  tx.status = I2C_ISR_BUSY;
  I2C1->TIMEOUTR |= I2C_TIMEOUTR_TIMOUTEN;
  I2C1->CR2 =
    I2C_CR2_START |
    1 << I2C_CR2_NBYTES_Pos |
    (dev & 0xff) << 1;
}


/* --- STM32_I2C1_EVENT_HANDLER -------------------------------------------- */

OSAL_IRQ_HANDLER(STM32_I2C1_EVENT_HANDLER)
{
  uint32_t isr = I2C1->ISR;
  I2C1->ICR = isr & (I2C_ICR_STOPCF | I2C_ICR_NACKCF);

  /* abort transaction on NACK */
  if (isr & I2C_ISR_NACKF) {
    dmaStreamDisable(tx.dma);
    dmaStreamDisable(rx.dma);

    I2C1->ICR |= I2C_ICR_ADDRCF;
    I2C1->CR2 |= I2C_CR2_STOP;
    tx.status |= I2C_ISR_NACKF;
    return;
  }

  /* transfer complete */
  if (isr & I2C_ISR_TC) {
    dmaStreamDisable(tx.dma);

    /* pending read */
    if (dmaStreamGetTransactionSize(rx.dma) > 0) {
      dmaStreamEnable(rx.dma);

      I2C1->CR2 =
        (I2C1->CR2 & ~I2C_CR2_NBYTES) |
        I2C_CR2_RD_WRN |
        (dmaStreamGetTransactionSize(rx.dma) << I2C_CR2_NBYTES_Pos);

      I2C1->CR2 |= I2C_CR2_START;
      return;
    }

    dmaStreamDisable(rx.dma);

    I2C1->CR2 |= I2C_CR2_STOP;
    return;
  }

  /* transaction complete */
  if (isr & I2C_ISR_STOPF) {
    I2C1->TIMEOUTR &= ~I2C_TIMEOUTR_TIMOUTEN;
    tx.status &= ~I2C_ISR_BUSY;
    tk3ev_schedule(&aiocb, TK3EV_NOW);
    return;
  }
}


/* --- STM32_I2C1_ERROR_HANDLER -------------------------------------------- */

OSAL_IRQ_HANDLER(STM32_I2C1_ERROR_HANDLER)
{
  uint32_t isr;

  isr = I2C1->ISR & (I2C_ISR_TIMEOUT | I2C_ISR_BERR | I2C_ISR_ARLO);
  I2C1->ICR = isr;
  tx.status |= isr;

  if (isr & I2C_ISR_TIMEOUT) tk3msg_log(TK3CH_USB0, "Esensor bus timeout");
  if (isr & I2C_ISR_BERR) tk3msg_log(TK3CH_USB0, "Esensor bus error");
  if (isr & I2C_ISR_ARLO) tk3msg_log(TK3CH_USB0, "Esensor bus arbitration lost");

  /* cancel transaction */
  I2C1->TIMEOUTR &= ~I2C_TIMEOUTR_TIMOUTEN;
  dmaStreamDisable(tx.dma);
  dmaStreamDisable(rx.dma);

  /* invoke callback */
  tx.status &= ~I2C_ISR_BUSY;
  tk3ev_schedule(&aiocb, TK3EV_NOW);
}
