/*
 * Copyright (c) 2019-2021,2024 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Thu Aug  1 2019
 */
#include "autoconf.h"

#include "tk3-paparazzi.h"

static const stm32_dma_stream_t *dma;

static struct __attribute__((__packed__)) {
  uint16_t vref;
  uint16_t vbat;
} measure __attribute__((section(".nocache")));

static struct {
  uint16_t vbat;
} minpeak;


/* --- tk3pwr_hwinit ------------------------------------------------------- */

/* Measure VREFINT and VBAT_MEAS (PA4). Since VREF+ is wired to the 3.3V
 * battery power, VREFINT (~1.22V) will be measured either as 1.22/3.3 = ~0.37
 * (~0x5ea with 12bits precision) when battery is powered or +inf (0xfff) when
 * there is no battery power. */

int
tk3pwr_hwinit()
{
  /* reset min peak values */
  minpeak.vbat = 0xffff;

  /* configure VBAT_MEAS GPIO */
  palSetPadMode(GPIOA, 4, PAL_MODE_INPUT_ANALOG);

  /* setup dma for ADC1 */
  dma = dmaStreamAlloc(
    STM32_ADC_ADC1_DMA_STREAM, STM32_ADC_ADC1_DMA_IRQ_PRIORITY, NULL, NULL);
  dmaStreamSetMode(
    dma,
    STM32_DMA_CR_PSIZE_HWORD | STM32_DMA_CR_MSIZE_HWORD |
    STM32_DMA_CR_MINC | STM32_DMA_CR_CIRC | STM32_DMA_CR_DIR_P2M |
    STM32_DMA_CR_CHSEL(
      STM32_DMA_GETCHANNEL(STM32_ADC_ADC1_DMA_STREAM, STM32_ADC1_DMA_CHN)) |
    STM32_DMA_CR_PL(STM32_ADC_ADC1_DMA_PRIORITY));
  dmaStreamSetPeripheral(dma, &ADC1->DR);
  dmaStreamSetMemory0(dma, &measure);
  dmaStreamSetTransactionSize(dma, 2);
  dmaStreamEnable(dma);

  /* enable ADC1 */
  rccEnableADC1(true);
  nvicEnableVector(STM32_ADC_NUMBER, STM32_ADC_IRQ_PRIORITY);

  /* Enable VREFINT, PCLK2/8 (6.75MHz), 20 ticks delay between samples */
  ADC->CCR =
    ADC_CCR_TSVREFE | (3 << ADC_CCR_ADCPRE_Pos) | (15 << ADC_CCR_DELAY_Pos);

  /* VREFINT and ADC1_IN4 sensing */
  ADC1->SMPR1 = 7 << ADC_SMPR1_SMP17_Pos;	/* 480 cycles on channel 17 */
  ADC1->SMPR2 = 7 << ADC_SMPR2_SMP4_Pos;	/* 480 cycles on channel 4 */
  ADC1->SQR1  = 1 << ADC_SQR1_L_Pos;		/* 2 channels */
  ADC1->SQR2  = 0;
  ADC1->SQR3  =
    17 << ADC_SQR3_SQ1_Pos |			/* #1 channel 17 VREFINT */
    4 << ADC_SQR3_SQ2_Pos;			/* #2 channel 4 PA4 */

  ADC1->CR1 =
    (0 << ADC_CR1_RES_Pos) |			/* 12 bits */
    ADC_CR1_AWDEN | ADC_CR1_AWDIE |		/* analog watchdog */
    ADC_CR1_EOCIE |				/* interrupt after group */
    ADC_CR1_AWDSGL | (17 << ADC_CR1_AWDCH_Pos) |/* watchdog on channel 17 */
    ADC_CR1_SCAN;				/* scan mode */

  ADC1->HTR   = 0xfff;				/* power on detect */
  ADC1->LTR   = 0x7ff;

  /* start continous mode with DMA */
  ADC1->CR2 = ADC_CR2_DMA | ADC_CR2_DDS | ADC_CR2_CONT;
  ADC1->CR2 |= ADC_CR2_ADON;
  ADC1->CR2 |= ADC_CR2_SWSTART;

  return 0;
}


/* --- tk3pwr_vbat --------------------------------------------------------- */

uint16_t
tk3pwr_vbat()
{
  uint32_t vbat;

  /* 1000 (mV) * 3.3/0xfff (ADC) * (22.3k + 3.3k / 3.3k) (voltage divider) */
  vbat = measure.vref < 0x7ff ? (uint32_t)minpeak.vbat * 35420 / 5733 : 0;

  minpeak.vbat = 0xffff;
  return vbat;
}


/* --- STM32_ADC_HANDLER --------------------------------------------------- */

OSAL_IRQ_HANDLER(STM32_ADC_HANDLER)
{
  static tk3timer(timer);
  uint32_t sr = ADC1->SR;

  /* EOC flag is reset by DMA transfer, so never asserted here */
  if (minpeak.vbat > measure.vbat) minpeak.vbat = measure.vbat;

  /* deal with watchdog interrupts */
  if (!(sr & ADC_SR_AWD)) return;

  ADC1->SR &= ~ADC_SR_AWD;

  if (ADC1->LTR == 0) {
    /* power off event: switch to power on (< 0x7ff) detection */
    ADC1->HTR   = 0xfff;
    ADC1->LTR   = 0x7ff;

    /* trigger callback after 100ms steady off */
    tk3clk_settimer(&timer, 0, 100000, TK3EV_IDLE, tk3pwr_off, tk3pwr_off);
  } else {
    /* power on event: switch to power off (> 0x7ff) detection */
    ADC1->HTR   = 0x7ff;
    ADC1->LTR   = 0x0;

    /* trigger callback after 250ms steady on */
    tk3clk_settimer(&timer, 0, 250000, TK3EV_IDLE, tk3pwr_on, tk3pwr_on);
  }
}
