/*
 * Copyright (c) 2019-2021,2024 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Thu Nov 21 2019
 */
#include "autoconf.h"

#include <string.h>

#include "tk3-paparazzi.h"
#include "tk3-boot.h"


void	set_option_bytes(void);


/* --- Reset_Handler ------------------------------------------------------- */

static tk3_nocache(uint32_t *bootaddr);
static tk3_nocache(uint32_t bootmode);

#define BOOTMODE_BOOTADDR	0xb0074dd2

void __attribute__((section(".trampoline"), naked, noreturn))
Reset_Handler(void)
{
  extern uintptr_t _crt0_entry;

  /* check jump address after a software reset */
  if ((RCC->CSR & RCC_CSR_SFTRSTF) && bootmode == BOOTMODE_BOOTADDR) {
    bootmode = 0;
    __DSB();
    __set_MSP(bootaddr[0]);
    goto *(void *)bootaddr[1];
  }

  /* reset reset flags */
  RCC->CSR = RCC_CSR_RMVF;
  while (RCC->CSR & 0xff000000);
  bootmode = 0;

  /* copy bootloader to ram - can't use memcpy() yet */
  extern uint32_t _text_load, _text_start, _text_end;
  extern uint32_t _rodata_load, _rodata_start, _rodata_end;
  uint32_t *s, *d;

  if (&_text_start != &_text_end) {
    s = &_text_load;
    d = &_text_start;
    do { *d++ = *s++; } while(d < &_text_end);
  }

  if (&_rodata_start != &_rodata_end) {
    s = &_rodata_load;
    d = &_rodata_start;
    do { *d++ = *s++; } while(d < &_rodata_end);
  }

  /* go */
  __DSB();
  goto *&_crt0_entry;
}


/* --- __early_init -------------------------------------------------------- */

void
__early_init(void)
{
  extern uint8_t _ram_vectors[], _vectors_start[], _vectors_end[];

  /* remap vectors in ITCM RAM */
  memcpy(_ram_vectors, _vectors_start, _vectors_end - _vectors_start);
  __DSB();

  SCB->VTOR = (uint32_t)_ram_vectors;

  /* zero out .nocache area */
  extern uint8_t _nocache_start[], _nocache_end[];
  memset(_nocache_start, 0, _nocache_end - _nocache_start);

  /* enable clocks */
  stm32_clock_init();
}


/* --- boardInit ----------------------------------------------------------- */

void
boardInit(void)
{
  /* disable AXIM caches */
  SCB_DisableICache();
  SCB_DisableDCache();

  /* disable ART */
  FLASH->ACR &= ~(FLASH_ACR_ARTEN | FLASH_ACR_PRFTEN);
  do {} while (FLASH->ACR & (FLASH_ACR_ARTEN | FLASH_ACR_PRFTEN));
  FLASH->ACR |= FLASH_ACR_ARTRST;

  /* enable GPIO clocks */
  rccResetAHB1(STM32_GPIO_EN_MASK);
  rccEnableAHB1(STM32_GPIO_EN_MASK, true);

  /* check SWDIO/SWDCLK high for safety reboot in DFU mode */
  palSetPadMode(GPIOA, 13, PAL_MODE_INPUT_PULLDOWN);
  palSetPadMode(GPIOA, 14, PAL_MODE_INPUT_PULLDOWN);
  for(int i = 0; i < 100; i++) {
    if (palReadPad(GPIOA, 13) && palReadPad(GPIOA, 14)) {
      bootaddr = (void *)dfuaddr;
      bootmode = BOOTMODE_BOOTADDR;
      __DSB();
      NVIC_SystemReset();
    }
  }

  /* enable SWDIO/SWDCLK */
  palSetPadMode(GPIOA, 13, PAL_MODE_ALTERNATE(0));
  palSetPadMode(GPIOA, 14, PAL_MODE_ALTERNATE(0));
}


/* --- main ---------------------------------------------------------------- */

int
main(void)
{
  halInit();
  osalSysEnable();

  tk3set_init();
  tk3fb_init();
  tk3usb_init(NULL, NULL);

  bootaddr = (void *)bootloader(tk3usb_iniob(0), tk3usb_outiob(0));

  tk3usb_fini();

  /* update option bytes */
  set_option_bytes();

  /* reboot into application */
  if (bootaddr == (void *)-1) /* timeout: boot according to option bytes */
    bootaddr = (void *)((FLASH->OPTCR1 & 0xffff) << 14);
  bootmode = BOOTMODE_BOOTADDR;
  __DSB();

  NVIC_SystemReset();
  return 0;
}


/* --- set_option_bytes ---------------------------------------------------- */

void
set_option_bytes(void)
{
  extern uint32_t boot_itcm, flash_itcm;

  uint32_t optcr, optcr1, lock;

  /* lock bootloader flash */
  optcr = FLASH->OPTCR & ~FLASH_OPTCR_nWRP_0;
  lock = tk3set_get(TK3SET_BLLOCK, 0);
  if (lock == 1234567890) optcr |= FLASH_OPTCR_nWRP_0;

  /* set boot address option byte */
  optcr1 =
    (((uint32_t)&boot_itcm >> 14) << 16) |
    (uint32_t)&flash_itcm >> 14;

  /* update if needed */
  if (FLASH->OPTCR == optcr && FLASH->OPTCR1 == optcr1) return;

  do {} while (FLASH->SR & FLASH_SR_BSY);
  FLASH->OPTKEYR = 0x08192a3b;
  FLASH->OPTKEYR = 0x4c5d6e7f;
  do {} while (FLASH->OPTCR & FLASH_OPTCR_OPTLOCK);

  FLASH->OPTCR1 = optcr1;
  FLASH->OPTCR = optcr & ~(FLASH_OPTCR_OPTSTRT | FLASH_OPTCR_OPTLOCK);
  FLASH->OPTCR |= FLASH_OPTCR_OPTSTRT;

  __DSB();
  do {} while (FLASH->SR & FLASH_SR_BSY);
  FLASH->OPTCR |= FLASH_OPTCR_OPTLOCK;
}


/* --- flash access -------------------------------------------------------- */

static const uint32_t flash_errs =
FLASH_SR_ERSERR | FLASH_SR_PGPERR | FLASH_SR_PGAERR | FLASH_SR_WRPERR;

void
unlock_flash()
{
  if (!(FLASH->CR & FLASH_CR_LOCK)) return;

  do {} while (FLASH->SR & FLASH_SR_BSY);

  FLASH->KEYR = 0x45670123;
  FLASH->KEYR = 0xcdef89ab;
  do {} while (FLASH->CR & FLASH_CR_LOCK);
}

int
erase_flash(size_t sector)
{
  FLASH->SR |= flash_errs;

  do {} while (FLASH->SR & FLASH_SR_BSY);
  FLASH->CR =
    ((sector << FLASH_CR_SNB_Pos) & FLASH_CR_SNB_Msk) | FLASH_CR_SER;
  FLASH->CR |= FLASH_CR_STRT;
  __DSB();

  do {} while (FLASH->SR & FLASH_SR_BSY);
  FLASH->CR = 0;

  return FLASH->SR & flash_errs;
}

int
write_flash(uintptr_t addr, const uint8_t *data, size_t size)
{
  uint32_t value;

  FLASH->SR |= flash_errs;
  do {
    do {} while (FLASH->SR & FLASH_SR_BSY);
    if (FLASH->SR & flash_errs) break;

    with_syslock() {
      FLASH->CR = (2 << FLASH_CR_PSIZE_Pos) | FLASH_CR_PG;

      value = data[0] | (data[1] << 8) | (data[2] << 16) | (data[3] << 24);
      *(volatile uint32_t *)(void *)addr = value;
    }

    addr += 4;
    data += 4;
    size -= 4;
    __DSB();
  } while(size);

  do {} while (FLASH->SR & FLASH_SR_BSY);
  FLASH->CR = 0;

  return FLASH->SR & flash_errs;
}

void
lock_flash()
{
  do {} while (FLASH->SR & FLASH_SR_BSY);
  FLASH->CR |= FLASH_CR_LOCK;
}
