/*
 * Copyright (c) 2014-2017 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution and use  in source  and binary  forms,  with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 *					Anthony Mallet on Tue Jun  3 2014
 */
#include "acheader.h"

#include <avr/eeprom.h>
#include <avr/pgmspace.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "common/tk3-mikrokopter.h"
#include "mkfl.h"

/* default settings, stored in the MCU EEPROM. The format is a list
 * of nul-terminated strings, interpreted as a key/value/descr */
static const char tk3_dparams[] EEMEM __attribute__((used)) = {
  TK3SET_MAGIC "\0"
  TK3SET_VERSION "\0"	"\0\0\0\0"	"board version, 0 for auto" "\0"
  TK3SET_ROBOT "\0"	"\0\0\0\0"	"robot number" "\0"
  TK3SET_MOTORS "\0"	"\0\0\0\4"	"motors controlled" "\0"
  TK3SET_UART "\0"	"\0\x7\xa1\x20"	"uart baud rate" "\0"

  TK3SET_GYRO_YOFF "\0"	"\0\0\0\0"	"yaw offset, mrad/s" "\0"
  TK3SET_GYRO_YSEN "\0"	"\0\0\x12\xf0"	"yaw sensivity, mrad/s/V" "\0"
  TK3SET_GYRO_POFF "\0"	"\0\0\0\0"	"pitch offset, mrad/s" "\0"
  TK3SET_GYRO_PSEN "\0"	"\0\0\x12\xf0"	"pitch sensivity, mrad/s/V" "\0"
  TK3SET_GYRO_ROFF "\0"	"\0\0\0\0"	"roll offset, mrad/s" "\0"
  TK3SET_GYRO_RSEN "\0"	"\0\0\x12\xf0"	"roll sensivity, mrad/s/V" "\0"

  TK3SET_ACC_XOFF "\0"	"\0\0\0\0"	"x offset, mm/s²" "\0"
  TK3SET_ACC_XSEN "\0"	"\0\0\x41\x2a"	"x sensivity, mm/s²/V" "\0"
  TK3SET_ACC_YOFF "\0"	"\0\0\0\0"	"y offset, mm/s²" "\0"
  TK3SET_ACC_YSEN "\0"	"\0\0\x41\x2a"	"y sensivity, mm/s²/V" "\0"
  TK3SET_ACC_ZOFF "\0"	"\0\0\0\0"	"z offset, mm/s²" "\0"
  TK3SET_ACC_ZSEN "\0"	"\0\0\x4c\xa4"	"z sensivity, mm/s²/V" "\0"
};

struct tk3_settings settings;


/* --- load_settings ------------------------------------------------------- */

void
tk3_load_settings(void)
{
  struct tk3_settings set;
  int32_t value;
  char key[16];

  /* load eeprom settings */
  if (tk3_settings_init()) abort();

  /* flags for parameters that are set */
  set.version = 0;
  set.robot_id = 0;
  set.motors = 0;
  set.uart = 0;
  set.gyro_roff = 0;
  set.gyro_rsen = 0;
  set.gyro_poff = 0;
  set.gyro_psen = 0;
  set.gyro_yoff = 0;
  set.gyro_ysen = 0;
  set.acc_xoff = 0;
  set.acc_xsen = 0;
  set.acc_yoff = 0;
  set.acc_ysen = 0;
  set.acc_zoff = 0;
  set.acc_zsen = 0;

  /* loop in EEPROM */
  while(!tk3_settings_next(key, sizeof(key), &value)) {
    if (!strcmp_P(key, PSTR(TK3SET_VERSION))) {
      if (!value) {
        /* auto-detect board version */
        DDRB = 0; PORTB = 0;	/* green led, no pullup */
        DDRD = 0; PORTD = 0x10;	/* J4, pullup */

        if (PINB & 0x2) /* green led on VCC */ {
          if (PIND & 0x10) /* J4 not on GND */
            value = 21;
          else
            value = 22;
        } else
          value = 25;

        tk3_settings_update(tk3_settings_addr(TK3SET_VERSION), value);
      }
      settings.version = value;
      set.version = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_ROBOT))) {
      settings.robot_id = value;
      set.robot_id = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_MOTORS))) {
      settings.motors = value;
      set.motors = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_UART))) {
      settings.uart = value;
      set.uart = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_GYRO_ROFF))) {
      settings.gyro_roff = value;
      set.gyro_roff = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_GYRO_RSEN))) {
      settings.gyro_rsen = value;
      set.gyro_rsen = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_GYRO_POFF))) {
      settings.gyro_poff = value;
      set.gyro_poff = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_GYRO_PSEN))) {
      settings.gyro_psen = value;
      set.gyro_psen = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_GYRO_YOFF))) {
      settings.gyro_yoff = value;
      set.gyro_yoff = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_GYRO_YSEN))) {
      settings.gyro_ysen = value;
      set.gyro_ysen = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_ACC_XOFF))) {
      settings.acc_xoff = value;
      set.acc_xoff = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_ACC_XSEN))) {
      settings.acc_xsen = value;
      set.acc_xsen = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_ACC_YOFF))) {
      settings.acc_yoff = value;
      set.acc_yoff = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_ACC_YSEN))) {
      settings.acc_ysen = value;
      set.acc_ysen = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_ACC_ZOFF))) {
      settings.acc_zoff = value;
      set.acc_zoff = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_ACC_ZSEN))) {
      settings.acc_zsen = value;
      set.acc_zsen = 1;
    } else {
      tk3_led(TK3_LED_RED2, 0xff);
      while(1) tk3_led_update(tk3_clock_gettime());
    }

  }

  if (!(set.version && set.robot_id && set.motors && set.uart &&
        set.gyro_roff && set.gyro_rsen && set.gyro_poff && set.gyro_psen &&
        set.gyro_yoff && set.gyro_ysen && set.acc_xoff && set.acc_xsen &&
        set.acc_yoff && set.acc_ysen && set.acc_zoff && set.acc_zsen)) {
    tk3_led(TK3_LED_RED2, 0xff);
    while(1) tk3_led_update(tk3_clock_gettime());
  }

  settings.gyro_roff_addr = tk3_settings_addr(TK3SET_GYRO_ROFF);
  settings.gyro_rsen_addr = tk3_settings_addr(TK3SET_GYRO_RSEN);
  settings.gyro_poff_addr = tk3_settings_addr(TK3SET_GYRO_POFF);
  settings.gyro_psen_addr = tk3_settings_addr(TK3SET_GYRO_PSEN);
  settings.gyro_yoff_addr = tk3_settings_addr(TK3SET_GYRO_YOFF);
  settings.gyro_ysen_addr = tk3_settings_addr(TK3SET_GYRO_YSEN);

  settings.acc_xoff_addr = tk3_settings_addr(TK3SET_ACC_XOFF);
  settings.acc_xsen_addr = tk3_settings_addr(TK3SET_ACC_XSEN);
  settings.acc_yoff_addr = tk3_settings_addr(TK3SET_ACC_YOFF);
  settings.acc_ysen_addr = tk3_settings_addr(TK3SET_ACC_YSEN);
  settings.acc_zoff_addr = tk3_settings_addr(TK3SET_ACC_ZOFF);
  settings.acc_zsen_addr = tk3_settings_addr(TK3SET_ACC_ZSEN);
}
