/*
 * Copyright (c) 2015,2017 LAAS/CNRS
 * Copyright (c) 2008-2014 Max Planck Institute for Biological Cybernetics
 * All rights reserved.
 *
 * Redistribution and use  in source  and binary  forms,  with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 */
#include "acheader.h"

#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <util/atomic.h>

#include "common/tk3-mikrokopter.h"
#include "mkfl.h"

struct tk3_sensors tk3_sensors;


/* --- tk3_sensor_init ----------------------------------------------------- */

void
tk3_sensor_init(void)
{
  /* external 3V at AREF pin */
  ADMUX = (0 << REFS1) | (0 << REFS0) | (0 << ADLAR);

  /* disable GPIO connected to the ADC */
  DIDR0 =
    (1 << ADC7D) | (1 << ADC6D) | (1 << ADC5D) | (1 << ADC4D) |
    (1 << ADC3D) | (1 << ADC2D) | (1 << ADC1D) | (1 << ADC0D);

  /* free running mode - actually not used because one has to switch channels
   * after each conversion, and the free running mode does not provide a
   * reliable way to do this */
  ADCSRB = (0 << ADTS2) | (0 << ADTS1) | (0 << ADTS0);

  /* clock divisor/128 (~150kHz or 83µs per conversion) */
  ADCSRA = (1 << ADEN) | (1 << ADSC) | (1 << ADIE) |
    (1 << ADPS2) | (1 << ADPS1) | (1 << ADPS0);
}


/* --- tk3_sensor_gyrocal -------------------------------------------------- */

static tk3_time deadline;
static enum tk3_channel report;
static int32_t roff, poff, yoff;

static void	tk3_sensor_gyrocalcb(void);

void
tk3_sensor_gyrocal(tk3_time d, enum tk3_channel r)
{
  tk3_time t = tk3_clock_gettime();

  deadline = t + d;
  roff = poff = yoff = 0;
  report = r;
  tk3_clock_callback(t + 1000, tk3_sensor_gyrocalcb);
}

static void
tk3_sensor_gyrocalcb(void)
{
  tk3_time t = tk3_clock_gettime();

  roff = tk3_sensors.gyro_roll + 255 * roff / 256;
  poff = tk3_sensors.gyro_pitch + 255 * poff / 256;
  yoff = tk3_sensors.gyro_yaw + 255 * yoff / 256;

  if ((t - deadline) & 0x80000000U) {
    tk3_clock_callback(t + 1000, tk3_sensor_gyrocalcb);
    return;
  }

  settings.gyro_roff -= roff / 256;
  settings.gyro_poff -= poff / 256;
  settings.gyro_yoff -= yoff / 256;

  if (settings.gyro_roff_addr)
    tk3_settings_update(settings.gyro_roff_addr, settings.gyro_roff);
  if (settings.gyro_poff_addr)
    tk3_settings_update(settings.gyro_poff_addr, settings.gyro_poff);
  if (settings.gyro_yoff_addr)
    tk3_settings_update(settings.gyro_yoff_addr, settings.gyro_yoff);

  tk3_log(report, PSTR("Z"));
}


/* --- tk3_sensor_acccal --------------------------------------------------- */

static int32_t acc_xmin, acc_ymin, acc_zmin;
static int32_t acc_xmax, acc_ymax, acc_zmax;

static void	tk3_sensor_acccalcb(void);

void
tk3_sensor_acccal(tk3_time d, enum tk3_channel r)
{
  tk3_time t = tk3_clock_gettime();

  deadline = t + d;
  acc_xmin = acc_ymin = acc_zmin = 65536;
  acc_xmax = acc_ymax = acc_zmax = -65536;
  report = r;

  tk3_clock_callback(t + 1000, tk3_sensor_acccalcb);
}

static void
tk3_sensor_acccalcb(void)
{
  tk3_time t = tk3_clock_gettime();

  if (tk3_sensors.gyro_roll < 100 &&
      tk3_sensors.gyro_roll > -100 &&
      tk3_sensors.gyro_pitch < 100 &&
      tk3_sensors.gyro_pitch > -100 &&
      tk3_sensors.gyro_yaw < 100 &&
      tk3_sensors.gyro_yaw > -100) {
    if (tk3_sensors.acc_x < acc_xmin)
      acc_xmin = tk3_sensors.acc_x;
    if (tk3_sensors.acc_x > acc_xmax)
      acc_xmax = tk3_sensors.acc_x;

    if (tk3_sensors.acc_y < acc_ymin)
      acc_ymin = tk3_sensors.acc_y;
    if (tk3_sensors.acc_y > acc_ymax)
      acc_ymax = tk3_sensors.acc_y;

    if (tk3_sensors.acc_z < acc_zmin)
      acc_zmin = tk3_sensors.acc_z;
    if (tk3_sensors.acc_z > acc_zmax)
      acc_zmax = tk3_sensors.acc_z;
  }


  if ((t - deadline) & 0x80000000U) {
    tk3_clock_callback(t + 1000, tk3_sensor_acccalcb);
    return;
  }

  if (acc_xmax - acc_xmin > 9810) {
    settings.acc_xoff -= (acc_xmin + acc_xmax) / 2;
    settings.acc_xoff =
      (int32_t)settings.acc_xoff * 19620 / (acc_xmax - acc_xmin);
    settings.acc_xsen =
      (int32_t)settings.acc_xsen * 19620 / (acc_xmax - acc_xmin);
  }
  if (acc_ymax - acc_ymin > 9810) {
    settings.acc_yoff -= (acc_ymin + acc_ymax) / 2;
    settings.acc_yoff =
      (int32_t)settings.acc_yoff * 19620 / (acc_ymax - acc_ymin);
    settings.acc_ysen =
      (int32_t)settings.acc_ysen * 19620 / (acc_ymax - acc_ymin);
  }
  if (acc_zmax - acc_zmin > 9810) {
    settings.acc_zoff -= (acc_zmin + acc_zmax) / 2;
    settings.acc_zoff =
      (int32_t)settings.acc_zoff * 19620 / (acc_zmax - acc_zmin);
    settings.acc_zsen =
      (int32_t)settings.acc_zsen * 19620 / (acc_zmax - acc_zmin);
  }

  if (settings.acc_xoff_addr)
    tk3_settings_update(settings.acc_xoff_addr, settings.acc_xoff);
  if (settings.acc_yoff_addr)
    tk3_settings_update(settings.acc_yoff_addr, settings.acc_yoff);
  if (settings.acc_zoff_addr)
    tk3_settings_update(settings.acc_zoff_addr, settings.acc_zoff);

  if (settings.acc_xsen_addr)
    tk3_settings_update(settings.acc_xsen_addr, settings.acc_xsen);
  if (settings.acc_ysen_addr)
    tk3_settings_update(settings.acc_ysen_addr, settings.acc_ysen);
  if (settings.acc_zsen_addr)
    tk3_settings_update(settings.acc_zsen_addr, settings.acc_zsen);

  tk3_log(report, PSTR("Z"));
}


/* --- ADC_vect ------------------------------------------------------------ */

ISR(ADC_vect)
{
  uint8_t channel = ADMUX & 0x7;
  int16_t adc = ADC;

  switch(channel) {
    case 0:
      adc = (adc - 512) * 3;
      tk3_sensors.gyro_yaw = (int32_t)settings.gyro_ysen * -adc / 1024;
      tk3_sensors.gyro_yaw += settings.gyro_yoff;
      break;

    case 1:
      adc = (adc - 512) * 3;
      tk3_sensors.gyro_roll = (int32_t)settings.gyro_rsen * -adc / 1024;
      tk3_sensors.gyro_roll += settings.gyro_roff;
      break;

    case 2:
      adc = (adc - 512) * 3;
      tk3_sensors.gyro_pitch = (int32_t)settings.gyro_psen * adc / 1024;
      tk3_sensors.gyro_pitch += settings.gyro_poff;
      break;

    case 3:
      tk3_sensors.barometer = adc;
      break;

    case 4:
      tk3_sensors.battery =
        (uint32_t)adc * 129/4; /* (10000+1000)/1000 * 3/1024 * 1000 */
      break;

    case 5:
      adc = (adc - 512) * 3;
      tk3_sensors.acc_z = (int32_t)settings.acc_zsen * adc / 1024;
      tk3_sensors.acc_z += settings.acc_zoff;
      break;

    case 6:
      adc = (adc - 512) * 3;
      if (settings.version < 25) {
        tk3_sensors.acc_y = (int32_t)settings.acc_ysen * adc / 1024;
        tk3_sensors.acc_y += settings.acc_yoff;
      } else {
        tk3_sensors.acc_x = (int32_t)settings.acc_xsen * -adc / 1024;
        tk3_sensors.acc_x += settings.acc_xoff;
      }
      break;

    case 7:
      adc = (adc - 512) * 3;
      if (settings.version < 25) {
        tk3_sensors.acc_x = (int32_t)settings.acc_xsen * -adc / 1024;
        tk3_sensors.acc_x += settings.acc_xoff;
      } else {
        tk3_sensors.acc_y = (int32_t)settings.acc_ysen * adc / 1024;
        tk3_sensors.acc_y += settings.acc_yoff;
      }
      break;

    default:
      channel = 0;
      break;
  }

  /* start sampling next channel */
  channel++;
  ADMUX = (ADMUX & 0xe0) | (channel & 0x7);
  ADCSRA |= (1 << ADSC);
}
