/*
 * Copyright (c) 2014-2017 LAAS/CNRS
 * Copyright (c) 2008-2014 Max Planck Institute for Biological Cybernetics
 * All rights reserved.
 *
 * Redistribution and use  in source  and binary  forms,  with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 */

#include "acheader.h"

#include <avr/pgmspace.h>

#include <stdlib.h>

#include "common/tk3-mikrokopter.h"
#include "mkfl.h"

#include <util/delay.h>


/* --- local data ---------------------------------------------------------- */

static void	freeze(void);
static uint8_t inited;

/* --- main ---------------------------------------------------------------- */

int
main()
{
  uint8_t functions, count;
  tk3_time date, stamp;
  struct tk3_iorecv *msg;

  /* setup clock & interrupts */
  if (tk3_clock_init(0, NULL))
    freeze();

  /* load settings */
  inited = 0;
  tk3_led_init();
  tk3_load_settings();

  /* initialize hardware */
  tk3_led(TK3_LED_RED2, 0x80);
  tk3_led(TK3_LED_GRN1, functions = 0xff);

  {
    /* check TXD1 / RXD1 junction (JETI jumper) */
    int16_t addr = -1;

    DDRD &= ~(1 << DDD2); DDRD |= (1 << DDD3);
    PORTD |= (1 << PD2); PORTD &= ~(1 << PD3);
    _delay_us(100);
    while(addr != PIND) addr = PIND; /* synchronize pin output */

    if (!(addr & (1 << PIND2)))
      functions = functions * 15 / 16;
  }

  if (tk3_uart_init(0, settings.uart) || tk3_uart_init(1, settings.uart))
    freeze();
  tk3_twi_init(settings.motors);

  tk3_sensor_init();
  count = tk3_motor_init(settings.motors * 2000000);
  if (count < settings.motors)
    functions = functions * count / settings.motors;

  tk3_led(TK3_LED_GRN1, functions);
  tk3_led(TK3_LED_RED2, 0);
  inited = 1;

  /* main loop */
  stamp = tk3_clock_gettime();
  while(1) {

    date = tk3_clock_gettime();

    /* clock timestamps */
    if (date - stamp > 1000000) {
      tk3_log(TK3_TWI, PSTR("t%4"), date);
      stamp += 1000000;
    }

    /* update led status */
    tk3_led_update(date);

    /* process received messages */
    while ((msg = tk3_recv())) {
      uint8_t cmd = msg->data[0];

      switch(cmd) {
        case 'i': /* IMU data */
          if (msg->len == 5) {
            tk3_time p =
              ((uint32_t)msg->data[1] << 24) |
              ((uint32_t)msg->data[2] << 16) |
              ((uint32_t)msg->data[3] << 8) |
              msg->data[4];

            tk3_logdef_imu(msg->state.channel, p, date);
          }
          break;

        case 'b': /* battery level */
          if (msg->len == 5) {
            tk3_time p =
              ((uint32_t)msg->data[1] << 24) |
              ((uint32_t)msg->data[2] << 16) |
              ((uint32_t)msg->data[3] << 8) |
              msg->data[4];

            tk3_logdef_battery(msg->state.channel, p, date);
          }
          break;

        case 'z': /* calibration */
          switch(msg->data[1]) {
            case 'g':
              if (msg->len == 3)
                tk3_sensor_gyrocal(msg->data[2] * 1000000, msg->state.channel);
              break;

            case 'a':
              if (msg->len == 3)
                tk3_sensor_acccal(msg->data[2] * 1000000, msg->state.channel);
              break;
          }
          break;

        case 'm': /* motor data */
          if (msg->len == 5) {
            tk3_time p =
              ((uint32_t)msg->data[1] << 24) |
              ((uint32_t)msg->data[2] << 16) |
              ((uint32_t)msg->data[3] << 8) |
              msg->data[4];

            tk3_logdef_motor(msg->state.channel, !!p);
            tk3_log_buffer(TK3_TWI, msg->data, msg->len);
          }
          break;

        case 'g': /* start motors */
        case 'x': /* stop motors */
        case 'q': /* PWM array */
        case 'w': /* velocity array */
        case '~': /* beep */
          tk3_log_buffer(TK3_TWI, msg->data, msg->len);
          break;

        case 'M': /* motor data forwarder */
          tk3_log_motor(msg->data, msg->len);
          break;

        case 'T': /* clock rate */
          tk3_log_buffer(TK3_UART0, msg->data, msg->len);
          break;

        case '?': /* id */
          if (msg->len == 1) {
            tk3_log(msg->state.channel, PSTR("?mkfl" PACKAGE_VERSION));
          }
          break;
      }
    }

    /* data logging */
    tk3_log_imu(date);
    tk3_log_battery(date);

    /* flush pending data */
    tk3_send();
  }

  return 0;
}


/* --- freeze -------------------------------------------------------------- */

static void
freeze(void)
{
  tk3_led(TK3_LED_GRN1, 0);
  while(1)
    tk3_led_update(tk3_clock_gettime());

}

/* --- tk3_comm_overflow --------------------------------------------------- */

void
tk3_comm_err(void)
{
  /* red led on */
  if (inited) tk3_led(TK3_LED_RED2, 0xff);
}

void
tk3_comm_overflow(void)
{
  /* red led on */
  if (inited) tk3_led(TK3_LED_RED2, 0xff);
}

void
tk3_comm_noverflow(void)
{
  /* red led off */
  if (inited) tk3_led(TK3_LED_RED2, 0);
}
