/*
 * Copyright (c) 2015,2017 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Fri Jan 30 2015
 */
#include "acheader.h"

#include <avr/pgmspace.h>

#include "common/tk3-mikrokopter.h"
#include "mkfl.h"

#include <util/atomic.h>


/* --- tk3_log_imu --------------------------------------------------------- */

static uint32_t imu_log_period[TK3_CHANNEL_MAX];
static uint32_t imu_log_date[TK3_CHANNEL_MAX];
static uint8_t imu_log_id[TK3_CHANNEL_MAX];

void
tk3_logdef_imu(enum tk3_channel channel, uint32_t period, tk3_time date)
{
  imu_log_period[channel] = period;
  imu_log_date[channel] = date;
  imu_log_id[channel] = ~0;
}

void
tk3_log_imu(tk3_time date)
{
  uint8_t c, i;
  struct tk3_sensors s;
  tk3_time last;

  for (c = 0; c < TK3_CHANNEL_MAX; c++) {
    if (!imu_log_period[c]) continue;
    last = imu_log_date[c];
    if ((date - last) & 0x80000000U) continue;

    i = imu_log_id[c];
    do {
      last += imu_log_period[c];
      i++;
    } while ((last - date) & 0x80000000U);
    imu_log_date[c] = last;
    imu_log_id[c] = i;

    ATOMIC_BLOCK(ATOMIC_RESTORESTATE) {
      s.acc_x = tk3_sensors.acc_x;
      s.acc_y = tk3_sensors.acc_y;
      s.acc_z = tk3_sensors.acc_z;

      s.gyro_roll = tk3_sensors.gyro_roll;
      s.gyro_pitch = tk3_sensors.gyro_pitch;
      s.gyro_yaw = tk3_sensors.gyro_yaw;
    }

    tk3_log(c, PSTR("I%1%2%2%2%2%2%2"), i,
            s.acc_x, s.acc_y, s.acc_z,
            s.gyro_roll, s.gyro_pitch, s.gyro_yaw);
  }
}


/* --- tk3_logdef_motor ---------------------------------------------------- */

static bool motor_log[TK3_CHANNEL_MAX];

void
tk3_logdef_motor(enum tk3_channel channel, bool log)
{
  motor_log[channel] = log;
}


void
tk3_log_motor(const uint8_t *msg, uint8_t len)
{
  uint8_t c;

  for (c = 0; c < TK3_CHANNEL_MAX; c++) {
    if (!motor_log[c]) continue;

    tk3_log_buffer(c, msg, len);
  }
}


/* --- tk3_log_battery ----------------------------------------------------- */

static uint32_t battery_log_period[TK3_CHANNEL_MAX];
static uint32_t battery_log_date[TK3_CHANNEL_MAX];
static uint8_t battery_log_id[TK3_CHANNEL_MAX];

void
tk3_logdef_battery(enum tk3_channel channel, uint32_t period, tk3_time date)
{
  battery_log_period[channel] = period;
  battery_log_date[channel] = date;
  battery_log_id[channel] = ~0;
}

void
tk3_log_battery(tk3_time date)
{
  uint8_t c, i;
  uint16_t battery;
  tk3_time last;

  for (c = 0; c < TK3_CHANNEL_MAX; c++) {
    if (!battery_log_period[c]) continue;
    last = battery_log_date[c];
    if ((date - last) & 0x80000000U) continue;

    i = battery_log_id[c];
    do {
      last += battery_log_period[c];
      i++;
    } while ((last - date) & 0x80000000U);
    battery_log_date[c] = last;
    battery_log_id[c] = i;


    ATOMIC_BLOCK(ATOMIC_RESTORESTATE) {
      battery = tk3_sensors.battery;
    }

    tk3_log(c, PSTR("B%1%2"), i, battery);
  }
}
