/*
 * Copyright (c) 2014-2017 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Thu Sep 25 2014
 */
#include "acheader.h"

#include <avr/pgmspace.h>
#include <util/atomic.h>

#include "common/tk3-mikrokopter.h"
#include "mkbl.h"

struct tk3_sensors tk3_sensors;
static uint8_t channel;

static uint16_t adc_offset;

/* --- tk3_sensor_init ----------------------------------------------------- */

int
tk3_sensor_init(void)
{
  uint8_t i;

  channel = 0;

  /* internal 1.1V with external capacitor at AREF pin */
  ADMUX = (1 << REFS1) | (1 << REFS0) | (0 << ADLAR);

  /* disable GPIO connected to the ADC */
  DIDR0 =
    (1 << ADC5D) | (1 << ADC4D) |
    (1 << ADC3D) | (1 << ADC2D) | (1 << ADC1D) | (1 << ADC0D);

  /* manual trigger */
  ADCSRB = (0 << ADTS2) | (0 << ADTS1) | (0 << ADTS0);

  /* clock divisor/8 */
  ADCSRA =
    (0 << ADEN) | (0 << ADATE) | (0 << ADIE) |
    (0 << ADPS2) | (1 << ADPS1) | (1 << ADPS0);

  /* calibrate ADC offset */
  tk3_sensor_update(TK3_SENSOR_CURRENT);
  tk3_sensor_update(TK3_SENSOR_CURRENT);
  for(i = 0; i < 255; i++) {
    tk3_sensor_update(TK3_SENSOR_CURRENT);
    if (adc_offset < ADC) adc_offset = ADC;
  }

  /* read all once */
  tk3_sensors.peak_current = 0;
  tk3_sensor_update(TK3_SENSOR_BATTERY);
  tk3_sensor_update(TK3_SENSOR_CURRENT);
  tk3_sensor_update(TK3_SENSOR_MCU_TEMP);
  tk3_sensor_update(TK3_SENSOR_PCB_TEMP);

  return 0;
}


/* --- tk3_sensor_update --------------------------------------------------- */

void
tk3_sensor_update(enum tk3_sensor_id id)
{
  uint16_t adc;
  uint8_t c;

  if (!(id & 0x80)) channel = id;

  switch(channel) {
    case TK3_SENSOR_BATTERY:	c = 7; break;
    case TK3_SENSOR_CURRENT:	c = 6; break;
    case TK3_SENSOR_MCU_TEMP:	c = 8; break;
    case TK3_SENSOR_PCB_TEMP:	c = 3; break;

    default:
      channel = TK3_SENSOR_BATTERY;
      c = 7;
      break;
  }

  ATOMIC_BLOCK(ATOMIC_RESTORESTATE) {
    ADMUX = (ADMUX & 0xf0) | c;
    ADCSRA |= (1 << ADEN) | (1 << ADSC) | (1 << ADIF);

    NONATOMIC_BLOCK(NONATOMIC_FORCEOFF) {
      while (ADCSRA & (1 << ADSC)) /* empty body */;
    }

    adc = ADC;
    ADCSRA &= ~(1 << ADEN);
    if (adc < adc_offset) adc = 0; else adc -= adc_offset;

    switch(channel) {
      case TK3_SENSOR_BATTERY:
        adc = adc * 30; /* ~ (18000 + 680)/680 * 1.1/1024 * 1000 */
        tk3_sensors.battery = adc;
        break;

      case TK3_SENSOR_CURRENT:
        adc *= 230; /* ~ 4.6 miliohms */
        tk3_sensors.current = (tk3_sensors.current + adc)/2;
        if (tk3_sensors.current > tk3_sensors.peak_current)
          tk3_sensors.peak_current = tk3_sensors.current;
        break;

      case TK3_SENSOR_MCU_TEMP:
        adc = adc * 11 - 2865;
        tk3_sensors.mcu_temp = adc;
        break;

      case TK3_SENSOR_PCB_TEMP:
        adc = adc * 5 / 16;
        tk3_sensors.pcb_temp = adc;
        break;
    }
  }

  channel++;
}
