/*
 * Copyright (c) 2014-2017,2024 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *					Anthony Mallet on Tue Sep 23 2014
 */
#ifndef H_MKBL
#define H_MKBL

#include <stdarg.h>
#include <stdint.h>
#include <stdbool.h>

/* sensor */
struct tk3_sensors {
  uint16_t battery;
  uint16_t current, peak_current;
  uint16_t mcu_temp;
  uint16_t pcb_temp;
};
extern struct tk3_sensors tk3_sensors;

enum __attribute__((__packed__)) tk3_sensor_id {
  TK3_SENSOR_BATTERY,
  TK3_SENSOR_CURRENT,
  TK3_SENSOR_MCU_TEMP,
  TK3_SENSOR_PCB_TEMP
};

int		tk3_sensor_init(void);
void		tk3_sensor_update(enum tk3_sensor_id id);

/* motor */
enum __attribute__((__packed__)) mkbldir {
  MKBL_CLOCKWISE,
  MKBL_ANTICLOCKWISE
};

struct tk3_motor {
  volatile unsigned align:1;
  volatile unsigned starting:1;
  volatile unsigned spinning:1;
  volatile unsigned emerg:1;
  volatile unsigned servo:1;
  volatile unsigned reverse:1;
  volatile unsigned braking:1;

  volatile tk3_time stamp;
  volatile int8_t sign;
  volatile uint16_t period;
  volatile uint16_t target;
  volatile uint16_t pwm;

  uint16_t max_period;		/* min velocity fitting on 16 bits */
  uint16_t max75_period;	/* 25% above min velocity (~20Hz) */

  int16_t smc_bias;
  int16_t smc_gain;
  int16_t smc_err;
};
extern struct tk3_motor tk3_motor;

int		tk3_motor_init(enum mkbldir direction);
int		tk3_motor_test(void);
void		tk3_motor_start(void);
void		tk3_motor_stop(void);
int		tk3_motor_tone(uint16_t freq);
void		tk3_motor_pwm(int8_t sign, uint16_t u);
void		tk3_motor_period(int8_t sign, uint16_t u);


/* settings */
#define TK3SET_MOTORID		"id"
#define TK3SET_ROTATION		"rotation"
#define TK3SET_POLES		"poles"
#define TK3SET_MCURRENT		"max-current"
#define TK3SET_WDOG		"watchdog"
#define TK3SET_DISARM		"disarm"
#define TK3SET_STARTACC		"start-acc"
#define TK3SET_STARTPWM		"start-pwm"
#define TK3SET_STARTDEL		"start-delay"
#define TK3SET_STARTPER		"start-period"
#define TK3SET_POST		"post"

extern struct tk3_settings {
  uint8_t motor_id;
  uint8_t *motor_id_addr;

  enum mkbldir rotation;
  uint8_t motor_period_mul;
  uint16_t mcurrent;
  uint32_t wdog;
  uint32_t disarm;
  uint32_t uart;
  uint16_t sdelay;
  uint16_t speriod;
  uint8_t spwm;
  uint8_t sacc;
  uint8_t post;
  int8_t clk_adj;
  uint8_t *clk_adj_addr;
} settings;


/* logging */
void	tk3_log_velocity(enum tk3_channel channel);
void	tk3_log_current(enum tk3_channel channel);

void	tk3_logdef_motor(enum tk3_channel channel, uint32_t period,
                tk3_time date);
void	tk3_log_motor(tk3_time date);

void	tk3_log_sensors(enum tk3_channel channel);
void	tk3_log_control(enum tk3_channel channel);

void	tk3_logdef_battery(enum tk3_channel channel, uint32_t period,
                tk3_time date);
void	tk3_log_battery(tk3_time date);


/* misc */
void		red_led_abort(void);
void		flash_red_led_abort(void);
void		dim_red_led_abort(void);

#endif /* H_MKBL */
