/*
 * Copyright (c) 2014-2018,2024 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Tue Sep 23 2014
 */
#include "acheader.h"

#include <stdlib.h>
#include <string.h>

#include <avr/cpufunc.h>
#include <avr/pgmspace.h>
#include <avr/eeprom.h>

#include "common/tk3-mikrokopter.h"
#include "mkbl.h"

/* default settings, stored in the MCU EEPROM. The format is a list
 * of nul-terminated strings, interpreted as a key/value/descr */
static const char tk3_dparams[] EEMEM __attribute__((used)) = {
  TK3SET_MAGIC "\0"
  TK3SET_MOTORID "\0"	"\0\0\0\0"		"1-15 or 0 for jumpers" "\0"
  TK3SET_ROTATION "\0"	"\0\0\0\x1"		"+1 CCW/-1 CW" "\0"
  TK3SET_POLES "\0"	"\0\0\0\xe"		"motor # poles" "\0"
  TK3SET_MCURRENT "\0"	"\0\0\x23\x28"		"current limit, mA" "\0"
  TK3SET_WDOG "\0"	"\0\0\x1\xf4"		"propeller watchdog, ms" "\0"
  TK3SET_DISARM "\0"	"\0\0\0\xa"		"disarm watchdog 0-300, s" "\0"
  TK3SET_STARTACC "\0"	"\0\0\0\x14"		"start pwm delta, %" "\0"
  TK3SET_STARTPWM "\0"	"\0\0\0\x6"		"start pwm, %" "\0"
  TK3SET_STARTDEL "\0"	"\0\0\x1\x5e"		"start delay, ms" "\0"
  TK3SET_STARTPER "\0"	"\0\0\x40\x00"		"start period, us" "\0"
  TK3SET_POST "\0"	"\0\0\0\x1"		"power on self tests" "\0"
  TK3SET_UART "\0"	"\0\x7\xa1\x20"		"uart baud rate" "\0"
  TK3SET_CLKADJ "\0"	"\0\0\0\0"		"internal oscilator offset" "\0"
};

struct tk3_settings settings;

static void	tk3_load_settings(struct tk3_settings *s);
static void	tk3_update_leds(tk3_time date);

/* intialization tones */
static const uint16_t tones[] = {
  523, 587, 659, 698, 783, 880, 987
};


/* --- main ---------------------------------------------------------------- */

int
main()
{
  tk3_time date;
  struct tk3_iorecv *msg;

  /* red led on */
  DDRD |= (1 << DDD7);
  PORTD |= (1 << PD7);

  /* load settings */
  tk3_load_settings(&settings);

  if (!settings.motor_id) {
    int16_t addr = -1;

    /* I2C address via board jumbers */
    DDRB &= ~((1 << DDB0) | (1 << DDB6) | (1 << DDB7));
    PORTB |= ((1 << PB0) | (1 << PB6) | (1 << PB7));
    while(addr != PINB) addr = PINB; /* synchronize pin output */

    settings.motor_id = 1 +
      ((!(addr & (1 << PINB0)) << 2) |
       (!(addr & (1 << PINB6)) << 1) |
       (!(addr & (1 << PINB7)) << 0));
    PORTB = 0;

    if (settings.motor_id_addr) /* update eeprom */
      tk3_settings_update(settings.motor_id_addr, settings.motor_id);
  }

  /* initialize hardware */
  if (tk3_clock_init(settings.clk_adj, settings.clk_adj_addr))
    dim_red_led_abort();

  if (tk3_uart_init(0, settings.uart))
    dim_red_led_abort();
  tk3_twi_init(TK3TWI_BASEADDR + settings.motor_id);

  if (tk3_sensor_init())
    dim_red_led_abort();

  if (tk3_motor_init(settings.rotation))
    dim_red_led_abort();
  if (tk3_motor_test())
    dim_red_led_abort();

  /* red led off */
  DDRD |= (1 << DDD7);
  PORTD &= ~(1 << PD7);

  /* wait 2s and do hardware testing */
  if (settings.post) {
    date = 2000000 + 500000 * settings.motor_id;
    tk3_clock_delay(date);

    if (tk3_motor_tone(tones[settings.motor_id-1]) ||
        tk3_motor_tone(tones[settings.motor_id]))
      dim_red_led_abort();
  }

  /* main loop */
  while(1) {
    date = tk3_clock_gettime();

    if (!tk3_motor.starting && !tk3_motor.spinning)
      tk3_sensor_update(-1);

    /* process received messages */
    while ((msg = tk3_recv())) {
      uint8_t cmd = msg->data[0];
      switch(cmd) {
        case 't': /* clock tick - µs */
          if (msg->len == 5) {
            tk3_time d =
              ((uint32_t)msg->data[1] << 24) |
              ((uint32_t)msg->data[2] << 16) |
              ((uint32_t)msg->data[3] << 8) |
              msg->data[4];

            if (tk3_clock_reftick(date, d)) {
              /* reply if clock rate changed */
              tk3_log(msg->state.channel,
                      PSTR("T%1%1"), settings.motor_id, OSCCAL);
            }
          }
          break;

        case 'p': /* PWM */
          if (msg->len == 3) {
            int16_t p =
              ((uint16_t)msg->data[1] << 8) |
              ((uint16_t)msg->data[2]);

            tk3_motor_pwm(p<0 ? -1:1, p<0 ? -p:p);
          }
          break;

        case 'q': /* PWM array */
          if (msg->len >= 1 + 2 * settings.motor_id) {
            int16_t p =
              ((uint16_t)msg->data[2 * settings.motor_id - 1] << 8) |
              ((uint16_t)msg->data[2 * settings.motor_id]);
            tk3_motor_pwm(p<0 ? -1:1, p<0 ? -p:p);
          }
          break;

        case 'v': /* revolution period - µs */
          if (msg->len == 3) {
            int16_t t =
              ((uint16_t)msg->data[1] << 8) |
              ((uint16_t)msg->data[2]);

            tk3_motor_period(t<0 ? -1:1, (t<0 ? -t:t)*2);
          }
          break;

        case 'w': /* revolution period array - µs */
          if (msg->len >= 1 + 2 * settings.motor_id) {
            int16_t t =
              ((uint16_t)msg->data[2 * settings.motor_id - 1] << 8) |
              ((uint16_t)msg->data[2 * settings.motor_id]);

            tk3_motor_period(t<0 ? -1:1, (t<0 ? -t:t)*2);
          }
          break;

        case 'g': /* startup */
          if (msg->len > 1 && msg->data[1] != settings.motor_id) break;

          if (!tk3_motor.spinning && !tk3_motor.starting) {
            tk3_motor_stop();
            tk3_motor_start();
          }
          break;

        case 'x': /* stop */
          if (msg->len > 1 && msg->data[1] != settings.motor_id) break;

          tk3_motor_stop();
          break;

        case 's': /* log motor velocity */
          tk3_log_velocity(msg->state.channel);
          break;

        case 'a': /* log motor current */
          tk3_log_current(msg->state.channel);
          break;

        case 'm': /* log motor data */
          if (msg->len == 5) {
            tk3_time p =
              ((uint32_t)msg->data[1] << 24) |
              ((uint32_t)msg->data[2] << 16) |
              ((uint32_t)msg->data[3] << 8) |
              msg->data[4];

            tk3_logdef_motor(msg->state.channel, p, date);
          }
          break;

        case 'k': /* log controller */
          tk3_log_control(msg->state.channel);
          break;

        case 'd': /* log sensors */
          tk3_log_sensors(msg->state.channel);
          break;

        case 'b': /* battery level */
          if (msg->len == 5) {
            tk3_time p =
              ((uint32_t)msg->data[1] << 24) |
              ((uint32_t)msg->data[2] << 16) |
              ((uint32_t)msg->data[3] << 8) |
              msg->data[4];

            tk3_logdef_battery(msg->state.channel, p, date);
          }
          break;

        case '~': /* beep */
          if (!tk3_motor.starting && !tk3_motor.spinning) {
            if (msg->len == 3) {
              uint16_t f =
                ((uint16_t)msg->data[1] << 8) |
                msg->data[2];

              tk3_motor_tone(f);
            }
          }
          break;

        case '?': /* id */
          if (msg->len == 1) {
            tk3_log(msg->state.channel,
                    PSTR("?%1mkbl" PACKAGE_VERSION), settings.motor_id);
          }
          break;
      }
    }

    /* data logging */
    tk3_log_motor(date);
    tk3_log_battery(date);

    /* flush pending data */
    tk3_send();

    /* blink leds */
    tk3_update_leds(date);
  }

  return 0;
}


/* --- load_settings ------------------------------------------------------- */

static void
tk3_load_settings(struct tk3_settings *s)
{
  struct tk3_settings set;
  int32_t value;
  char key[16];

  if (tk3_settings_init()) dim_red_led_abort();

  /* flags for parameters that are set */
  set.motor_id = 0;
  set.rotation = 0;
  set.mcurrent = 0;
  set.wdog = 0;
  set.disarm = 0;
  set.spwm = 0;
  set.sdelay = 0;
  set.speriod = 0;
  set.motor_period_mul = 0;
  set.post = 0;
  set.uart = 0;
  set.clk_adj = 0;

  /* loop in EEPROM */
  while(!tk3_settings_next(key, sizeof(key), &value)) {
    if (!strcmp(key, TK3SET_MOTORID)) {
      s->motor_id = value;
      set.motor_id = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_ROTATION))) {
      if (value > 0)
        s->rotation = MKBL_ANTICLOCKWISE;
      else if (value < 0)
        s->rotation = MKBL_CLOCKWISE;
      else
        dim_red_led_abort();
      set.rotation = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_MCURRENT))) {
      if (value > 0 && value <= 35000)
        s->mcurrent = value;
      else
        dim_red_led_abort();
      set.mcurrent = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_WDOG))) {
      if (value > 0 && value < 60000)
        s->wdog = value * 1000;
      else
        dim_red_led_abort();
      set.wdog = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_DISARM))) {
      if (value > 0 && value < 301)
        s->disarm = value * 1000000;
      else if (value <= 0)
        s->disarm = -1U;
      else
        dim_red_led_abort();
      set.disarm = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_STARTACC))) {
      if (value > 0 && value < 100)
        s->sacc = value;
      else
        dim_red_led_abort();
      set.sacc = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_STARTPWM))) {
      if (value > 0 && value < 100)
        s->spwm = value;
      else
        dim_red_led_abort();
      set.spwm = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_STARTDEL))) {
      if (value >= 0)
        s->sdelay = value;
      else
        dim_red_led_abort();
      set.sdelay = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_STARTPER))) {
      if (value > 0)
        s->speriod = value;
      else
        dim_red_led_abort();
      set.speriod = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_POLES))) {
      if (value > 0)
        s->motor_period_mul = value / 2 * 6;
      else
        dim_red_led_abort();
      set.motor_period_mul = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_POST))) {
      s->post = !!value;
      set.post = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_UART))) {
      s->uart = value;
      set.uart = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_CLKADJ))) {
      s->clk_adj = value;
      set.clk_adj = 1;

    } else
      dim_red_led_abort();
  }

  /* check all parameters are set */
  if (!(set.motor_id && set.rotation && set.mcurrent && set.wdog &&
        set.disarm && set.sacc && set.spwm && set.sdelay && set.speriod &&
        set.motor_period_mul && set.post && set.uart && set.clk_adj))
    dim_red_led_abort();

  s->motor_id_addr = tk3_settings_addr(TK3SET_MOTORID);
  s->clk_adj_addr = tk3_settings_addr(TK3SET_CLKADJ);
}


/* --- tk3_update_leds ----------------------------------------------------- */

static void __attribute__((used))
tk3_update_leds(tk3_time date)
{
  /* dim red led blink for motor emergency */
  if (tk3_motor.emerg) {
    if (date & (1UL<<19)) /* ~524ms */
      DDRD |= (1 << DDD7);
    else
      DDRD &= ~(1 << DDD7);
  } else {
    /* blink red led for comm issues */
    if (tk3_comm_errs || tk3_comm_warns)
      DDRD &= ~(1 << DDD7);
    else
      DDRD |= (1 << DDD7);
  }
}


/* --- *_abort ------------------------------------------------------------- */

void
red_led_abort(void)
{
  DDRD |= (1 << DDD7);
  PORTD |= (1 << PD7);
  abort();
}

void
flash_red_led_abort(void)
{
  tk3_time t;

  DDRD |= (1 << DDD7);

  t = tk3_clock_gettime();
  while(1) {
    PORTD ^= (1 << PD7);

    t += 500000;
    tk3_clock_delay(t);
  }
}

void
dim_red_led_abort(void)
{
  DDRD &= ~(1 << DDD7);
  PORTD &= ~(1 << PD7);
  abort();
}
