/*
 * Copyright (c) 2015-2017 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Fri Jan 30 2015
 */
#include "acheader.h"

#include <avr/pgmspace.h>
#include <util/atomic.h>

#include "common/tk3-mikrokopter.h"
#include "mkbl.h"


/* --- tk3_log_state ------------------------------------------------------- */

static uint8_t
tk3_log_state(void)
{
  uint8_t state;

  state = settings.motor_id & 0xf;
  if (tk3_motor.starting) state |= 1<<4;
  if (tk3_motor.spinning) state |= 1<<5;
  if (tk3_motor.servo) state |= 1<<6;
  if (tk3_motor.emerg) state |= 1<<7;

  return state;
}


/* --- tk3_log_velocity ---------------------------------------------------- */

void
tk3_log_velocity(enum tk3_channel channel)
{
  int8_t sign;
  uint8_t state;
  uint16_t period;

  ATOMIC_BLOCK(ATOMIC_RESTORESTATE) {
    state = tk3_log_state();
    if (tk3_motor.starting) state |= 1<<4;
    if (tk3_motor.spinning) state |= 1<<5;
    if (tk3_motor.servo) state |= 1<<6;
    if (tk3_motor.emerg) state |= 1<<7;
    sign = tk3_motor.sign;
    period = tk3_motor.period;
  }

  if (period > tk3_motor.max_period)
    period = 0;
  else
    period *= settings.motor_period_mul;
  period /= 2;
  if (sign < 0) period = -period;

  tk3_log(channel, PSTR("S%1%2"), state, period);
}


/* --- tk3_log_current ----------------------------------------------------- */

void
tk3_log_current(enum tk3_channel channel)
{
  uint8_t state;
  uint16_t current;

  ATOMIC_BLOCK(ATOMIC_RESTORESTATE) {
    state = tk3_log_state();
    current = tk3_sensors.current;
  }

  tk3_log(channel, PSTR("A%1%2"), state, current);
}


/* --- tk3_log_motor ------------------------------------------------------- */

static uint32_t motor_log_period[TK3_CHANNEL_MAX];
static uint32_t motor_log_date[TK3_CHANNEL_MAX];
static uint8_t motor_log_id[TK3_CHANNEL_MAX];

void
tk3_logdef_motor(enum tk3_channel channel, uint32_t period, tk3_time date)
{
  motor_log_period[channel] = period;
  motor_log_date[channel] = date;
  motor_log_id[channel] = ~0;
}

void
tk3_log_motor(tk3_time date)
{
  uint8_t c, i;
  uint8_t state;
  int8_t sign;
  uint16_t period, pwm, current;
  tk3_time last;

  for (c = 0; c < TK3_CHANNEL_MAX; c++) {
    if (!motor_log_period[c]) continue;
    last = motor_log_date[c];
    if ((date - last) & 0x80000000U) continue;

    i = motor_log_id[c];
    do {
      last += motor_log_period[c];
      i++;
    } while ((last - date) & 0x80000000U);
    motor_log_date[c] = last;
    motor_log_id[c] = i;

    ATOMIC_BLOCK(ATOMIC_RESTORESTATE) {
      state = tk3_log_state();
      sign = tk3_motor.sign;
      period = tk3_motor.period;
      pwm = tk3_motor.pwm;
      current = tk3_sensors.peak_current;
      tk3_sensors.peak_current = tk3_sensors.current;
    }

    if (period > tk3_motor.max_period) {
      if (tk3_motor.spinning)
        period = 0xffff;
      else
        period = 0;
    } else
      period *= settings.motor_period_mul;
    period /= 2;
    if (sign < 0) {
      period = -period;
      pwm = -pwm;
    }

    tk3_log(c, PSTR("M%1%1%2%2%2"), i, state, period, pwm, current);
  }
}


/* --- tk3_log_sensors ----------------------------------------------------- */

void
tk3_log_sensors(enum tk3_channel channel)
{
  uint8_t state;
  uint16_t battery, current, mcu, pcb;

  ATOMIC_BLOCK(ATOMIC_RESTORESTATE) {
    state = tk3_log_state();
    battery = tk3_sensors.battery;
    current = tk3_sensors.current;
    mcu = tk3_sensors.mcu_temp;
    pcb = tk3_sensors.pcb_temp;
  }

  tk3_log(channel, PSTR("D%1%2%2%2%2"), state, battery, current, mcu, pcb);
}


/* --- tk3_log_control ----------------------------------------------------- */

void
tk3_log_control(enum tk3_channel channel)
{
  uint8_t state;
  int8_t sign;
  uint16_t target;
  int16_t bias, gain, err;

  ATOMIC_BLOCK(ATOMIC_RESTORESTATE) {
    state = tk3_log_state();
    sign = tk3_motor.sign;
    target = tk3_motor.target;
    bias = tk3_motor.smc_bias;
    gain = tk3_motor.smc_gain;
    err = (tk3_motor.smc_err/2) & 0xffff;
  }

  if (target > tk3_motor.max_period)
    target = 0xffff;
  else
    target *= settings.motor_period_mul;
  target /= 2;
  if (sign < 0) target = -target;

  tk3_log(channel, PSTR("K%1%2%2%2%2"), state, target, bias, gain, err);
}


/* --- tk3_log_battery ----------------------------------------------------- */

static uint32_t battery_log_period[TK3_CHANNEL_MAX];
static uint32_t battery_log_date[TK3_CHANNEL_MAX];
static uint8_t battery_log_id[TK3_CHANNEL_MAX];

void
tk3_logdef_battery(enum tk3_channel channel, uint32_t period, tk3_time date)
{
  battery_log_period[channel] = period;
  battery_log_date[channel] = date;
  battery_log_id[channel] = ~0;
}

void
tk3_log_battery(tk3_time date)
{
  uint8_t c, i;
  uint16_t battery;
  tk3_time last;

  for (c = 0; c < TK3_CHANNEL_MAX; c++) {
    if (!battery_log_period[c]) continue;
    last = battery_log_date[c];
    if ((date - last) & 0x80000000U) continue;

    i = battery_log_id[c];
    do {
      last += battery_log_period[c];
      i++;
    } while ((last - date) & 0x80000000U);
    battery_log_date[c] = last;
    battery_log_id[c] = i;


    ATOMIC_BLOCK(ATOMIC_RESTORESTATE) {
      battery = tk3_sensors.battery;
    }

    tk3_log(c, PSTR("B%1%2"), i, battery);
  }
}
