/*
 * Copyright (c) 2017 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Tue Jan 31 2017
 */
#include "acheader.h"

#include <avr/eeprom.h>
#include <avr/pgmspace.h>

#include <stdlib.h>
#include <string.h>

#include "common/tk3-mikrokopter.h"
#include "flymu.h"

/* default settings, stored in the MCU EEPROM. The format is a list
 * of nul-terminated strings, interpreted as a key/value/descr */
static const char tk3_dparams[] EEMEM __attribute__((used)) = {
  TK3SET_MAGIC "\0"
  TK3SET_VERSION "\0"	"\0\0\0\0"	"board version, 0 for auto" "\0"
  TK3SET_ROBOT "\0"	"\0\0\0\0"	"robot number" "\0"
  TK3SET_MOTORS "\0"	"\0\0\0\4"	"motors controlled" "\0"
  TK3SET_UART "\0"	"\0\x7\xa1\x20"	"uart baud rate" "\0"
};

struct fmu_settings settings;


/* --- load_settings ------------------------------------------------------- */

void
fmu_load_settings(void)
{
  struct fmu_settings set;
  int32_t value;
  char key[16];

  /* load eeprom settings */
  if (tk3_settings_init()) abort();

  /* flags for parameters that are set */
  set.version = 0;
  set.robot_id = 0;
  set.motors = 0;
  set.uart = 0;

  /* loop in EEPROM */
  while(!tk3_settings_next(key, sizeof(key), &value)) {
    if (!strcmp_P(key, PSTR(TK3SET_VERSION))) {
      if (!value) {
        /* auto-detect board version */
        DDRB = 0; PORTB = 0x3; /* PB0-1 input, pullup */
        DDRD = 0; PORTD = 0x80; /* PD7 input, pullup */

        value = 10;
        value += PINB & 0x3; /* ver0-1 */
        if (PIND & 0x80) value += 10; /* ver2 */

        PORTB &= ~0x3;
        PORTD &= ~0x80;
        tk3_settings_update(tk3_settings_addr(TK3SET_VERSION), value);
      }
      settings.version = value;
      set.version = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_ROBOT))) {
      settings.robot_id = value;
      set.robot_id = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_MOTORS))) {
      settings.motors = value;
      set.motors = 1;

    } else if (!strcmp_P(key, PSTR(TK3SET_UART))) {
      settings.uart = value;
      set.uart = 1;

    } else {
      fmu_led(FMULED_ERR, 0xff);
      while(1) fmu_led_update(tk3_clock_gettime());
    }
  }

  if (!(set.version && set.robot_id && set.motors && set.uart)) {
    fmu_led(FMULED_ERR, 0xff);
    while(1) fmu_led_update(tk3_clock_gettime());
  }
}
