/*
 * Copyright (c) 2017 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Tue Feb  7 2017
 */
#include "acheader.h"

#include <avr/interrupt.h>
#include <avr/pgmspace.h>

#include <stdbool.h>
#include <stdlib.h>

#include <util/atomic.h>
#include <util/twi.h>

#include "common/tk3-mikrokopter.h"
#include "flymu.h"


/* sensors I2C bus # */
#define SEN_TW(x)		TK3_TWI_REG(x, /* 0, empty */)

/* sensors I2C addr */
#define FMUSEN_AG_ADDR		(0x6b)
#define FMUSEN_M_ADDR		(0x1e)
#define FMUSEN_P_ADDR		(0x77)

/* accelererometer / gyroscope registers */
#define FMUSEN_AG_WHO_AM_I	(0x0f)

#define FMUSEN_AG_CTRL_REG1_G	(0x10)
#define FMUSEN_AG_ODR_G2		(1<<7)
#define FMUSEN_AG_ODR_G1		(1<<6)
#define FMUSEN_AG_ODR_G0		(1<<5)
#define FMUSEN_AG_FS_G1			(1<<4)
#define FMUSEN_AG_FS_G0			(1<<3)
#define FMUSEN_AG_BW_G1			(1<<1)
#define FMUSEN_AG_BW_G0			(1<<0)

#define FMUSEN_AG_OUT_G		(0x18) /* X, Y, Z (3 * int16_t) */

#define FMUSEN_AG_CTRL_REG6_XL	(0x20)
#define FMUSEN_AG_ODR_XL2		(1<<7)
#define FMUSEN_AG_ODR_XL1		(1<<6)
#define FMUSEN_AG_ODR_XL0		(1<<5)
#define FMUSEN_AG_FS1_XL		(1<<4)
#define FMUSEN_AG_FS0_XL		(1<<3)
#define FMUSEN_AG_BW_SCAL_ODR		(1<<2)
#define FMUSEN_AG_BW_XL1		(1<<1)
#define FMUSEN_AG_BW_XL0		(1<<0)

#define FMUSEN_AG_CTRL_REG8	(0x22)
#define FMUSEN_AG_BOOT			(1<<7)
#define FMUSEN_AG_BDU			(1<<6)
#define FMUSEN_AG_H_LACTIVE		(1<<5)
#define FMUSEN_AG_PP_OD			(1<<4)
#define FMUSEN_AG_SIM			(1<<3)
#define FMUSEN_AG_IF_ADD_INC		(1<<2)
#define FMUSEN_AG_BLE			(1<<1)
#define FMUSEN_AG_SW_RESET		(1<<0)

#define FMUSEN_AG_STATUS_REG_G	(0x17)
#define FMUSEN_AG_STATUS_REG_A	(0x27)
#define FMUSEN_AG_IG_XL			(1<<6)
#define FMUSEN_AG_IG_G			(1<<5)
#define FMUSEN_AG_IG_INACT		(1<<4)
#define FMUSEN_AG_BOOT_STATUS		(1<<3)
#define FMUSEN_AG_TGA			(1<<2)
#define FMUSEN_AG_GDA			(1<<1)
#define FMUSEN_AG_XLDA			(1<<0)

/* magnetometer registers */
#define FMUSEN_M_WHO_AM_I	(0x0f)

/* barometer registers */
#define FMUSEN_P_I2C_SLAVE	(0x88)


/* --- local data ---------------------------------------------------------- */

enum __attribute__((__packed__)) fmu_sensor_id {
  FMUSEN_AG = 1<<0,	/* accelererometer / gyroscope */
  FMUSEN_M = 1<<1,	/* magnetometer */
  FMUSEN_P = 1<<2,	/* barometer */
  FMUSEN_BAT = 1<<3,	/* battery */
};

static uint8_t fmu_senp;
volatile struct fmu_sensors fmu_sensors;

static void	fmu_sensor_update_init(void);

static uint8_t	fmu_i2c_read(uint8_t addr, uint8_t reg, uint8_t *bytes,
                             uint8_t count, void (*done)(volatile uint8_t *));
static uint8_t	fmu_i2c_write(uint8_t addr, uint8_t reg, uint8_t byte,
                              void (*done)(volatile uint8_t *));


/* --- fmu_sensor_init ----------------------------------------------------- */

uint8_t
fmu_sensor_init(void)
{
  uint8_t d = 0;
  uint8_t s;

  /* enable ADC: AVCC ref, ADC3 (PC3), clock/128, free running */
  ADMUX = (1 << REFS0) | (1 << MUX1) | (1 << MUX0);
  DIDR0 |= (1 << ADC3D);
  ADCSRA = (1 << ADEN) | (1 << ADPS2) | (1 << ADPS1) | (1 << ADPS0);
  ADCSRB = 0;

  /* enable I2C bus, 500kHz */
  SEN_TW(SR) &= ~(( 1 << TWPS1) | (1 << TWPS0)) /* prescaler 1 */;
  SEN_TW(BR_) = ((F_CPU / 500000) - 16) / (2 * 1);
  SEN_TW(CR) = (1 << TWEN);

  /* callback ISR on PE3 */
  DDRE |= (1 << DDE3);
  PCICR |= (1 << PCIE3);
  PCMSK3 |= (1 << PCINT27);

  /* detect I2C sensors */
  fmu_senp = FMUSEN_BAT;

  s = fmu_i2c_read(FMUSEN_AG_ADDR, FMUSEN_AG_WHO_AM_I, &d, 1, NULL);
  if (!s && d == 0x68)
    fmu_senp |= FMUSEN_AG;
  else abort();

  s = fmu_i2c_read(FMUSEN_M_ADDR, FMUSEN_M_WHO_AM_I, &d, 1, NULL);
  if (!s && d == 0x3d)
    fmu_senp |= FMUSEN_M;

  s = fmu_i2c_read(FMUSEN_P_ADDR, FMUSEN_P_I2C_SLAVE, &d, 1, NULL);
  if (!s && (d & 0x7f) == FMUSEN_P_ADDR)
    fmu_senp |= FMUSEN_P;

  /* configure accelererometer/gyroscope */
  if (fmu_senp & FMUSEN_AG) {
    uint8_t u;

    s = fmu_i2c_write(
        FMUSEN_AG_ADDR, FMUSEN_AG_CTRL_REG8,
        FMUSEN_AG_SW_RESET /* reset */, NULL);
    if (s) return s;
    do {
      s = fmu_i2c_read(FMUSEN_AG_ADDR, FMUSEN_AG_STATUS_REG_G, &u, 1, NULL);
      if (s) return s;
    } while(u & FMUSEN_AG_BOOT_STATUS);

    s =
      fmu_i2c_write(
        FMUSEN_AG_ADDR, FMUSEN_AG_CTRL_REG1_G,
        FMUSEN_AG_ODR_G2|FMUSEN_AG_ODR_G1 /* 952Hz */ |
        FMUSEN_AG_FS_G1|FMUSEN_AG_FS_G0 /* 2000⁰/s */ |
        FMUSEN_AG_BW_G1|FMUSEN_AG_BW_G0 /* cutoff 100Hz */,
        NULL) ||
      fmu_i2c_write(
        FMUSEN_AG_ADDR, FMUSEN_AG_CTRL_REG6_XL,
        FMUSEN_AG_ODR_XL2|FMUSEN_AG_ODR_XL1 /* 952Hz */ |
        FMUSEN_AG_FS1_XL|FMUSEN_AG_FS0_XL /* 8g */,
        NULL) ||
      fmu_i2c_write(
        FMUSEN_AG_ADDR, FMUSEN_AG_CTRL_REG8,
        FMUSEN_AG_BDU /* block data update */ |
        FMUSEN_AG_IF_ADD_INC /* auto incr address */,
        NULL);
    if (s) return s;
  }

  fmu_sensor_update_init();
  return 0;
}


/* --- fmu_sensor_update --------------------------------------------------- */

static void	fmu_sensor_update_a(volatile uint8_t *data);
static void	fmu_sensor_update_g(volatile uint8_t *data);

static void
fmu_sensor_update_init()
{
  /* start ADC */
  ADCSRA |= (1 << ADSC) | (1 << ADIF) | (1 << ADIE);

  /* start I2C */
  if (fmu_senp & FMUSEN_AG)
    fmu_i2c_read(
      FMUSEN_AG_ADDR, FMUSEN_AG_STATUS_REG_A, NULL, 7, fmu_sensor_update_a);
}

static void
fmu_sensor_update_a(volatile uint8_t *data)
{
  if (data && (data[0] & FMUSEN_AG_XLDA)) {
    fmu_sensors.acc_x = - *(int16_t *)(data + 3);
    fmu_sensors.acc_y = - *(int16_t *)(data + 1);
    fmu_sensors.acc_z =   *(int16_t *)(data + 5);
  }

  fmu_i2c_read(
    FMUSEN_AG_ADDR, FMUSEN_AG_STATUS_REG_G, NULL, 7, fmu_sensor_update_g);
}

static void
fmu_sensor_update_g(volatile uint8_t *data)
{
  if (data && (data[0] & FMUSEN_AG_GDA)) {
    fmu_sensors.gyro_roll  = - *(int16_t *)(data+3);
    fmu_sensors.gyro_pitch = - *(int16_t *)(data+1);
    fmu_sensors.gyro_yaw   =   *(int16_t *)(data+5);
  }

  fmu_i2c_read(
    FMUSEN_AG_ADDR, FMUSEN_AG_STATUS_REG_A, NULL, 7, fmu_sensor_update_a);
}


/* --- fmu_i2c ------------------------------------------------------------- */

enum __attribute__((__packed__)) fmu_i2c_xfer {
  FMU_I2C_DONE = 0,
  FMU_I2C_ERR = 1,
  FMU_I2C_READ,
  FMU_I2C_WRITE
};

struct fmu_i2c_req {
  volatile enum fmu_i2c_xfer xfer;
  volatile uint8_t addr, reg, len;
  volatile uint8_t iobuf[8];
  void (*done)(volatile uint8_t *);
};
static struct fmu_i2c_req i2c_req;

static uint8_t
fmu_i2c_read(uint8_t addr, uint8_t reg, uint8_t *bytes, uint8_t count,
             void (*done)(volatile uint8_t *))
{
  switch (i2c_req.xfer) {
    case FMU_I2C_READ: case FMU_I2C_WRITE: return 2;
    default: break;
  }

  i2c_req.xfer = FMU_I2C_READ;
  i2c_req.addr = addr;
  i2c_req.reg = reg;
  i2c_req.len = count;
  i2c_req.done = done;
  SEN_TW(CR) = (1 << TWINT) | (1 << TWEN) | (1 << TWIE) | (1 << TWSTA);

  if (done) return 0;

  while(i2c_req.xfer == FMU_I2C_READ) /* wait */;

  for(; count; count--)
    bytes[count-1] = i2c_req.iobuf[count-1];
  return i2c_req.xfer;
}

static uint8_t
fmu_i2c_write(uint8_t addr, uint8_t reg, uint8_t byte,
              void (*done)(volatile uint8_t *))
{
  switch (i2c_req.xfer) {
    case FMU_I2C_READ: case FMU_I2C_WRITE: return 2;
    default: break;
  }

  i2c_req.xfer = FMU_I2C_WRITE;
  i2c_req.addr = addr;
  i2c_req.reg = reg;
  i2c_req.len = 1;
  i2c_req.iobuf[0] = byte;
  i2c_req.done = done;
  SEN_TW(CR) = (1 << TWINT) | (1 << TWEN) | (1 << TWIE) | (1 << TWSTA);

  if (done) return 0;

  while(i2c_req.xfer == FMU_I2C_WRITE) /* wait */;
  if (i2c_req.xfer != FMU_I2C_DONE) return 1;
  return i2c_req.xfer;
}

ISR(TWI_vect)
{
  static uint8_t p;

  switch (SEN_TW(SR) & TW_STATUS_MASK) {
    case TW_START:
      /* start condition transmitted */
      SEN_TW(DR) = (i2c_req.addr << 1) | TW_WRITE;
      SEN_TW(CR) = (1 << TWINT) | (1 << TWEN) | (1 << TWIE);
      break;

    case TW_REP_START:
      /* repeated start condition transmitted */
      SEN_TW(DR) = (i2c_req.addr << 1) | TW_READ;
      SEN_TW(CR) = (1 << TWINT) | (1 << TWEN) | (1 << TWIE);
      break;

    case TW_MT_SLA_ACK:
      /* SLA+W transmitted, ACK received */
      SEN_TW(DR) = i2c_req.reg;
      SEN_TW(CR) = (1 << TWINT) | (1 << TWEN) | (1 << TWIE);
      p = 0;
      break;

    case TW_MR_SLA_ACK:
      /* SLA+R transmitted, ACK received */
      if (i2c_req.len > 1)
        SEN_TW(CR) = (1 << TWINT) | (1 << TWEN) | (1 << TWIE) | (1 << TWEA);
      else
        SEN_TW(CR) = (1 << TWINT) | (1 << TWEN) | (1 << TWIE);
      p = 0;
      break;

    case TW_MT_SLA_NACK:
    case TW_MR_SLA_NACK:
      /* SLA+W transmitted, NACK received */
      /* SLA+R transmitted, NACK received */
      SEN_TW(CR) = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO);
      i2c_req.xfer = FMU_I2C_ERR;
      PORTE ^= (1 << PORTE3);
      break;

    case TW_MT_DATA_ACK:
      /* data transmitted, ACK received */
      switch(i2c_req.xfer) {
        case FMU_I2C_WRITE:
          if (p < i2c_req.len) {
            SEN_TW(DR) = i2c_req.iobuf[p++];
            SEN_TW(CR) = (1 << TWINT) | (1 << TWEN) | (1 << TWIE);
          } else {
            SEN_TW(CR) = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO);
            i2c_req.xfer = FMU_I2C_DONE;
            PORTE ^= (1 << PORTE3);
          }
          break;

        case FMU_I2C_READ:
          SEN_TW(CR) = (1 << TWINT) | (1 << TWEN) | (1 << TWIE) | (1 << TWSTA);
          break;

        default:
          SEN_TW(CR) = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO);
          i2c_req.xfer = FMU_I2C_ERR;
          PORTE ^= (1 << PORTE3);
          break;
      }
      break;

    case TW_MT_DATA_NACK:
      /* data transmitted, NACK received */
      SEN_TW(CR) = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO);
      i2c_req.xfer = FMU_I2C_ERR;
      PORTE ^= (1 << PORTE3);
      break;

    case TW_MR_DATA_ACK:
      /* data received, ACK returned */
      i2c_req.iobuf[p++] = SEN_TW(DR);
      if (p < i2c_req.len - 1) /* NACK last byte */
        SEN_TW(CR) = (1 << TWINT) | (1 << TWEN) | (1 << TWIE) | (1 << TWEA);
      else
        SEN_TW(CR) = (1 << TWINT) | (1 << TWEN) | (1 << TWIE);
      break;

    case TW_MR_DATA_NACK:
      /* data received, NACK returned */
      i2c_req.iobuf[p++] = SEN_TW(DR);
      SEN_TW(CR) = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO);
      i2c_req.xfer = FMU_I2C_DONE;
      PORTE ^= (1 << PORTE3);
      break;


    case TW_MT_ARB_LOST:
    case TW_BUS_ERROR:
      /* arbitration lost in SLA+W or data */
      /* illegal start or stop condition */
      SEN_TW(CR) = (1 << TWINT); /* reset TWI internal state */
      while(SEN_TW(CR) & (1 << TWINT));

      /* start over */
      SEN_TW(CR) = (1 << TWINT) | (1 << TWEN) | (1 << TWIE) | (1 << TWSTA);
      break;
  }
}

ISR(PCINT3_vect, ISR_NOBLOCK)
{
  if (i2c_req.done)
    i2c_req.done(i2c_req.xfer == FMU_I2C_DONE ? i2c_req.iobuf : NULL);
}


/* --- ADC_vect ------------------------------------------------------------ */

ISR(ADC_vect)
{
  uint16_t adc = ADC;

  fmu_sensors.battery = (fmu_sensors.battery + adc)/2;
  ADCSRA |= (1 << ADSC);
}
