/*
 * Copyright (c) 2017 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Fri Jan 27 2017
 */
#include "acheader.h"

#include <avr/interrupt.h>
#include <avr/pgmspace.h>

#include "common/tk3-mikrokopter.h"
#include "flymu.h"


/* --- local data ---------------------------------------------------------- */

static uint8_t hwmissing;

static uint8_t	fmu_idle_time(tk3_time date);
static void	freeze(void);


/* --- main ---------------------------------------------------------------- */

int
main()
{
  tk3_time date, stamp;
  uint8_t count;
  struct tk3_iorecv *msg;

  /* setup clock & interrupts, leds */
  if (tk3_clock_init(0, NULL)) freeze();
  fmu_led_init();
  fmu_led(FMULED_PKTLOSS, 0x80);
  fmu_led(FMULED_ERR, 0xff);

  /* load settings */
  fmu_load_settings();

  /* initialize hardware */
  if (tk3_uart_init(0, settings.uart)) freeze();
  tk3_twi_init(4);

  count = 0;
  count = fmu_motor_init(settings.motors * 2000000);
  hwmissing = 0xff * (settings.motors - count) / settings.motors;

  if (fmu_sensor_init()) hwmissing = 0xff;

  /* main loop */
  date = stamp = tk3_clock_gettime();

  while(1) {
    date = tk3_clock_gettime();

    /* blink leds */
    fmu_led(FMULED_IDLE, fmu_idle_time(date));
    fmu_led(FMULED_PKTLOSS, tk3_comm_warns);
    fmu_led(FMULED_ERR, tk3_comm_errs > hwmissing ? tk3_comm_errs : hwmissing);
    fmu_led_update(date);

    /* process messages */
    while ((msg = tk3_recv())) fmu_domsg(msg);

    /* data logging */
    fmu_log_imu(date);
    fmu_log_battery(date);

    /* clock timestamps */
    if (date - stamp > 1000000) {
      tk3_log(TK3_TWI, PSTR("t%4"), date);
      stamp += 1000000;
    }

    /* flush pending data */
    tk3_send();
  }

  return 0;
}


/* --- fmu_domsg ----------------------------------------------------------- */

void
fmu_domsg(struct tk3_iorecv *msg)
{
  uint8_t cmd = msg->data[0];
  switch(cmd) {
    case 'i': /* IMU data */
      if (msg->len == 5) {
        tk3_time p =
          ((uint32_t)msg->data[1] << 24) |
          ((uint32_t)msg->data[2] << 16) |
          ((uint32_t)msg->data[3] << 8) |
          msg->data[4];

        fmu_logdef_imu(msg->state.channel, p, tk3_clock_gettime());
      }
      break;

    case 'b': /* battery level */
      if (msg->len == 5) {
        tk3_time p =
          ((uint32_t)msg->data[1] << 24) |
          ((uint32_t)msg->data[2] << 16) |
          ((uint32_t)msg->data[3] << 8) |
          msg->data[4];

        fmu_logdef_battery(msg->state.channel, p, tk3_clock_gettime());
      }
      break;

    case 'm': /* motor data */
      if (msg->len == 5) {
        tk3_time p =
          ((uint32_t)msg->data[1] << 24) |
          ((uint32_t)msg->data[2] << 16) |
          ((uint32_t)msg->data[3] << 8) |
          msg->data[4];

        fmu_logdef_motor(msg->state.channel, !!p);
        tk3_log_buffer(TK3_TWI, msg->data, msg->len);
      }
      break;

    case 'g': /* start motors */
    case 'x': /* stop motors */
    case 'q': /* PWM array */
    case 'w': /* velocity array */
    case '~': /* beep */
      tk3_log_buffer(TK3_TWI, msg->data, msg->len);
      break;

    case 'M': /* motor data forwarder */
      fmu_log_motor(msg->data, msg->len);
      break;

    case 'T': /* clock rate */
      tk3_log_buffer(TK3_UART0, msg->data, msg->len);
      break;

    case '?': /* id */
      if (msg->len == 1) {
        tk3_log(msg->state.channel, PSTR("?flymu" PACKAGE_VERSION));
      }
      break;

    default:
      tk3_comm_warns = 0xff;
      break;
  }
}


/* --- fmu_idle_time ------------------------------------------------------- */

static uint8_t
fmu_idle_time(tk3_time date)
{
  static tk3_time last;
  tk3_time cycle;
  uint8_t idle;

  cycle = date - last;
  last = date;
  if (cycle > 1023) /* more than 1.024ms, full off */
    idle = 0;
  else
    idle = 255 - (uint16_t)cycle / 4; /* from (0 = 100%) to (1.024ms = 0%) */

  return idle;
}


/* --- freeze -------------------------------------------------------------- */

static void
freeze(void)
{
  fmu_led(FMULED_IDLE, 0);
  fmu_led(FMULED_ERR, 0xff);
  while(1) fmu_led_update(tk3_clock_gettime());
}


/* --- extend vector section ----------------------------------------------- */

/* compared to 328p, 328pb has more interrupts - hopefully, this goes after
 * gcrt0 vectors */
void  __attribute__((used, naked, noreturn, section(".vectors")))
__vectors_328pb(void)
{
#define isr_callvect_328pb(x)	__asm__( "jmp " #x )
#define isr_vect_328pb(x)	isr_callvect_328pb(x)

  isr_vect_328pb(__bad_interrupt);	/* USART0_START */
  isr_vect_328pb(PCINT3_vect);		/* PCINT3 */
  isr_vect_328pb(USART1_RX_vect);	/* USART1_RX */
  isr_vect_328pb(__bad_interrupt);	/* USART1_UDRE */
  isr_vect_328pb(__bad_interrupt);	/* USART1_TX */
  isr_vect_328pb(__bad_interrupt);	/* USART1_START */
  isr_vect_328pb(__bad_interrupt);	/* TIMER3_CAPT */
  isr_vect_328pb(__bad_interrupt);	/* TIMER3_COMPA */
  isr_vect_328pb(__bad_interrupt);	/* TIMER3_COMPB */
  isr_vect_328pb(__bad_interrupt);	/* TIMER3_OVF */
  isr_vect_328pb(__bad_interrupt);	/* CFD */
  isr_vect_328pb(__bad_interrupt);	/* PTC_EOC */
  isr_vect_328pb(__bad_interrupt);	/* PTC_WCOMP */
  isr_vect_328pb(__bad_interrupt);	/* SPI1_STC */
  isr_vect_328pb(TWI1_vect);		/* TWI1 */
  isr_vect_328pb(__bad_interrupt);	/* TIMER4_CAPT */
  isr_vect_328pb(__bad_interrupt);	/* TIMER4_COMPA */
  isr_vect_328pb(__bad_interrupt);	/* TIMER4_COMPB */
  isr_vect_328pb(__bad_interrupt);	/* TIMER4_OVF */
}
