/*
 * Copyright (c) 2014-2015 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Mon Dec  1 2014
 */
#include "acheader.h"

#include <string.h>

#include "tk3-mikrokopter.h"

const uint8_t *tk3_settings_start;
const uint8_t *tk3_settings_end;

static const uint8_t *tk3_settings_ptr;

static int	tk3_estrncpy(char *dst, const uint8_t **src, size_t n);
static void	tk3_estrskip(const uint8_t **src);


/* --- tk3_settings_init --------------------------------------------------- */

int
tk3_settings_init()
{
  char key[16];

  /* read magic key */
  tk3_settings_start = 0;
  if (tk3_estrncpy(key, &tk3_settings_start, sizeof(key)) ||
      strcmp(key, TK3SET_MAGIC))
    return -1;

  /* find end address */
  tk3_settings_end = tk3_settings_start;
  while(1) {
    if (tk3_estrncpy(key, &tk3_settings_end, sizeof(key))) return -1;
    if (!*key) break;

    tk3_settings_end += 4; /* skip value */
    tk3_estrskip(&tk3_settings_end); /* skip description */

    if (tk3_settings_end >= (uint8_t *)E2END) /* eeprom limit */ return -1;
  }

  tk3_settings_ptr = tk3_settings_start;
  return 0;
}


/* --- tk3_settings_next --------------------------------------------------- */

int
tk3_settings_next(char *key, size_t len, int32_t *value)
{
  if (tk3_settings_ptr >= tk3_settings_end - 1) return -1;

  if (tk3_estrncpy(key, &tk3_settings_ptr, len))
    return -1;

  *value = (int32_t)eeprom_read_byte(tk3_settings_ptr++) << 24;
  *value|= (int32_t)eeprom_read_byte(tk3_settings_ptr++) << 16;
  *value|= (int32_t)eeprom_read_byte(tk3_settings_ptr++) << 8;
  *value|= eeprom_read_byte(tk3_settings_ptr++);

  tk3_estrskip(&tk3_settings_ptr); /* skip description */
  return 0;
}


/* --- tk3_settings_addr --------------------------------------------------- */

uint8_t *
tk3_settings_addr(const char *key)
{
  const uint8_t *p;
  char k[16];

  for(p = tk3_settings_start; p < tk3_settings_end - 1; ) {
    if (tk3_estrncpy(k, &p, sizeof(k))) return NULL;

    if (!strcmp(k, key)) return (uint8_t *)p;

    p += 4; /* skip value */
    tk3_estrskip(&p); /* skip description */
  }

  return NULL;
}


/* --- tk3_settings_update ------------------------------------------------- */

void
tk3_settings_update(uint8_t *addr, int32_t value)
{
  eeprom_write_byte(addr, (value >> 24) & 0xff);
  eeprom_write_byte(addr+1, (value >> 16) & 0xff);
  eeprom_write_byte(addr+2, (value >> 8) & 0xff);
  eeprom_write_byte(addr+3, value & 0xff);
}


/* --- tk3_estrncpy -------------------------------------------------------- */

static int
tk3_estrncpy(char *dst, const uint8_t **src, size_t n)
{
  uint8_t c;

  if (!n) return -1;
  do {
    if (!n--) return -1;
    c = eeprom_read_byte((*src)++);
    *dst++ = c;
  } while(c);

  return 0;
}


/* --- tk3_estrskip -------------------------------------------------------- */

static void
tk3_estrskip(const uint8_t **src)
{
  while(eeprom_read_byte((*src)++)) /* empty body */;
}
