/*
 * Copyright (c) 2014-2015,2017,2019-2020 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Tue Sep 23 2014
 */
#include "acheader.h"

#include <avr/io.h>
#include <avr/sleep.h>
#include <util/atomic.h>

#include "tk3-mikrokopter.h"

/* --- local data ---------------------------------------------------------- */

/* timer0 configuration */
#if F_CPU == 8000000UL

# define USECS_PER_ISR		256
# define TIMER0_RESOLUTION	255
# define TIMER0_PRESCALER	((0 << CS02) | (1 << CS01) | (0 << CS00))
# define USECS_FRAC(x)		(x)
# define FRAC_USECS(x)		(x)

#elif F_CPU == 20000000UL

# define USECS_PER_ISR		64
# define TIMER0_RESOLUTION	159
# define TIMER0_PRESCALER	((0 << CS02) | (1 << CS01) | (0 << CS00))
# define USECS_FRAC(x)		((uint8_t)((uint16_t)(x) * 205 / 512))
# define FRAC_USECS(x)		((x) * 2 + (x) / 2) /* x * 160/64 */

#elif F_CPU == 16000000UL

# define USECS_PER_ISR		128
# define TIMER0_RESOLUTION	255
# define TIMER0_PRESCALER	((0 << CS02) | (1 << CS01) | (0 << CS00))
# define USECS_FRAC(x)		((x) / 2)
# define FRAC_USECS(x)		((x) * 2)

#else

# error "unknown clock frequency - update me"

#endif

/* clock adjustment */
static uint8_t default_osccal;
static uint8_t *clk_adj_addr;

/* current time since last timer overflow */
static volatile tk3_time systick_usecs;


/* --- tk3_clock_init ------------------------------------------------------ */

/** Configure timer0
 */
int
tk3_clock_init(int8_t clk_adj, uint8_t *adj_addr)
{
  clk_adj_addr = adj_addr;
  default_osccal = OSCCAL;
  OSCCAL += clk_adj;

  /* timer0 in CTC mode, compare match interrupt */
  TCCR0A = (1 << WGM01) | (0 << WGM00);
  TIMSK0 = (1 << OCIE0A) | (0 << OCIE0B) | (0 << TOIE0);
  TCCR0B = TIMER0_PRESCALER;
  OCR0A = TIMER0_RESOLUTION;
  TCNT0 = 0;

  sei();
  return 0;
}


/* --- tk3_clock_gettime --------------------------------------------------- */

/* Return current time. It can happen that the timer overflows while another
 * interrupt is being served. In this case, the OCF0A flags will be high, and
 * USECS_PER_ISR must be added to the current time to get the right value.
 */
tk3_time
tk3_clock_gettime(void)
{
  tk3_time t;
  uint8_t r;

  r = TCNT0;
  ATOMIC_BLOCK(ATOMIC_RESTORESTATE) {
    t = systick_usecs;
    if (TIFR0 & (1 << OCF0A)) {
      t += USECS_PER_ISR;
      r = TCNT0;
    }
  }

  return t + USECS_FRAC(r);
}


/* --- tk3_clock_delay ----------------------------------------------------- */

void
tk3_clock_delay(tk3_time deadline)
{
  tk3_time ovf, t;
  uint8_t r;

  r = deadline % USECS_PER_ISR;
  ovf = deadline - r;
  r = FRAC_USECS(r);

  do {
    ATOMIC_BLOCK(ATOMIC_RESTORESTATE) {
      t = systick_usecs;
    }
  } while((t - ovf) & 0x80000000U);

  if (t == ovf)
    while(TCNT0 < r);
}


/* --- tk3_clock_reftick --------------------------------------------------- */

uint8_t
tk3_clock_reftick(tk3_time time, tk3_time rtime)
{
  static tk3_time ptime, prtime;
  static int32_t ioffset;

  int32_t delay;
  int32_t offset;
  int8_t fix = 0;

  if (ptime && prtime) {
    delay = time - ptime;
    offset = rtime - prtime - delay;
    if (offset < delay/2 && -offset < delay/2) {
      ioffset += offset;

      if (ioffset > 8192) fix = 1;
      else if (ioffset < -8192) fix = -1;

      if (fix) {
        OSCCAL += fix;
        ioffset = 0;

        if (clk_adj_addr) {
          /* update eeprom only once after the first 30s after boot - writing 4
           * EEPROM bytes takes as much as 12ms and we must avoid this when
           * flying. 30s is enough to let the oscillator converge, and if it
           * did not, it will start from a better value at next boot anyway. */
          tk3_settings_update(clk_adj_addr, OSCCAL - default_osccal);
          if (tk3_clock_gettime() > 30000000)
            clk_adj_addr = NULL;
        }
      }
    }
  }
  ptime = time;
  prtime = rtime;

  return fix;
}


/* --- tk3_clock_callback -------------------------------------------------- */

static tk3_time cb_time;
static void (*cb_func)(void);

/* execute callback at the specified time  */
void
tk3_clock_callback(tk3_time time, void (*callback)(void))
{
  ATOMIC_BLOCK(ATOMIC_RESTORESTATE) {
    uint8_t r = time % USECS_PER_ISR;
    tk3_time ovf = time - r;

    cb_time = ovf;
    cb_func = callback;

    OCR0B = FRAC_USECS(r);
    TIFR0 |= (1 << OCF2B);
    TIMSK0 |= (1 << OCIE0B);
  }
}


/* --- TIMER0_COMPA_vect --------------------------------------------------- */

/** triggers every "usecs_per_isr" µs */

/* ISR(TIMER0_OVF_vect) */
ISR(TIMER0_COMPA_vect)
{
  systick_usecs += USECS_PER_ISR;
}


/* --- TIMER0_COMPB_vect --------------------------------------------------- */

/** trigger when a callback must be executed */

ISR(TIMER0_COMPB_vect)
{
  tk3_time t = systick_usecs;

  if (!((t - cb_time) & 0x80000000U)) {
    TIMSK0 &= ~(1 << OCIE0B);
    if (cb_func) cb_func();
  }
}
