/*
 * Copyright (c) 2017 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Mon Jan  2 2017
 */
#include "acheader.h"

#include <avr/boot.h>
#include <avr/eeprom.h>
#include <avr/pgmspace.h>

#include <stdint.h>

#include "common/tk3-mikrokopter.h"

/* AVR109: Self Programming
   http://www.atmel.com/images/doc1644.pdf */


/* --- local data ---------------------------------------------------------- */

static const uint32_t timeout_ms = 1000;

static void	fmu_chkprg(void) __attribute__ ((used, noreturn));
static void	fmu_prg(uint8_t unit);

static void	fmu_uart_init(uint32_t baud);
static int16_t	fmu_uart_getc(uint8_t unit, uint8_t block);
static void	fmu_uart_putpstr(uint8_t u, const char *c, uint8_t count);
static void	fmu_uart_putc(uint8_t unit, char c);


/* --- boot ---------------------------------------------------------------- */

void boot(void) __attribute__ ((used, noreturn, section(".vectors")));

void boot()
{
  /* nostartfiles: clear zero register, clear status register, set stack */
  __asm__ ( "clr __zero_reg__" );
  SREG = 0;
  SP = RAMEND;

  goto *(void *)fmu_chkprg;
}


/* --- init ---------------------------------------------------------------- */

static void
fmu_chkprg()
{
  uint16_t progen[2];
  int16_t c;
  uint8_t u;

  /* all leds on */
  DDRC |= (1 << DDC2) | (1 << DDC1) | (1 << DDC0);
  PORTC &= ~((1 << PORTC2) | (1 << PORTC1) | (1 << PORTC0));

  /* init uarts */
  fmu_uart_init(57600);

  /* configure timeout */
  TCCR1A = 0;
  TCCR1B = (1 << CS12) | (1 << CS10); /* 1024 divisor */
  TCCR1C = 0;
  TCNT1 = 0;
  TIFR1 |= (1 << TOV1);

  /* wait for programming enable */
  progen[0] = progen[1] = 0;
  for(; TCNT1 < timeout_ms * (F_CPU/1024/1000) && !(TIFR1 & (1 << TOV1));) {
    for(u = 0; u < 2; u++) {
      c = fmu_uart_getc(u, 0);
      if (c >= 0) {
        progen[u] = ((progen[u] & 0xff) << 8) | (c & 0xff);

        if (progen[u] == 0x1baa || progen[u] == 0x1b53) {
          fmu_prg(u);
          goto run;
        }
      }
    }
  }

run:
  /* reset uarts/timer to default state */
  UCSR0B = UCSR1B = 0;
  UCSR0A = UCSR1A = 0;
  UBRR0 = UBRR1 = 0;

  TCCR1A = TCCR1B = TCCR1C = 0;
  TCNT1 = 0;

  /* all leds off */
  DDRC &= ~((1 << DDC2) | (1 << DDC1) | (1 << DDC0));

  /* run application */
  goto *0;
}


/* --- fmu_prg ------------------------------------------------------------- */

static void
fmu_prg(uint8_t unit)
{
  uint16_t addr = 0;
  int16_t c;

  /* all leds off */
  DDRC &= ~((1 << DDC2) | (1 << DDC1) | (1 << DDC0));

  fmu_uart_putpstr(unit, PSTR("flymu\0"), 7);

  for(;;) {
    /* get char, toggle leds */
    DDRC |= (1 << DDC0);
    DDRC &= ~((1 << DDC2) | (1 << DDC1));

    c = fmu_uart_getc(unit, 1);

    DDRC &= ~(1 << DDC0);
    DDRC |= (1 << DDC2);

    switch(c) {
      case 'E': /* exit */
        if (unit) UCSR1A |= (1<<TXC1); else UCSR0A |= (1<<TXC0);
        fmu_uart_putc(unit, '\r');

        if (unit)
          while (!(UCSR1A & (1<<TXC1)));
        else
          while (!(UCSR0A & (1<<TXC0)));

        DDRC |= (1 << DDC0);
        DDRC &= ~((1 << DDC2) | (1 << DDC1));
        return;

      case 'P': /* enter programming mode */
      case 'L': /* leave programming mode */
        fmu_uart_putc(unit, '\r');
        break;

      case 'T': /* select device */
        fmu_uart_getc(unit, 1);
        fmu_uart_putc(unit, '\r');
        break;

      case 'A': /* set address */
        addr = fmu_uart_getc(unit, 1) << 8;
        addr |= fmu_uart_getc(unit, 1);
        fmu_uart_putc(unit, '\r');
        break;

      case 'B': /* block load */ {
        uint16_t count;

        count = fmu_uart_getc(unit, 1) << 8;
        count |= fmu_uart_getc(unit, 1) & 0xff;
        if (count > SPM_PAGESIZE) count = SPM_PAGESIZE;

        switch (fmu_uart_getc(unit, 1)) {
          case 'F': {
            uint16_t b = addr << 1;
            uint16_t word;

            for(;count; count -= 2, b += 2) {
              word = fmu_uart_getc(unit, 1) & 0xff;
              if (count) word |= fmu_uart_getc(unit, 1) << 8;
              if (b < (uint16_t)boot)
                boot_page_fill(b, word);
            }
            boot_page_write(addr << 1);
            addr = b >> 1;
            boot_spm_busy_wait();
            boot_rww_enable();
            break;
          }

          case 'E': {
            uint8_t words[count], *w;
            uint16_t i;

            for(i = 0, w = words; i < count; i++)
              *w++ = fmu_uart_getc(unit, 1);

            eeprom_write_block(words, (uint8_t *)addr, count);
            addr += count;
            eeprom_busy_wait();
            break;
          }
        }
        fmu_uart_putc(unit, '\r');
        break;
      }

      case 'e': /* chip erase */
        for (addr = 0; addr < (uint16_t)boot; addr += SPM_PAGESIZE) {
          boot_page_erase(addr);
          boot_spm_busy_wait();
        }
        boot_rww_enable();
        fmu_uart_putc(unit, '\r');
        break;

      case 'g': /* read block */ {
        uint16_t count;

        count = fmu_uart_getc(unit, 1) << 8;
        count |= fmu_uart_getc(unit, 1) & 0xff;

        switch (fmu_uart_getc(unit, 1)) {
          case 'F': {
            uint16_t b = addr << 1;
            for(;count; count--, b++)
              fmu_uart_putc(unit, pgm_read_byte(b));
            addr = b >> 1;
            break;
          }

          case 'E':
            for(;count; count--, addr++)
              fmu_uart_putc(unit, eeprom_read_byte((uint8_t *)addr));
            break;
        }
        break;
      }

      case 'b': /* buffer writes / page size */
        fmu_uart_putc(unit, 'Y');
        fmu_uart_putc(unit, (SPM_PAGESIZE >> 8) & 0xff);
        fmu_uart_putc(unit, SPM_PAGESIZE & 0xff);
        break;

      case 'p': /* programmer type */
        fmu_uart_putc(unit, 's');
        break;

      case 'a': /* autoincrement */
        fmu_uart_putc(unit, 'Y');
        break;

      case 't': /* part */
        fmu_uart_putc(unit, 0x86);
        fmu_uart_putc(unit, 0);
        break;

      case 's': /* signature bytes */
        fmu_uart_putc(unit, boot_signature_byte_get(4));
        fmu_uart_putc(unit, boot_signature_byte_get(2));
        fmu_uart_putc(unit, boot_signature_byte_get(0));
        break;

      case 'r': /* lock bits */
        fmu_uart_putc(unit, boot_lock_fuse_bits_get(GET_LOCK_BITS));
        break;

      case 'F': /* fuse byte */
        fmu_uart_putc(unit, boot_lock_fuse_bits_get(GET_LOW_FUSE_BITS));
        break;

      case 'N': /* high fuse byte */
        fmu_uart_putc(unit, boot_lock_fuse_bits_get(GET_HIGH_FUSE_BITS));
        break;

      case 'Q': /* extended fuse byte */
        fmu_uart_putc(unit, boot_lock_fuse_bits_get(GET_EXTENDED_FUSE_BITS));
        break;

      case 'V': /* software version */
        fmu_uart_putc(unit, PACKAGE_VERSION[0]);
        fmu_uart_putc(unit, PACKAGE_VERSION[2]);
        break;

      case 'v': /* hardware version */
        DDRD &= ~(1 << DDD7);
        PORTD |= (1 << PORTD7);

        DDRB &= ~((1 << DDB1) | (1 << DDB0));
        PORTB |= (1 << PORTB1) | (1 << PORTB0);

        fmu_uart_putc(unit, '1' + ((PIND & 0x80)?1:0));
        fmu_uart_putc(unit, '0' + (PINB & 0x3));

        PORTD &= ~(1 << PORTD7);
        PORTB &= ~(1 << PORTB1) | (1 << PORTB0);
        break;

      default: /* unknown */
        fmu_uart_putc(unit, '?');
        break;
    }
  }
}


/* --- fmu_uart_init ------------------------------------------------------- */

static void
fmu_uart_init(uint32_t baud)
{
  uint16_t b = F_CPU / baud / 8;

  /* round to the nearest integer */
  if (F_CPU % (baud * 8) <= baud * 4)
    b--;

  /* set baudrate */
  UBRR0 = b;
  UBRR1 = b;

  /* double clock speed */
  UCSR0A = (1 << U2X0);
  UCSR1A = (1 << U2X1);

  /* 8/N/1 */
  UCSR0C = (1 << UCSZ01) | (1 << UCSZ00);
  UCSR1C = (1 << UCSZ11) | (1 << UCSZ10);

  /* enable transmission and reception */
  UCSR0B = (1 << RXEN0) | (1 << TXEN0);
  UCSR1B = (1 << RXEN1) | (1 << TXEN1);
}


/* --- fmu_uart_getc ------------------------------------------------------- */

static int16_t
fmu_uart_getc(uint8_t unit, uint8_t block)
{
  uint8_t s, c;

  do {
    if (unit) {
      s = UCSR1A;
      if (s & (1 << RXC1)) {
        c = UDR1;
        if (s & ((1 << FE1)|(1 << DOR1))) DDRC |= (1 << DDC1);
        return c;
      }
    } else {
      s = UCSR0A;
      if (s & (1 << RXC0)) {
        c = UDR0;
        if (s & ((1 << FE0)|(1 << DOR0))) DDRC |= (1 << DDC1);
        return c;
      }
    }
  } while (block);

  return -1;
}


/* --- fmu_uart_putpstr ---------------------------------------------------- */

static void
fmu_uart_putpstr(uint8_t unit, const char *s, uint8_t count)
{
  while(count--)
    fmu_uart_putc(unit, pgm_read_byte(s++));
}


/* --- fmu_uart_putc ------------------------------------------------------- */

static void
fmu_uart_putc(uint8_t unit, char c)
{
  if (unit) {
    while (!(UCSR1A & (1<<UDRE1)));
    UDR1 = c;
  } else {
    while (!(UCSR0A & (1<<UDRE0)));
    UDR0 = c;
  }
}
