/*
 * Copyright (c) 2017, 2019, 2024-2025 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Fri May 12 2017
 */
#include "ac_tk3_flash.h"

#include <err.h>
#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>

#include <libusb.h>
#ifdef __linux__
# include <libudev.h>
#endif

#include "flash.h"

/* older libusb-1.0 don't have LIBUSB_API_VERSION, neither LIBUSB_LOG_LEVEL_* */
#ifndef LIBUSB_API_VERSION
# define LIBUSB_LOG_LEVEL_NONE		0
# define LIBUSB_LOG_LEVEL_WARNING	2
#else
/* deal with deprecated libusb_set_debug() */
# if LIBUSB_API_VERSION < 0x01000106
#  define libusb_set_option(ctx, option, level) libusb_set_debug(ctx, level)
# endif
#endif


static int	ftdi_update_eeprom(libusb_device_handle *d,
                        const char *current, const char *new);
static int	tk3_update_serial(libusb_device_handle *d,
                        const char *current, const char *new);


/* --- usb_serial_to_tty --------------------------------------------------- */

/* Return a tty device matching the "serial" string */

const char *
usb_serial_to_tty(const char *serial)
{
#ifdef __linux__
  struct udev *udev;
  struct udev_enumerate *scan = NULL;
  struct udev_list_entry *ttys, *tty;
  struct udev_device *dev = NULL, *p;
  const char *path = NULL;
  const char *s;

  udev = udev_new();
  if (!udev) return NULL;

  /* iterate over tty devices */
  scan = udev_enumerate_new(udev);
  if (udev_enumerate_add_match_subsystem(scan, "tty")) goto done;
  if (udev_enumerate_scan_devices(scan)) goto done;

  ttys = udev_enumerate_get_list_entry(scan);
  udev_list_entry_foreach(tty, ttys) {
    const char *sysfs, *pserialstr, *pifacestr, *pnifacestr;
    int suf, piface, pniface;

    /* get sysfs entry for the device and create a corresponding udev_device */
    if (dev) udev_device_unref(dev);
    sysfs = udev_list_entry_get_name(tty);
    dev = udev_device_new_from_syspath(udev, sysfs);

    /* iterate over parents and look for serial or bInterfaceNumber for usb */
    pserialstr = pifacestr = pnifacestr = NULL;
    for(p = udev_device_get_parent(dev); p; p = udev_device_get_parent(p)) {
      if (!pserialstr)
        pserialstr = udev_device_get_sysattr_value(p, "serial");
      if (!pifacestr)
        pifacestr = udev_device_get_sysattr_value(p, "bInterfaceNumber");
      if (!pnifacestr)
        pnifacestr = udev_device_get_sysattr_value(p, "bNumInterfaces");
    }
    piface = pifacestr ? atoi(pifacestr) : 0;
    pniface = pnifacestr ? atoi(pnifacestr) : 0;

    /* check match */
    if (!pserialstr) continue; /* no device serial */
    if (strstr(serial, pserialstr) != serial) continue; /* no prefix match */
    s = serial + strlen(pserialstr); /* advance after prefix */
    if (pniface > 1 && *s == '\0' && piface == 0) {
      /* ambiguous match, use first iface */
    } else {
      if (sscanf(s, ".%d", &suf) != 1) suf = -1;
      if (pniface < 2 && *s && suf != piface) continue; /* no suffix match */
      if (pniface > 1 && suf != piface) continue; /* no suffix match */
    }

    /* got a match, return the tty path */
    path = strdup(udev_device_get_devnode(dev)); /* this will leak ... */
    break;
  }
  if (dev) udev_device_unref(dev);

done:
  if (scan) udev_enumerate_unref(scan);
  if (udev) udev_unref(udev);
  return path;

#else
  return NULL; /* if needed, implement this for other OSes */
#endif
}


/* --- usb_tty_to_serial --------------------------------------------------- */

/* Return the serial string corresponding to the tty device */

const char *
usb_tty_to_serial(const char *node)
{
#ifdef __linux__
  struct udev *udev;
  struct udev_enumerate *scan = NULL;
  struct udev_list_entry *ttys, *tty;
  struct udev_device *dev = NULL, *p;
  char *serial = NULL;

  udev = udev_new();
  if (!udev) return NULL;

  /* iterate over tty devices */
  scan = udev_enumerate_new(udev);
  if (udev_enumerate_add_match_subsystem(scan, "tty")) goto done;
  if (udev_enumerate_scan_devices(scan)) goto done;

  ttys = udev_enumerate_get_list_entry(scan);
  udev_list_entry_foreach(tty, ttys) {
    const char *sysfs, *path, *pserial, *pifacestr, *pnifacestr;
    int piface, pniface;

    /* get sysfs entry for the device and create a corresponding udev_device */
    if (dev) udev_device_unref(dev);
    sysfs = udev_list_entry_get_name(tty);
    dev = udev_device_new_from_syspath(udev, sysfs);

    /* check /dev */
    path = udev_device_get_devnode(dev);
    if (!path || strcmp(node, path)) continue;

    /* iterate over parents and look for serial or bInterfaceNumber for usb */
    pserial = pifacestr = pnifacestr = NULL;
    for(p = udev_device_get_parent(dev); p; p = udev_device_get_parent(p)) {
      if (!pserial)
        pserial = udev_device_get_sysattr_value(p, "serial");
      if (!pifacestr)
        pifacestr = udev_device_get_sysattr_value(p, "bInterfaceNumber");
      if (!pnifacestr)
        pnifacestr = udev_device_get_sysattr_value(p, "bNumInterfaces");
    }
    if (!pserial) continue;
    piface = pifacestr ? atoi(pifacestr) : -1;
    pniface = pnifacestr ? atoi(pnifacestr) : 0;

    if (asprintf(
          &serial, "%s%c%d", pserial, pniface < 2 ? '\0' : '.', piface) < 0)
      serial = NULL;
    break;
  }
  if (dev) udev_device_unref(dev);

done:
  if (scan) udev_enumerate_unref(scan);
  if (udev) udev_unref(udev);
  return serial;

#else
  return NULL; /* if needed, implement this for other OSes */
#endif
}


/* --- usb_open_serial ----------------------------------------------------- */

/* Find a libusb device with matching 'serial' */

static libusb_device_handle *
usb_open_serial(libusb_context *ctx, const char *serial,
                struct libusb_device_descriptor *desc)
{
  libusb_device **devs;
  libusb_device_handle *d;
  char match[256];
  ssize_t n, i;
  int s;

  n = libusb_get_device_list(ctx, &devs);
  if (n < 0) return NULL;

  /* iterate over all usb devices */
  d = NULL;
  for(i = 0; i < n; i++) {
    /* try opening dev, but don't print errors - e.g. permission denied. This
     * would be confusing as unrelated devices may not be openable by a regular
     * user */
    libusb_set_option(ctx, LIBUSB_OPTION_LOG_LEVEL, LIBUSB_LOG_LEVEL_NONE);
    s = libusb_open(devs[i], &d);
    libusb_set_option(ctx, LIBUSB_OPTION_LOG_LEVEL, LIBUSB_LOG_LEVEL_WARNING);
    if (s != LIBUSB_SUCCESS) continue;

    /* get usb descriptor, check if it matches the given serial */
    if (!libusb_get_device_descriptor(devs[i], desc)) {
      s = libusb_get_string_descriptor_ascii(
        d, desc->iSerialNumber, (unsigned char *)match, sizeof(match));
      if (!strncmp(serial, match, s)) break;
    }

    libusb_close(d);
    d = NULL;
  }

  libusb_free_device_list(devs, 1/*unref*/);
  return d;
}


/* --- usb_set_serial ------------------------------------------------------ */

/* Program the serial string of an FTDI chip */

int
usb_set_serial(const char *current, const char *new)
{
  struct libusb_device_descriptor desc;
  libusb_context *ctx;
  libusb_device_handle *d;
  int reattach;
  int i, e;

  if (libusb_init(&ctx)) return -1;
  libusb_set_option(ctx, LIBUSB_OPTION_LOG_LEVEL, LIBUSB_LOG_LEVEL_WARNING);
  e = -1; /* by default */

  d = usb_open_serial(ctx, current, &desc);
  if (!d) {
    warnx("no device with serial %s", current);
    goto done;
  }

  /* on Linux, one must typically detach cdc_acm or ftdi_sio */
  reattach = 0;
  if (libusb_kernel_driver_active(d, 0) == 1) {
    if (libusb_detach_kernel_driver(d, 0)) {
      warnx("cannot detach kernel driver");
      goto done;
    }
    reattach = 1;
  }

  /* update eeprom */
  switch(desc.idVendor << 16 | desc.idProduct) {
    case 0x04036001: /* FTDI/FT232 Serial (UART) IC */
      if (!ftdi_update_eeprom(d, current, new)) e = 0;
      break;

    case 0x12090001: /* Generic pid.codes/Test PID */
      if (!tk3_update_serial(d, current, new)) e = 0;
      break;
  }

  /* perform a soft reset so that the new serial can be used */
  if (reattach) libusb_attach_kernel_driver(d, 0);
  libusb_reset_device(d);
  if (e) goto done;

  /* reenumerate new device after reset, with timeout */
  libusb_close(d);
  for(i = 0; i < 4; i++) {
    nanosleep(&(struct timespec){.tv_nsec = 500000000 /*500ms*/ }, NULL);
    d = usb_open_serial(ctx, new, &desc);
    if (d) break;
  }
  if (!d) warnx("cannot find new device %s", new);

done:
  if (e) warnx("serial %s cannot be updated", current);
  if (d) libusb_close(d);
  libusb_exit(ctx);
  return e;
}


/* --- ftdi_update_eeprom -------------------------------------------------- */

/* This updates an FTDI eeprom. If you wonder why this is not done with
 * libftdi, here is the reason: after the first attempt to do it with
 * libftdi-0.20, the eeprom got perfectly well corrupted, and only FT_Prog
 * (a windows utility) could restore it (even ftdi_eeprom was not able to
 * generate a proper eeprom). libftdi1 might be less buggy, but it does not
 * ship on ubuntu-14.04 ...
 */

#define FTDI_IN \
  LIBUSB_REQUEST_TYPE_VENDOR | LIBUSB_RECIPIENT_DEVICE | LIBUSB_ENDPOINT_IN
#define FTDI_OUT \
  LIBUSB_REQUEST_TYPE_VENDOR | LIBUSB_RECIPIENT_DEVICE | LIBUSB_ENDPOINT_OUT

#define FTDI_WRITE_LATENCY	0x09
#define FTDI_READ_EEPROM	0x90
#define FTDI_WRITE_EEPROM	0x91

#define EEPROM_SERIAL_OFFSET	0x12
#define EEPROM_SERIAL_SIZE	0x13

static int
ftdi_update_eeprom(libusb_device_handle *d,
                   const char *current, const char *new)
{
  unsigned char eeprom[0x80];
  unsigned int len, off;
  uint16_t cksum;
  int i;

  /* read eeprom */
  for (i = 0; i < sizeof(eeprom)/2; i++) {
    if (libusb_control_transfer(
          d, FTDI_IN, FTDI_READ_EEPROM,
          0, i, &eeprom[i*2], 2, 500/*ms*/) != 2)
      return -1;
  }

  /* compute checksum, to make sure we have a known structure */
  cksum = 0xaaaa;
  for (i = 0; i < sizeof(eeprom)-2; i += 2) {
    cksum = (eeprom[i] + (eeprom[i+1] << 8)) ^ cksum;
    cksum = (cksum << 1) | (cksum >> 15);
  }
  if (eeprom[i] + (eeprom[i+1]<<8) != cksum) {
    warnx("current eeprom checksum does not match, aborting for safety");
    return -1;
  }

  /* decode old serial - do consistency checks */
  len = eeprom[EEPROM_SERIAL_SIZE]/2 - 1;
  off = (eeprom[EEPROM_SERIAL_OFFSET] & 0x7f) + 2;
  if (off + len >= sizeof(eeprom)) return -1;
  if (eeprom[off-2] != 2*(len+1) || eeprom[off-1] != 3 /* string type */) {
    warnx("current eeprom data does not match, aborting for safety");
    return -1;
  }

  for (i = 0; i < len; i++) {
    if (current[i] != eeprom[off + i*2] ||
        eeprom[off + i*2 + 1] != 0) {
      warnx("current eeprom serial does not match, aborting for safety");
      return -1;
    }
  }

  /* old serial matches, update with new serial */
  len = strlen(new);
  if (len > 15) {
    warnx("serial too long - 15 characters max");
    return -1;
  }
  eeprom[EEPROM_SERIAL_SIZE] = len*2 + 2;
  eeprom[off - 2] = eeprom[EEPROM_SERIAL_SIZE];
  for (i = 0; i < len; i++) {
    eeprom[off + i*2] = new[i];
    eeprom[off + i*2 + 1] = 0;
  }

  /* update checksum */
  cksum = 0xaaaa;
  for (i = 0; i < sizeof(eeprom)-2; i += 2) {
    cksum = (eeprom[i] + (eeprom[i+1] << 8)) ^ cksum;
    cksum = (cksum << 1) | (cksum >> 15);
  }
  eeprom[i]   = cksum;
  eeprom[i+1] = cksum >> 8;

  /* write back new eeprom - latency must be set to 77ms - don't ask why */
  if (libusb_control_transfer(
        d, FTDI_OUT, FTDI_WRITE_LATENCY, 77, 0, NULL, 0, 500/*ms*/) < 0)
    return -1;

  for (i = 0; i < sizeof(eeprom)/2; i++) {
    if (libusb_control_transfer(
          d, FTDI_OUT, FTDI_WRITE_EEPROM,
          eeprom[i*2] + (eeprom[i*2 + 1] << 8), i, NULL, 0, 500/*ms*/) < 0)
      return -1;
  }

  return 0;
}


/* --- tk3_update_serial --------------------------------------------------- */

/* This updates a tawaki serial. chimera boards use the OTP and can't do that.
 */
static int
tk3_update_serial(libusb_device_handle *d,
                  const char *current, const char *new)
{
  unsigned char serial[0x40];
  unsigned int len;
  uint32_t s;
  int e;

  /* limit length to 55+NUL to fit in a setup packet (8 bytes overhead) */
  len = strlen(new);
  if (len > 55) {
    warnx("serial too long - 55 characters max");
    return -1;
  }
  memcpy(serial, new, len + 1);

  e = libusb_control_transfer(
    d,
    LIBUSB_REQUEST_TYPE_VENDOR |
    LIBUSB_RECIPIENT_DEVICE |
    LIBUSB_ENDPOINT_OUT,
    1, 0, 0, serial, len + 1, 500/*ms*/);
  if (e < 0) {
    warnx("sending serial: %s (%d)", libusb_error_name(e), e);
    return -1;
  }

  /* poll for completion */
  do {
    struct timespec ts, rem;

    ts.tv_sec = 0;
    ts.tv_nsec = 100000000; /* 100ms */
    while (nanosleep(&ts, &rem) != 0 && errno == EINTR) ts = rem;

    e = libusb_control_transfer(
      d,
      LIBUSB_REQUEST_TYPE_VENDOR |
      LIBUSB_RECIPIENT_DEVICE |
      LIBUSB_ENDPOINT_IN,
      1, 0, 0, (uint8_t *)&s, sizeof(s), 500/*ms*/);
    if (e < 0) {
      warnx("polling completion: %s (%d)", libusb_error_name(e), e);
      return -1;
    }
  } while (s & 1 /* busy */);

  return s ? -1 : 0;
}
