/*
 * Copyright (c) 2014, 2017, 2019-2020, 2025 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Fri Nov 28 2014
 */

#include "ac_tk3_flash.h"

#include <err.h>
#include <errno.h>
#include <inttypes.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "flash.h"


/* --- print_settings ------------------------------------------------------ */

void
print_settings(const char *serial, const char *data)
{
  const char *p;
  int32_t v;

  printf("\n--- Settings --------------------------------------\n\n");
  /* usb settings */
  printf("%-16s: %s\n", "serial", serial ? serial : "<none>");

  /* custom settings */
  for(p = data; *p;) {
    printf("%-16s: ", p);
    p += 1 + strlen(p);
    v = *(unsigned char *)p++ << 24;
    v|= *(unsigned char *)p++ << 16;
    v|= *(unsigned char *)p++ << 8;
    v|= *(unsigned char *)p++;
    printf("%-16" PRId32 " (", v);
    printf("%s)\n", p);
    p += 1 + strlen(p);
  }

  printf("\n");
}


/* --- set_settings -------------------------------------------------------- */

int
set_char_settings(char *dst, const char *key, const char *value)
{
  int32_t v;
  char *e;

  /* usb serial # is handled separately */
  if (!strcmp("serial", key)) return 0;

  errno = 0;
  v = strtol(value, &e, 10 /* some values may conveniently be specified with
                            * leading zeroes in decimal representation, so
                            * prevent octal interpretation of those by forcing
                              base 10. (#452) */);
  if (errno || *e) {
    if (!errno) errno = EINVAL;
    warn("setting %s = %s", key, value);
    exit(2);
  }

  return set_settings(dst, key, v);
}

int
set_settings(char *dst, const char *key, int32_t value)
{
  char *p;

  /* usb serial # is handled separately */
  if (!strcmp("serial", key)) return 0;

  /* look for settings entry */
  for(p = dst + sizeof(TK3_MAGIC); *p;) {
    if (!strcmp(p, key)) break;
    p += 1 + strlen(p);
    p += 4;
    p += 1 + strlen(p);
  }
  if (!*p) { warnx("unknown setting: %s", key); return -1; }

  p += 1 + strlen(p);
  *p++ = value >> 24;
  *p++ = value >> 16;
  *p++ = value >> 8;
  *p++ = value;
  return 0;
}


/* --- merge_settings ------------------------------------------------------ */

int
merge_settings(char *dst, const char *src, int drop)
{
  const char *p, *k;
  int32_t v;

  for(p = src + sizeof(TK3_MAGIC); *p;) {
    k = p;
    p += 1 + strlen(p);
    v  = (*(int8_t *)p++) << 24;
    v |= (*(uint8_t *)p++) << 16;
    v |= (*(uint8_t *)p++) << 8;
    v |= *(uint8_t *)p++;
    p += 1 + strlen(p);

    if (set_settings(dst, k, v)) {
      if (drop)
        printf("dropping obsolete setting %s = %d\n", k, v);
      else return -1;
    }
  }

  return 0;
}


/* --- update_serial ------------------------------------------------------- */

int
update_serial(char **input, char **argv)
{
  const char *serial;
  int more, i, fd;

  for(more = i = 0; argv[i] && argv[i+1]; i+=2) {
    if (strcmp("serial", argv[i])) { more = 1; continue; }

    /* get current usb serial */
    fd = open_tty(*input, 0, 0, &serial);
    if (fd < 0) { warn("%s", *input); exit(2); }
    close(fd);

    /* update */
    if (!serial) {
      warnx("cannot update serial of a device with no serial");
      return 2;
    } else if (usb_set_serial(serial, argv[i+1]))
      return 2;

    printf("set %s serial to %s\n", *input, argv[i+1]);
    *input = argv[i+1];
  }

  return more ? 0 : 1;
}
