/*
 * Copyright (c) 2014,2016-2017,2019,2021 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Fri May 23 2014
 */
#include "ac_tk3_flash.h"

#include <sys/types.h>
#include <sys/stat.h>

#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <poll.h>
#include <termios.h>
#include <unistd.h>

#include "flash.h"


/* --- open_tty ------------------------------------------------------------ */

/* Open a serial port, configure to the given baud rate */
int
open_tty(const char *input, unsigned int baud, unsigned int parity,
         const char **serial)
{
  const char *path;
  struct termios t;
  speed_t s;
  int f, fd;

  /* try to match a serial id first */
  path = usb_serial_to_tty(input);
  if (path) {
    *serial = input;
    input = path;
  } else {
    *serial = usb_tty_to_serial(input);
  }

  /* open non-blocking before configuration */
  fd = open(input, O_RDWR | O_NOCTTY | O_NONBLOCK);
  if (fd < 0) return fd;
  if (!isatty(fd)) {
    errno = ENOTTY;
    return -1;
  }

  /* configure line discipline */
  if (tcgetattr(fd, &t)) return -1;

  t.c_iflag = IGNBRK;
  t.c_oflag = 0;
  t.c_lflag = 0;
  t.c_cflag &= ~(CSIZE | CSTOPB | PARENB | PARODD);
  t.c_cflag |= CS8 | CREAD | CLOCAL | (parity ? PARENB : 0);
  t.c_cc[VMIN] = 0;
  t.c_cc[VTIME] = 0;

  switch(baud) {
    case 57600:	s = B57600; break;
    default: s = B57600; break;
  }

  if (cfsetospeed(&t, s)) return -1;
  if (cfsetispeed(&t, s)) return -1;

  if (tcsetattr(fd, TCSADRAIN, &t)) return -1;

  /* set back to blocking reads */
  f = fcntl(fd, F_GETFL, 0);
  if (f != -1)
    fcntl(fd, F_SETFL, f & ~O_NONBLOCK);

  /* discard any pending data */
  tcflush(fd, TCIFLUSH);

  return fd;
}


/* --- write_serial -------------------------------------------------------- */

void
write_serial(int fd, const void *buf, size_t count)
{
  extern int delay_p;
  ssize_t l;

  /* writes byte by byte, and delay between each write - some controllers need
   * this, but why? */
  for(; count; count--) {
    do {
      l = write(fd, buf++, 1);
    } while(l < 0 && errno == EAGAIN);
    if (l < 0) err(2, "write to serial device");
    if (l < 1) errx(2, "short write to serial device");

    usleep(1000 * delay_p);
  }
}


/* --- write_serial -------------------------------------------------------- */

ssize_t
read_serial(int fd, void *buf, size_t count, int timeout)
{
  struct pollfd fds;
  ssize_t t, l;
  int n;

  t = 0;
  do {
    fds.fd = fd;
    fds.events = POLLIN;
    fds.revents = 0;
    n = poll(&fds, 1, timeout);

    if (n < 0) err(2, "read from serial device");
    if (n == 0 || (fds.revents & (POLLERR|POLLHUP))) return t;

    do {
      l = read(fd, buf, count-t);
    } while(l < 0 && errno == EAGAIN);
    if (l < 0) err(2, "read from serial device");

    buf += l;
    t += l;
  } while(t < count);

  return t;
}
