/*
 * Copyright (c) 2019-2021 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Tue Dec  3 2019
 */
#include "ac_tk3_flash.h"

#include <sys/ioctl.h>
#include <err.h>
#include <poll.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <termios.h>
#include <unistd.h>

#include "flash.h"

/* --- probe_bootloader ---------------------------------------------------- */

/* connect to the bootloader */

int
probe_bootloader(const char *port, int baud,
                 const char **serial, enum bltype *bltype)
{
  char data[8];
  int timeout;
  ssize_t l;
  int fd, init;

  /* check access */
  fd = open_tty(port, baud, 0, serial);
  if (fd < 0) { warn("%s", port); exit(2); }
  close(fd);

  printf("Please power on or reset the device...");
  fflush(stdout);
  timeout = 60 /* trials, at least 3s */;

  do {
    fd = open_tty(port, baud, 0, serial);
    if (fd < 0) { poll(NULL, 0, 50); continue; }

    /* toggle DTR, ignore errors */
    ioctl(fd, TIOCMBIC, &(int){TIOCM_DTR});
    poll(NULL, 0, 150);
    ioctl(fd, TIOCMBIS, &(int){TIOCM_DTR});
    poll(NULL, 0, 100);

    /* enable programming - STM32 must come first, as 0x7f is used
     * to configure its UART baud rate. Also note that 8N1 is used here
     * (instead of 8E1) but this happens to work (parity bit is 1 in 0x7f, 0x79
     * and 0x1f, i.e like stop bit and only one byte is sent or read) */
    static const char *magic[] = {
      "\x7f",		/* stm32 boot loader init */
      "\x1b\xaa",	/* avr mk boot loader init */
    };
    for (init = 0; init < sizeof(magic)/sizeof(*magic); init++) {

      /* drain any dandling input */
      while (read_serial(fd, data, sizeof(data), 100) > 0);

      /* send magic sequence */
      if (write(fd, magic[init], strlen(magic[init])) < 0) {
        poll(NULL, 0, 100); break;
      }

      /* detect replies */
      do {
        l = read_serial(fd, data, sizeof(data)-2, 100);
        if (l <= 0) break;
        data[l] = '\0';

        /* guess bltype */
        if (!strncmp(data, "MKBL", 4)) {
          *bltype = MKBL_MKBL; goto match;
        } else if (!strncmp(data, "flymu", 5)) {
          *bltype = MKBL_FLYMU; goto match;
        } else if (l == 1 && (data[0] == 0x79 || data[0] == 0x1f)) {
          *bltype = STM32BL; goto match;
        }
      } while (l > 0);
    }

    /* no reply: try to send magic reset sequences */
    static const char *reset[] = {
      "E"		/* avr mk boot loader reset */
      "reset",		/* tk3 magic reset message */
    };
    for (init = 0; init < sizeof(reset)/sizeof(*reset); init++) {
      if (write(fd, reset[init], strlen(reset[init])) < 0) break;
    }
    tcdrain(fd);

    close(fd);
    printf("."); fflush(stdout);
  } while(--timeout);
  printf("\n");

  warnx("timeout while probing the device");
  return -1;

match:
  printf("\n");

  /* drain any remaining input */
  while (read_serial(fd, data, sizeof(data), 100) > 0);

  switch (*bltype) {
    case MKBL_MKBL: printf("Probed Mikrokopter bootloader\n"); break;
    case MKBL_FLYMU: printf("Probed flymu bootloader\n"); break;
    case STM32BL:
      /* must switch to 8E1 */
      close(fd);
      fd = open_tty(port, baud, 2, serial);
      if (fd < 0) {
        fd = open_tty(port, baud, 0, serial);
        if (fd < 0) {
          warn("%s", port); exit(2);
        }
      }
      printf("Probed STM32 bootloader\n");
      break;
  }

  return fd;
}
