/*
 * Copyright (c) 2014-2017,2019-2020,2024 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Fri May 23 2014
 */
#include "ac_tk3_flash.h"

#include <sys/ioctl.h>
#include <err.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <termios.h>
#include <unistd.h>

#include "flash.h"

enum memk { FL_FLASH, FL_EEPROM };

static int			mkbl_connect(int fd);
static const struct avr_dev *	mkbl_device(int fd);
static int			mkbl_write(int fd, const struct avr_dev *dev,
                                           enum bltype bltype, char *data,
                                           size_t size, enum memk kind);
static void *			mkbl_eeprom(int fd, const struct avr_dev *dev);

static const struct avr_dev {
  const char *name;
  unsigned char id;
  size_t flash_size, eeprom_size;
} avr_devs[] = {
  { .name = "ATMEGA1284P", .id = 0x7a,
    .flash_size = 0x20000, .eeprom_size = 0x1000 },
  { .name = "ATMEGA368PB", .id = 0x86,
    .flash_size = 0x8000, .eeprom_size = 0x400 },
  { .name = "ATMEGA168", .id = 0x78,
    .flash_size = 0x4000, .eeprom_size = 0x200 },
  { NULL, 0, 0 }
};


/* --- mkbl_flash ---------------------------------------------------------- */

int
mkbl_flash(const char *exe, int fd, const char *serial, enum bltype bltype,
           enum settype init_p)
{
  const struct avr_dev *dev;
  uintptr_t paddr;
  size_t size;
  char *data;
  Elf *elf;

  elf = elf_init(exe, EM_AVR);
  if (!elf) exit(2);

  if (mkbl_connect(fd)) return -1;
  dev = mkbl_device(fd);
  if (!dev) return -1;

  /* get flash data */
  paddr = 0;
  size = dev->flash_size;
  data = elf_loadable_data(elf, &paddr, &size, LD_DATA);
  if (!data) return -1;
  if (paddr > 0) {
    warnx("flash data not starting at address 0"); exit(2);
  }
  if (mkbl_write(fd, dev, bltype, data, size, FL_FLASH)) return -1;
  free(data);

  /* get settings data */
  paddr = 0x810000;
  size = dev->eeprom_size;
  data = elf_loadable_data(elf, &paddr, &size, LD_SETTINGS);
  if (!data) return -1;
  if (paddr != 0x810000) {
    warnx("EEPROM data not starting at address 0x810000"); exit(2);
  }
  if (!size) {
    data = strdup(TK3_MAGIC);
    size = sizeof(TK3_MAGIC);
  }

  switch (init_p) {
    case SET_KEEP: {
      char *oset;

      /* read old settings */
      oset = mkbl_eeprom(fd, dev);
      if (!oset) return -1;
      if (strcmp(oset, TK3_MAGIC)) {
        warnx("invalid eeprom data, please flash with option -i");
        exit(2);
      }

      /* check old settings */
      if (merge_settings(data, oset, 0/*drop*/) ||
          merge_settings(oset, data, 0/*drop*/)) {
        warnx("outdated settings data, please flash with option -i or -u");
        exit(2);
      }
      break;
    }

    case SET_MERGE: {
      char *oset;

      /* read old settings */
      oset = mkbl_eeprom(fd, dev);
      if (!oset) return -1;
      if (strcmp(oset, TK3_MAGIC)) {
        warnx("invalid eeprom data, please flash with option -i");
        exit(2);
      }

      /* merge old and new settings */
      merge_settings(data, oset, 1/*drop*/);
      free(oset);
    }
      /*FALLTHROUGH*/
    case SET_RESET:
      if (mkbl_write(fd, dev, bltype, data, size, FL_EEPROM)) return -1;
      break;
  }
  free(data);

  /* display eeprom data */
  data = mkbl_eeprom(fd, dev);
  if (!data) return -1;

  if (strcmp(data, TK3_MAGIC)) {
    warnx("invalid eeprom data, please flash with option -i");
    exit(2);
  }
  print_settings(serial, data + sizeof(TK3_MAGIC));
  free(data);

  return 0;
}


/* --- mkbl_params --------------------------------------------------------- */

int
mkbl_params(int fd, const char *serial, enum bltype bltype, char **argv)
{
  const struct avr_dev *dev;
  char *data, *p;
  int i;

  /* connect to the uc */
  if (mkbl_connect(fd)) return -1;
  dev = mkbl_device(fd);
  if (!dev) return -1;

  /* read old settings */
  data = mkbl_eeprom(fd, dev);
  if (!data) return -1;
  if (strcmp(data, TK3_MAGIC)) {
    warnx("invalid eeprom data, please flash with option -i");
    return -1;
  }

  /* merge old and new settings */
  for(i = 0; argv[i] && argv[i+1]; i+=2) {
    if (set_char_settings(data, argv[i], argv[i+1])) exit(2);
  }

  /* get actual size */
  for(p = data + sizeof(TK3_MAGIC); *p;) {
    p += 1 + strlen(p);
    p += 4;
    p += 1 + strlen(p);
  }
  p++;

  /* re-flash */
  if (i) {
    if (mkbl_write(fd, dev, bltype, data, p - data, FL_EEPROM))
      return -1;
  }

  print_settings(serial, data + sizeof(TK3_MAGIC));
  return 0;
}


/* --- mkbl_connect -------------------------------------------------------- */

/* connect to the bootloader */

static int
mkbl_connect(int fd)
{
  char data[8];
  ssize_t l;

  /* get version */
  write_serial(fd, "V", 1);
  l = read_serial(fd, data, 3, 100);
  if (l < 2) {
    warnx("cannot read bootloader version");
    return -1;
  }
  data[l] = 0;

  printf("Bootloader version %c.%c%s%c\n",
         data[0], data[1], data[2]?".":"", data[2]);

  write_serial(fd, "v", 1);
  l = read_serial(fd, data, 1, 100);
  if (l && data[0] != '?') {
    data[1] = 0;
    l = read_serial(fd, data + 1, 1, 100);

    printf("Board version %c%s%c\n", data[0], data[1]?".":"", data[1]);
  }

  return 0;
}


/* --- mkbl_device --------------------------------------------------------- */

/* check microcontroller device */

static const struct avr_dev *
mkbl_device(int fd)
{
  const struct avr_dev *dev;
  unsigned char c[3], r[2];
  ssize_t l;

  write_serial(fd, "t", 1);
  l = read_serial(fd, &c[1], 2, 1000);
  if (l != 2) { warnx("timeout while checking the device"); return NULL; }

  for(dev = avr_devs; dev->id ; dev++)
    if (dev->id == c[1]) break;
  if (dev->id != c[1]) {
    warnx("unknown device %x", c[1]); return NULL;
  }

  r[0] = c[2] ? c[2] + '@' : '\0';
  r[1] = '\0';
  printf("Probed an %s%s microcontroller\n", dev->name, r);

  c[0] = 'T';
  write_serial(fd, c, 2);
  l = read_serial(fd, c, 1, 1000);
  if (l != 1) { warnx("timeout while checking the device"); return NULL; }
  if (c[0] != 0xd) { warnx("cannot set microcontroller device"); return NULL; }

  return dev;
}


/* --- mkbl_write ---------------------------------------------------------- */

/* flash program */

static int
mkbl_write(int fd, const struct avr_dev *dev, enum bltype bltype,
           char *data, size_t size, enum memk kind)
{
  size_t block, percent;
  unsigned char c[4];
  ssize_t p, l, s;
  uint16_t crc;

  if (size > ((kind == FL_FLASH) ? dev->flash_size : dev->eeprom_size)) {
    warnx("program does not fit into %s memory",
          (kind == FL_FLASH) ? "flash" : "eeprom");
    return -1;
  }

  /* get write size */
  write_serial(fd, "b", 1);
  l = read_serial(fd, c, 3, 1000);
  if (l != 3 || c[0] != 'Y') {
    warnx("cannot get write buffer size");
    return -1;
  }

  block = (c[1]<<8) + c[2];
  printf("Using %zu bytes writes for a total of %zu bytes\n", block, size);

  if (kind == FL_FLASH) {
    /* clear flash */
    printf("Clearing flash...");
    fflush(stdout);
    write_serial(fd, "e", 1);
    l = read_serial(fd, c, 1, 30000 /* 30s timeout */);
    if (l != 1) { warnx("timeout while clearing flash"); return -1; }
    if (c[0] != 0xd) { warnx("cannot clear flash"); return -1; }
    printf(" ok\n");
  }

  /* write data */
  printf("Writing data...");
  fflush(stdout);
  percent = 0;

  c[0] = 'A';
  c[1] = 0;
  c[2] = 0;
  write_serial(fd, c, 3);
  l = read_serial(fd, c, 1, 1000);
  if (l != 1) { warnx("timeout while setting write address"); return -1; }
  if (c[0] != 0xd) { warnx("cannot set write address"); return -1; }

  for(p = 0; p < size; p += block) {
    int retries = 0;

again:
    s = (p + block > size) ? size-p : block;
    printf("\rWriting data... %d%%", (int)((double)percent/size*100));
    fflush(stdout);

    c[0] = 'B';
    c[1] = (s >> 8) & 0xff;
    c[2] = s & 0xff;
    c[3] = (kind == FL_FLASH) ? 'F' : 'E';
    write_serial(fd, c, 4);
    write_serial(fd, data + p, s);

    /* writing to mk flash mem requires checksum */
    if (bltype == MKBL_MKBL && kind == FL_FLASH) {
      crc = mkbl_crc16(data + p, s);
      c[0] = crc >> 8;
      c[1] = crc & 0xff;
      write_serial(fd, c, 2);
    }

    l = read_serial(fd, c, 1, 1000);
    if (l != 1 || c[0] != 0xd) {
      if (++retries < 3) {
        char dummy[4];
        while (read_serial(fd, dummy, 4, 10) > 0);
        printf("\rError writing data at %d%% (retrying)\n",
               (int)((double)percent/size*100));

        c[0] = 'A';
        c[1] = (p >> 9) & 0xff;
        c[2] = (p >> 1) & 0xff;
        write_serial(fd, c, 3);
        l = read_serial(fd, c, 1, 1000);
        if (l != 1) { warnx("timeout while setting write address"); return -1; }
        if (c[0] != 0xd) { warnx("cannot set write address"); return -1; }

        goto again;
      }

      printf("\n");
      if (l != 1) { warnx("timeout writing to flash"); return -1; }
      if (c[0] != 0xd) { warnx("cannot write to flash"); return -1; }
    }

    percent += s;
  }
  printf("\n");

  return 0;
}


/* --- mkbl_eeprom --------------------------------------------------------- */

/* read data from eeprom */

static void *
mkbl_eeprom(int fd, const struct avr_dev *dev)
{
  unsigned char c[4];
  char *data;
  ssize_t p, l, n, block, percent;

  data = malloc(dev->eeprom_size);
  if (!data) { warnx("out of memory"); return NULL; }

  /* get read size */
  write_serial(fd, "b", 1);
  l = read_serial(fd, c, 3, 1000);
  if (l != 3 || c[0] != 'Y') {
    warnx("cannot get read buffer size");
    return NULL;
  }

  block = (c[1]<<8) + c[2];
  printf("Using %zu bytes reads for a total of %zu bytes\n",
         block, dev->eeprom_size);

  /* read eeprom */
  printf("Reading eeprom data...");
  fflush(stdout);
  percent = 0;

  c[0] = 'A';
  c[1] = 0;
  c[2] = 0;
  write_serial(fd, c, 3);
  l = read_serial(fd, c, 1, 1000);
  if (l != 1) { warnx("timeout while setting read address"); return NULL; }
  if (c[0] != 0xd) { warnx("cannot set read address"); return NULL; }

  for(p = 0; p < dev->eeprom_size; p += block) {
    int retries = 0;

again:
    l = (p + block > dev->eeprom_size) ? dev->eeprom_size-p : block;
    percent += l;
    printf("\rReading eeprom data... %d%%",
           (int)((double)percent/dev->eeprom_size*100));
    fflush(stdout);

    c[0] = 'g';
    c[1] = (l >> 8) & 0xff;
    c[2] = l & 0xff;
    c[3] = 'E';
    write_serial(fd, c, 4);

    n = read_serial(fd, data + p, l, 1000/* 1s timeout */);
    if (n != l) {
      if (++retries < 3) {
        char dummy[4];
        while (read_serial(fd, dummy, 4, 10) > 0);
        printf("\rError reading data at %d%% (retrying)\n",
               (int)((double)percent/dev->eeprom_size*100));

        c[0] = 'A';
        c[1] = (p >> 8) & 0xff;
        c[2] = p & 0xff;
        write_serial(fd, c, 3);
        l = read_serial(fd, c, 1, 1000);
        if (l != 1) {
          warnx("timeout while setting read address"); return NULL;
        }
        if (c[0] != 0xd) { warnx("cannot set read address"); return NULL; }

        goto again;
      }

      warnx("\ntimeout reading eeprom"); return NULL;
    }
  }
  printf("\n");

  return data;
}


/* --- mkbl_reset ---------------------------------------------------------- */

int
mkbl_reset(int fd)
{
  write_serial(fd, "E", 1);
  return 0;
}
