/*
 * Copyright (c) 2014, 2016-2017, 2019-2020, 2025 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Fri May 23 2014
 */
#include "ac_tk3_flash.h"

#include <err.h>
#include <getopt.h>
#include <libgen.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "flash.h"

static void	usage(FILE *channel, char *argv0);

static const char shortopts_string[] = "+iud:kvh";
static struct option longopts_list[] = {
  { "init",		no_argument,		NULL,	'i'},
  { "update",		no_argument,		NULL,	'u'},
  { "delay",		required_argument,	NULL,	'd'},
  { "no-reset",		no_argument,		NULL,	'k'},
  { "version",		no_argument,		NULL,	'v'},
  { "help",		no_argument,		NULL,	'h'},
  { NULL, 0, NULL, 0}
};


/* --- main ---------------------------------------------------------------- */

int delay_p = 0;

int
main(int argc, char **argv)
{
  enum settype init_p = SET_KEEP;
  int noreset_p = 0;
  int retries;

  char *argv0 = argv[0];
  const char *serial;
  enum bltype bltype;
  int fd;
  int c, s;

  /* parse command line options */
  while (
    (c = getopt_long(argc, argv, shortopts_string, longopts_list, NULL)) != -1)
    switch (c) {
      case 0: break;

      case 'i':
        if (init_p != SET_KEEP) { usage(stderr, argv0); exit(1); }
        init_p = SET_RESET;
        break;
      case 'u':
        if (init_p != SET_KEEP) { usage(stderr, argv0); exit(1); }
        init_p = SET_MERGE;
        break;
      case 'd': delay_p = atoi(optarg); break;
      case 'k': noreset_p++; break;

      case 'v': puts(PACKAGE_VERSION); exit(0); break;
      case 'h':  usage(stdout, argv0);  exit(0); break;

      case '?':
      default:
	usage(stderr, argv0);
        exit(1);
        break;
    }
  argc -= optind;
  argv += optind;

  if (argc && !strcmp(argv[0], "flash")) {
    if (argc != 3) { usage(stderr, argv0); exit(1); }

    fd = probe_bootloader(argv[1], 0, &serial, &bltype);
    if (fd < 0) exit(2);

    for(retries = 0;; retries++) {
      s = -1; /* Wmaybe-uninitialized, gcc-8 */
      switch (bltype) {
        case MKBL_MKBL:
        case MKBL_FLYMU:
          s = mkbl_flash(argv[2], fd, serial, bltype, init_p);
          if (!s && !noreset_p) mkbl_reset(fd);
          break;

        case STM32BL:
          s = stm32_flash(argv[2], fd, serial, init_p);
          if (!s && !noreset_p) stm32_reset(fd);
          break;
      }
      if (!s) break;
      if (retries > 2) exit(2);

      delay_p++;
      warnx("retrying with %dms delay between writes", delay_p);
    }

    close(fd);
    return 0;
  }

  if (argc && !strcmp(argv[0], "set")) {
    if (argc < 2 || argc % 2) { usage(stderr, argv0); exit(1); }

    /* update usb serial if needed */
    if (argc > 2) {
      switch (update_serial(&argv[1], &argv[2])) {
        case 0: break;	/* ok, with more settings to set */
        case 1: exit(0);	/* ok, no more settings */
        default: exit(2);	/* error */
      }
    }

    /* update other settings */
    fd = probe_bootloader(argv[1], 0, &serial, &bltype);
    if (fd < 0) exit(2);

    for(retries = 0;; retries++) {
      s = -1; /* Wmaybe-uninitialized, gcc-8 */
      switch (bltype) {
        case MKBL_MKBL:
        case MKBL_FLYMU:
          s = mkbl_params(fd, serial, bltype, &argv[2]);
          if (!s && !noreset_p) mkbl_reset(fd);
          break;

        case STM32BL:
          s = stm32_params(fd, serial, &argv[2]);
          if (!s && !noreset_p) stm32_reset(fd);
          break;
      }
      if (!s) break;
      if (retries > 2) exit(2);

      delay_p++;
      warnx("retrying with %dms delay between writes", delay_p);
    }

    close(fd);
    return 0;
  }

  usage(stderr, argv0);
  return 1;
}


/* --- usage --------------------------------------------------------------- */

static void
usage(FILE *channel, char *argv0)
{
  fprintf(
    channel,
    "Usage:\n"
    "	%1$s [-i|u] [-dk] flash <tty device | serial> <elf-file>\n"
    "		-i|--init	Reinitialize settings with defaults\n"
    "		-u|--update	Merge new and existing settings\n"
    "		-d|--delay ms	Delay between writes in ms\n"
    "		-k|--no-reset	Do not reset microcontroller after update\n"
    "\n"
    "	%1$s [-k] set <tty device | serial> [key value] [key value]...\n"
    "		-k|--no-reset	Do not reset microcontroller after update\n"
    "\n"
    "	%1$s -h|--help	Display this usage\n"
    "	%1$s -v|--version	Display version\n",
    basename(argv0));
}
